/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023                                                     **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <fstream>
#include <QtPlugin>
#include <QDebug>
#include <QPushButton>
#include <QVBoxLayout>
#include <QFileDialog>
#include "ContextFreeClientPluginExample.h"
#include "ContextFreeServices.h"
#include "Globals.h"
#include "RemoteFileDialog.h"

using namespace contextfreeclientpluginexample;
using namespace cubepluginapi;

void
ContextFreeClientPluginExample::version( int& major, int& minor, int& bugfix ) const
{
    major  = 1;
    minor  = 0;
    bugfix = 0;
}

QString
ContextFreeClientPluginExample::name() const
{
    return "Context Free Client Plugin";
}

void
ContextFreeClientPluginExample::opened( ContextFreeServices* service )
{
    this->service = service;
    qDebug() << "context free client plugin opened";

    QWidget*     widget = service->getWidget();
    QHBoxLayout* layout = new QHBoxLayout();
    widget->setLayout( layout );

    QPushButton* but = new QPushButton( "Use cube_server to receive a file" );
    but->setFixedSize( but->minimumSizeHint() );
    layout->addWidget( but );

    connect( but, &QPushButton::clicked, this, &ContextFreeClientPluginExample::startAction );
}

void
ContextFreeClientPluginExample::startAction()
{
    QUrl url = QUrl( cubegui::RemoteFileDialog::getFileUrl( "" ) );
    if ( !url.isValid() )
    {
        return;
    }

    QString file = url.path();
    QString host = "cube://" + url.host() + ":" + QString::number( url.port() );

    qDebug() << " load file from server:" <<  url;
    std::string                fn = file.toStdString();
    std::vector<unsigned char> fileName( fn.begin(), fn.end() );

    service->connect( host );
    std::vector<unsigned char> answer = service->sendToPlugin( "FileServerPlugin", fileName );

    std::string sourcef { answer.begin(), answer.end() };
    qDebug() << " received " << sourcef.length() << "bytes from server";

    // save to local file system
    if ( sourcef.length() > 0 )
    {
        QString format      = "cubex";
        QString initialPath = QDir::currentPath() + tr( "/cube." ) + format;
        QString outFile     = QFileDialog::getSaveFileName( service->getWidget(),
                                                            tr( "Save As" ), "",
                                                            tr( "%1 Files (*.%2);;All Files (*)" )
                                                            .arg( format.toUpper() )
                                                            .arg( format ) );

        if ( !outFile.isEmpty() )
        {
            std::ofstream ofs( outFile.toStdString(), std::ofstream::out );
            for ( unsigned char c : answer )
            {
                ofs << c;
            }
            ofs.close();
            qDebug() << " save file to " << outFile;
        }
    }
    else
    {
        qDebug() << " answer from server is empty";
    }

    service->disconnect();
}

void
ContextFreeClientPluginExample::closed()
{
    qDebug() << "context free client plugin closed";
    // free all resources allocated in ContextFreeClientPluginExample::opened()
    // children of service->getWidget() will be deleted automatically
}

QString
ContextFreeClientPluginExample::getHelpText() const
{
    return "context free client plugin help text";
}
