"""
cube-webservice.py - A Tornado-based WebSocket proxy for cube_server
Sets up a Tornado web server that serves static files (cube client). It starts cube_server (part of cubelib)
and proxies WebSocket connections to it.
It can be run with command line arguments to specify the port or unix socket path.
- client: http://localhost:<port>/cube/static/cube.html
- proxy:  http://localhost:<port>/cube/ws

"""
import argparse
import errno
import os
import shutil
import socket
import subprocess
import sys
import tornado.ioloop
import tornado.iostream
import tornado.web
import tornado.websocket


# logging
import logging
logging.basicConfig(
    level=logging.INFO,
    format="[%(levelname).1s %(asctime)s %(name)s] %(message)s",
    datefmt="%Y-%m-%d %H:%M:%S"
)
log = logging.getLogger("cube_webservice")


class StaticHandler(tornado.web.StaticFileHandler):
    """
    Serves cube client (cube.html, cube.wasm, cube.js, qtloader.js)
    """

    def set_default_headers(self):
        """
        Custom StaticFileHandler to set additional headers for Qt WebAssembly compatibility.
        fixes "Application exit (SharedArrayBuffer is not defined)" for Qt wasm
        """
        self.set_header("Cross-Origin-Embedder-Policy", "require-corp")
        self.set_header("Cross-Origin-Opener-Policy", "same-origin")

    async def get(self, path="", include_body=True):
        """
        Override get method to serve cube.html with system name in title if SYSTEMNAME is set
        """
        #log.info(f"StaticHandler GET: {path}")
        systemname = os.environ.get("SYSTEMNAME", "")
        if systemname and path == "cube.html":
            file_path = self.get_absolute_path(self.root, path)
            with open(file_path, "r") as f:
                content = f.read()
            # Replace the title in the HTML with the system name
            content = content.replace("<title>cube</title>", f"<title>cube ({systemname})</title>")
            self.set_header("Content-Type", "text/html")
            self.absolute_path = file_path  # required for StaticFileHandler
            self.finish(content)
        else:
            await super().get(path, include_body)


class WSProxyHandler(tornado.websocket.WebSocketHandler):
    """
    WebSocket handler that proxies messages between the client (web sockets) and cube_server (stream sockets).
    It starts cube_server on a unix socket or a free port and proxies WebSocket messages to it.
    """
    buffer_size = 8192

    def initialize(self, cube_server_path, unix_socket=None):
        """
        Initialize the WebSocket handler with the path to the cube_server executable.
        """
        self.cube_server_path = cube_server_path  # path to the cube_server executable
        self.unix_socket = unix_socket            # path of unix socket, otherwise a free TCP port will be used
        self.cube_proc = None
        log.info(f"WSProxyHandler initialized with cube_server_path: {self.cube_server_path}")
        # start periodic ping to keep the connection alive
        self.ping_callback = tornado.ioloop.PeriodicCallback(self.send_ping, 30000)  # in ms
        self.ping_callback.start()

    async def open(self):
       if self.unix_socket:
            await self.open_unix_socket()
       else:
            await self.open_tcp_socket()

    async def open_unix_socket(self):
        # start cube_server binary using a unix socket
        log.info(f"Starting cube_server using unix socket: {self.unix_socket}")
        try:
            self.cube_proc = subprocess.Popen(
                [self.cube_server_path, "-s", self.unix_socket, "-v", "full", "-c"],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
            )
        except Exception as e:
            log.error(f"Failed to start cube_server: {e}")
            sys.exit(1)

        # wait a moment for the cube server to start
        await tornado.gen.sleep(0.2)
        self.check_process()  # check if cube_server started successfully

        # stream to cube_server UNIX socket
        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        self.stream = tornado.iostream.IOStream(sock)
        await self.stream.connect(self.unix_socket)
        tornado.ioloop.IOLoop.current().add_callback(self.read_from_cube)
       
    async def open_tcp_socket(self):
        # start cube_server binary using a free port
        self.cube_port = self.find_free_port()
        log.info(f"Starting cube_server on port: {self.cube_port}")
        try:
            self.cube_proc = subprocess.Popen(
                [self.cube_server_path, "-p", str(self.cube_port)],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
            )
        except Exception as e:
            log.error(f"Failed to start cube_server: {e}")
            sys.exit(1)

        # wait a moment for the cube server to start
        await tornado.gen.sleep(0.2)
        self.check_process()  # check if cube_server started successfully

        # stream to cube_server TCP socket
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.stream = tornado.iostream.IOStream(sock)
        try:
            await self.stream.connect(("127.0.0.1", self.cube_port))
            tornado.ioloop.IOLoop.current().add_callback(self.read_from_cube)
        except Exception as e:
            log.error(f"Failed to connect to cube_server on port {self.cube_port}: {e}")
            sys.exit(1)

    async def on_message(self, message):
        """ read incoming WebSocket messages from cube client (ws) and forward them to cube_server. """
        if isinstance(message, str):
            message = message.encode()
        await self.stream.write(message)

    async def read_from_cube(self):
        """ read incoming messages from cube_server and forward them to cube_client """
        try:
            while True:
                data = await self.stream.read_bytes(self.buffer_size, partial=True)
                if data:
                    self.write_message(data, binary=True)
                else:
                    break
        except tornado.iostream.StreamClosedError:
            self.close()
 
    def send_ping(self):
        try:
            self.ping(b'keepalive')
        except Exception:
            pass

    def on_close(self):
        self.ping_callback.stop()
        log.info("cube web socket has been closed")
        if hasattr(self, 'stream'):
            self.stream.close()
        if hasattr(self, 'cube_proc'):
            self.cube_proc.terminate()
            try:
                self.cube_proc.wait(timeout=5)
            except Exception:    
                pass

    def check_process(self):
        if self.cube_proc.poll() is not None:
            log.error("cube_server process terminated unexpectedly.")
            out, err = self.cube_proc.communicate(timeout=1)
            log.error(f"cube_server stdout: \n{out.decode() if out else ''}")
            log.error(f"cube_server stderr: \n{err.decode() if err else ''}")
            self.close()
            sys.exit(1)  # exit the web service if cube_server fails  

    def find_free_port(self):
        s = socket.socket()
        s.bind(('', 0))
        port = s.getsockname()[1]
        s.close()
        return port
       

def create_webservice( internal_socket=None ):
    cube_server_path = os.environ.get('CUBE_SERVER_PATH', 'cube_server')

    if not shutil.which(cube_server_path):
        log.error("\nError: cube_server is not in path and CUBE_SERVER_PATH environment variable is not set. "
                  "\nPlease set it to the path of the cube_server executable, which is part of cubelib.\n")
        sys.exit(1)
    static_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), "static")
    log.info(f"Static files served from: {static_dir}")
    return tornado.web.Application([
        (r"/static/(.*)",   StaticHandler, {"path": static_dir}),          # static files (cube.html,...)
        (r"/(favicon.ico)", StaticHandler, {"path": static_dir}),          # favicon (isn't used by jupyter)
        (r"/ws", WSProxyHandler, {"cube_server_path": cube_server_path,    # WebSocket proxy 
                                  "unix_socket": internal_socket}),  
    ])

def main():
    parser = argparse.ArgumentParser(description="Cube webservice\n CUBE_SERVER_PATH")
    parser.add_argument('--port', type=int, default=8008, help='Port to listen on (web service)')
    parser.add_argument('--unix-socket', type=str, default='', help='Path of external unix socket (web service)')
    parser.add_argument('--internal-socket', type=str, default='', help='Path of internal unix socket (cube_server)')
    args = parser.parse_args()   

    app = create_webservice(args.internal_socket)
    try:
        if args.unix_socket:
            http_server = tornado.httpserver.HTTPServer(app)
            sock = tornado.netutil.bind_unix_socket(args.unix_socket)
            http_server.add_socket(sock)
        else:
            app.listen(args.port)
            print(f"cube-webservice: client is available on http://localhost:{args.port}/static/cube.html")

        try:
            tornado.ioloop.IOLoop.current().start()
        except KeyboardInterrupt:
            tornado.ioloop.IOLoop.current().stop()

    except OSError as e:
        if e.errno == errno.EADDRINUSE:  # Address already in use
            if args.unix_socket:
                log.error(f"Error: Unix socket {args.unix_socket} is already in use. Please choose another path.")
            else:
                log.error(f"Error: Port {args.port} is already in use. Please choose another port.")
        else:
            log.error(f"Failed to start cube-webservice: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()
