##*************************************************************************##
##  CUBE        http://www.scalasca.org/                                   ##
##*************************************************************************##
##  Copyright (c) 2023-2024                                                ##
##  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          ##
##                                                                         ##
##  This software may be modified and distributed under the terms of       ##
##  a BSD-style license.  See the COPYING file in the package base         ##
##  directory for details.                                                 ##
##*************************************************************************##


# This cmake script will define the following variables:
#
# CUBELIB_REVISION
#   Latest commit sha.
# CUBELIB_FULL_NAME
#   e.g. "CubeLib-4.8.2-rc1" if repo is tagged or "CubeLib-branch-issueXX" if repo is a branch
# CUBELIB_VERSION
#   Full git tag e.g. "4.8.2-rc1" if tagged, otherwise 0.0.0
# CUBELIB_VERSION_MAJOR
#   Major version from CUBELIB_VERSION.
# CUBELIB_VERSION_MINOR
#   Minor version from CUBELIB_VERSION.
# CUBELIB_VERSION_PATCH
#   Patch version from CUBELIB_VERSION.


# todo: CUBELIB_VERSION_SUFFIX seems not to be used anywhere

# --- default values
set(PACKAGE_NAME CubeLib)
set(CUBELIB_VERSION_MAJOR 4)
set(CUBELIB_VERSION_MINOR 9)
set(CUBELIB_VERSION_PATCH 0)
set(CUBELIB_VERSION 4.9.0)
set( CUBELIB_LIBRARY_CURRENT 0 )
set( CUBELIB_LIBRARY_REVISION 0 )
set( CUBELIB_LIBRARY_AGE 0 )


# --- get version info from build-config ---

if ( EXISTS "${CMAKE_SOURCE_DIR}/build-config/VERSION" )
    file(READ "${CMAKE_SOURCE_DIR}/build-config/VERSION" CONTENT)

    # get CubeLib version info from build-config
    string(REGEX MATCH "package.major=([0-9]*)" TMP "${CONTENT}")
    set( CUBELIB_VERSION_MAJOR "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "package.minor=([0-9]*)" TMP "${CONTENT}")
    set( CUBELIB_VERSION_MINOR "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "package.bugfix=([0-9]*)" TMP "${CONTENT}")
    set( CUBELIB_VERSION_PATCH "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "package.suffix=([^\n]*)" TMP "${CONTENT}")
    if( NOT ${CMAKE_MATCH_1} STREQUAL "" )
      set( CUBELIB_VERSION_SUFFIX "-${CMAKE_MATCH_1}" )
    endif()
    set( CUBELIB_VERSION "${CUBELIB_VERSION_MAJOR}.${CUBELIB_VERSION_MINOR}.${CUBELIB_VERSION_PATCH}" )
    set( CUBELIB_FULL_NAME ${PACKAGE_NAME}-${CUBELIB_VERSION}${CUBELIB_VERSION_SUFFIX} )

    # get library version info from build-config
    string(REGEX MATCH "library.current=([0-9]*)" TMP "${CONTENT}")
    set( CUBELIB_LIBRARY_CURRENT "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "library.revision=([0-9]*)" TMP "${CONTENT}")
    set( CUBELIB_LIBRARY_REVISION "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "library.age=([0-9]*)" TMP "${CONTENT}")
    set( CUBELIB_LIBRARY_AGE "${CMAKE_MATCH_1}" )
endif()

# --- helper function to execute git commands ---
# executes a git command and returns the RESULT_VARIABLE
# ARGUMENTS: the arguments of the git executable
# OUTPUT: the result is written to this variable name
function(git_execute ARGUMENTS OUTPUT )
    separate_arguments(ARGUMENTS)
    execute_process(
        COMMAND ${GIT_EXECUTABLE} ${ARGUMENTS}
        WORKING_DIRECTORY ${CMAKE_SOURCE_DIR}
        OUTPUT_VARIABLE OUTPUT_STR
        RESULT_VARIABLE ERROR_CODE
        ERROR_VARIABLE ERR
        OUTPUT_STRIP_TRAILING_WHITESPACE
    )
    set(${OUTPUT} "${OUTPUT_STR}" PARENT_SCOPE)
endfunction()

# --- check if source is git-controlled ---
set( GIT_CONTROLLED false )
if ( EXISTS "${CMAKE_SOURCE_DIR}/.git" )
    find_package(Git)
    if ( ${GIT_FOUND} )
        set( GIT_CONTROLLED true )
    endif()
else()
    message(STATUS "Not a git-controlled repository.")
endif()

# --- get version info from git ---
if ( ${GIT_CONTROLLED} )
  git_execute("rev-parse --short HEAD" CUBELIB_REVISION )
  if (NOT "${CUBELIB_REVISION}" STREQUAL "")
        git_execute("describe --tags --dirty --match v*" TAG )
        if ( NOT "${TAG}" STREQUAL "") # git tagged release
              string(REGEX REPLACE "^v" "" CUBELIB_VERSION "${TAG}")
              string(REGEX MATCH "^([0-9]+)\\.([0-9]+)\\.([0-9]+)(-([a-z0-9]+))?" CUBELIB_VERSION ${CUBELIB_VERSION})
              set(CUBELIB_VERSION_MAJOR ${CMAKE_MATCH_1})
              set(CUBELIB_VERSION_MINOR ${CMAKE_MATCH_2})
              set(CUBELIB_VERSION_PATCH ${CMAKE_MATCH_3})
              set(CUBELIB_VERSION_SUFFIX ${CMAKE_MATCH_4})
              set(CUBELIB_FULL_NAME ${PACKAGE_NAME}-${CUBELIB_VERSION})
        else () # git branch -> CUBELIB_FULL_PACKAGE_NAME shows branch name
              git_execute("describe --all" BRANCH )
              string(REGEX MATCH "^.+/([a-z0-9]+)" MATCH_PACKAGE_NAME ${BRANCH})
              set(CUBELIB_FULL_NAME ${PACKAGE_NAME}-branch-${CMAKE_MATCH_1})
        endif ()
  endif ()
endif()

#message(STATUS "CubeLibVersion.cmake::: ${CUBELIB_VERSION} ${CUBELIB_REVISION} ${CUBELIB_FULL_NAME}" )

if(NOT DEFINED CUBELIB_FULL_NAME) # git not available or repo not git-controlled
    message(STATUS "Failed to determine CUBELIB_VERSION. Using default version \"${CUBELIB_VERSION}\".")
endif()
