/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



/**
 * \file cube_part.cpp
 * \brief Parts the system-tree of nodes/processes/threads of a cube.
 *
 */
/******************************************

   Performance Algebra Operation: PART (System-tree)

 *******************************************/
#include "config.h"

#include <cstdlib>
#include <iostream>
#include <string>
// #ifndef CUBE_COMPRESSED
#include <fstream>
// #else
#include "CubeZfstream.h"
// #endif
#include <sstream>
#include <unistd.h>

#include <string.h>

#include "Cube.h"
#include "CubeCnode.h"
#include "Filter.h"
#include "CubeMachine.h"
#include "CubeNode.h"
#include "CubeProcess.h"
#include "CubeThread.h"
#include "CubeMetric.h"
#include "CubeRegion.h"
#include "algebra4.h"
#include "CubeServices.h"
#include "CubeError.h"

using namespace std;
using namespace cube;
using namespace services;


void
ranks_parse( std::string& str,
             vector<bool>& );                         // function to parse inut "1-4, .. " and mark in vector partitioning... for threads and processes


/**
 * Main program.
 * - Check calling arguments
 * - Read the .cube input file.
 * - Calls cube_part(...) to part the systemtree.
 * - Saves the result in "-o outputfile" or "part.cube|.gz" file.
 * - end.
 */
int
main( int argc, char* argv[] )
{
    int             ch;
    bool            invert    = true;
    bool            hard_void = false;
    string          ranks;
    string          threadids;
    vector <string> inputs;
    const char*     output = "part";

    const string USAGE =
        "Partition some of system tree (processes) and void the remainder\n"
        "Usage: " + string( argv[ 0 ] ) +
        " [-h] [-I] [-R ranks] [-o output] <cubefile>\n"
        "  -I     Invert sense of partition\n"
        "  -R     List of process ranks for partition (e.g., \"0-3,7,13-\")\n"
        "  -T     List of thread ids for partition (e.g., \"0-3,7,13-\")\n"
        "  -F     Enforces hard partitioning, skipping voided processes in the resulting profile. Default: no -F\n"
        "  -o     Name of the output file (default: " + output + ")\n"
        "  -h     Help; Show this brief help message and exit.\n\n"
        "Report bugs to <" PACKAGE_BUGREPORT ">\n";

    while ( ( ch = getopt( argc, argv, "IFT:R:o:h" ) ) != -1 )
    {
        switch ( ch )
        {
            case 'I':
                invert = false;
                break;
            case 'R':
                ranks = optarg;
                break;
            case 'T':
                threadids = optarg;
                break;
            case 'F':
                std::cout << " Create cube file with only selected partition" << std::endl;
                hard_void = true;
                break;
            case 'o':
                output = optarg;
                break;
            case 'h':
            case '?':
                cerr << USAGE << endl;
                return EXIT_SUCCESS;
                break;
            default:
                cerr << USAGE << "\nError: Wrong arguments.\n";
                return EXIT_FAILURE;
        }
    }

    if ( argc - optind != 1 )
    {
        cerr << USAGE << "\nError: Wrong arguments.\n";
        return EXIT_FAILURE;
    }

    for ( int i = optind; i < argc; i++ )
    {
        inputs.push_back( argv[ i ] );
    }

    if ( ranks.empty() && threadids.empty() )
    {
        cerr << "Must specify partition criteria" << endl;
        return EXIT_FAILURE;
    }

    Cube* theCube = NULL;
    Cube* inCube  = NULL;

    try
    {
        cout << "Open the cube " << inputs[ 0 ] << "...";
        inCube = new Cube();
        inCube->openCubeReport( inputs[ 0 ] );
        cout << "done." << endl;

        if ( !hard_void )  // if not hard voiding -> old behaviour
        {
            cout << "(Soft) Copy the  cube object...";
            theCube = new Cube( *inCube, CUBE_DEEP_COPY, CUBE_ENFORCE_ZERO );
            cout << "done." << endl;
        }

        cout << "++++++++++++ Part operation begins ++++++++++++++++++++++++++" << endl;

        const vector<Process*>&  procv     = ( hard_void ) ? inCube->get_procv() : theCube->get_procv();
        const vector<Location*>& locationv = ( hard_void ) ? inCube->get_locationv() : theCube->get_locationv();
        vector<bool>             partproc( procv.size(), true );    // default value true == keep
        vector<bool>             partloc( locationv.size(), true ); // default value true == keep
        ranks_parse( ranks, partproc );
        ranks_parse( threadids, partloc );

        // voiding threads
        size_t void_locations = 0;
        if ( !threadids.empty() )  // we do voiding only if -T is specified.. otherwise keep , doesnt matter on -I
        {
            for ( size_t t = 0; t < locationv.size(); t++ )
            {
                Location* loc = locationv[ t ];
                if ( loc == nullptr )
                {
                    continue;
                }
                if ( partloc[ t ] == invert )  // default partloc is true == KEEP .... if invert is true -> keep->drop
                {
                    loc->set_name( loc->get_name() + " <VOID>" );
                    ++void_locations;
                }
            }
        }


        // voiding processes

        size_t void_procs = 0;
        for ( size_t p = 0; p < procv.size(); p++ )
        {
            Process* proc = procv[ p ];
            if ( proc == nullptr )
            {
                continue;
            }

            // first void parent in case all children are voided
            bool all_void = true;
            std::for_each( proc->get_children().begin(),  proc->get_children().end(), [ &all_void ]( cube::Vertex*& l ){
                cube::Location* _l = static_cast<cube::Location*>( l );
                if ( _l->get_name().find( "<VOID>" ) == std::string::npos )
                {
                    all_void = false;
                }
            } );
            if ( all_void ) // no single thread had no >VOID>
            {
                proc->set_name( proc->get_name() + " <VOID>" );
                ++void_procs;
            }

            // now void proc due to the commadn line option
            if ( !ranks.empty() )              // process selection only if -R is specified
            {
                if ( partproc[ p ] == invert ) // default partloc is true == KEEP .... if invert is true -> keep->drop
                {
                    proc->set_name( proc->get_name() + " <VOID>" );
                    ++void_procs;
                    // setting all children in void as process is being voided
                    bool all_void = false;
                    std::for_each( proc->get_children().begin(),  proc->get_children().end(), [ &all_void ]( cube::Vertex*& l ){
                        cube::Location* _l = static_cast<cube::Location*>( l );
                        if ( _l->get_name().find( "<VOID>" ) == std::string::npos ) // no VOID found
                        {
                            _l->set_name( _l->get_name() + " <VOID>" );
                        }
                    } );
                }
            }
        }


        const vector<Location*>&        thrdv = ( hard_void ) ? inCube->get_locationv() : theCube->get_locationv();
        vector<Thread*>::const_iterator tit   = thrdv.begin();
        for ( tit = thrdv.begin(); tit != thrdv.end(); ++tit )
        {
            int tid          = ( *tit )->get_id();
            int process_rank = ( *tit )->get_parent()->get_rank();
            partloc[ tid ]  = partloc[ tid ] && ( partproc[ process_rank ] != invert );
            void_locations += ( partloc[ tid ] ? 0 : 1 );
        }


        if ( hard_void )  // if hard voiding -> we create copy here knowing that in inCube we have "<VOID>ed" processes
        {
            cout << "(Hard) Copy the  cube object...";
            theCube = new Cube( *inCube, CUBE_DEEP_COPY, CUBE_ENFORCE_ZERO );
            cout << "done." << endl;
        }




        if ( !hard_void )
        {
            const vector<Metric*>&          metv   = theCube->get_metv();
            const vector<Cnode*>&           cnodev = theCube->get_cnodev();
            vector<Metric*>::const_iterator mit;
            for ( mit = metv.begin(); mit != metv.end(); ++mit )
            {
                if ( ( *mit ) == nullptr ) // vector with metrucs might have gaps
                {
                    continue;
                }
                std::cout << " check metric " << ( *mit )->get_uniq_name() << std::endl;
                if ( ( *mit )->get_val() == "VOID" )
                {
                    continue;                          // no values to modify
                }
                if ( ( *mit )->get_uniq_name() == "visits" )
                {
                    continue;                                   // retain visit counts
                }
                if ( ( *mit )->get_uniq_name() == "imbalance" ) // non-VOID!
                {
                    cerr << "Warning: Voiding metrics for " << ( *mit )->get_disp_name() << "!" << endl;
                    ( *mit )->set_val( "VOID" );
                }
                vector<Cnode*>::const_iterator cit;
                for ( cit = cnodev.begin(); cit != cnodev.end(); ++cit )
                {
                    for ( tit = thrdv.begin(); tit != thrdv.end(); ++tit )
                    {
                        int tid = ( *tit )->get_id();
                        if ( partloc[ tid ] )
                        {
                            continue;
                        }
                        theCube->set_sev( *mit, *cit, *tit, 0.0 );
                    }
                }
            }
        }
        cout << "++++++++++++ Part operation ends successfully ++++++++++++++++" << endl;

        cout << "Writing " << output << " ... " << flush;
        theCube->writeCubeReport( output );
        cout << "done." << endl;
    }
    catch ( const RuntimeError& error )
    {
        cerr << error.what() << endl;
        delete inCube;
        delete theCube;
        return EXIT_FAILURE;
    }

    inputs.clear();
    delete inCube;
    delete theCube;
    return EXIT_SUCCESS;
}


void
ranks_parse( std::string& ranges, std::vector<bool>& flags )
{
    // mark the processes based on given rank's range
    char* copy  = strdup( ranges.c_str() );
    char* token = strtok( copy, "," );
    while ( token )
    {
        char*  sep   = strchr( token, '-' );
        size_t start = atoi( token ), end = start;
        if ( sep == token )
        {
            start = 0;
        }
        if ( sep != NULL )
        {
            sep++;
            end = atoi( sep );
            if ( end == 0 )
            {
                end = flags.size() - 1;
            }
        }
        if ( end < start )
        {
            cerr << "Invalid range: " << token << endl;
        }
        else
        {
            for ( size_t p = start; p <= end; p++ )
            {
                flags[ p ] = false;
            }
        }
        token = strtok( NULL, "," );
    }
    free( copy );
}
