/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>

#include "POPTransferTest.h"


using namespace mpianalysis;

POPTransferTest::POPTransferTest( cube::CubeProxy* cube ) : popcalculation::PerformanceTest( cube )
{
    setName( " * * Transfer Efficiency" );
    setWeight( 1 );   // need to be adjusted
    max_total_time       = cube->getMetric( "max_total_time" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal" );
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        adjustForTest( cube );
    }
    max_total_time       = cube->getMetric( "max_total_time" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal" );

    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }
    cube::metric_pair metric;
    metric.first  = max_total_time;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );


    metric.first  = max_total_time_ideal;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_total_time_ideal.push_back( metric );
}


void
POPTransferTest::applyCnode( const cube::list_of_cnodes& cnodes,
                             const bool                    )
{
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        return;
    }
    setValue( analyze( cnodes ) );
}

double
POPTransferTest::analyze( const cube::list_of_cnodes& cnodes,
                          cube::LocationGroup*          ) const
{
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        return 0.;
    }
    cube::list_of_sysresources lsysres;

    cube::Value* max_total_time_value_ =
        cube->calculateValue( lmetrics, cnodes, lsysres );
    double max_total_time_value_d = max_total_time_value_->getDouble();
    delete max_total_time_value_;


    cube::Value* max_total_time_ideal_value_ =
        cube->calculateValue( lmax_total_time_ideal, cnodes, lsysres );
    double max_total_time_ideal_value_d = max_total_time_ideal_value_->getDouble();
    delete max_total_time_ideal_value_;

    return ( max_total_time_value_d <= std::numeric_limits<double>::min() ) ? 0. : max_total_time_ideal_value_d / max_total_time_value_d;
}





const std::string&
POPTransferTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPTransferTest::isActive() const
{
    return ( max_total_time_ideal != nullptr ) && ( max_total_time != nullptr );
};

bool
POPTransferTest::isIssue() const
{
    return false;
};

void
POPTransferTest::adjustForTest( cube::CubeProxy* cube ) const
{
    cube::Metric* _mpi_time = cube->getMetric( "mpi" );
    if ( _mpi_time->isInactive() )
    {
        return;
    }
    if ( scout_metrics_available( cube ) )
    {
        add_max_total_time_ideal( cube );
        add_max_total_time( cube );
    }
}



void
POPTransferTest::add_max_total_time_ideal( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    cube::Metric* _transfer_time_mpi = cube->getMetric( "transfer_time_mpi" );
    if ( _transfer_time_mpi == nullptr )
    {
        return;
    }
    cube::Metric* _met = cube->getMetric( "max_total_time_ideal" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximum total time in ideal network",
            "max_total_time_ideal",
            "DOUBLE",
            "sec",
            "",
            POP_TRANSFER_EFF_METRIC_URL,
            "Maximum total time in ideal network, ( execution - transfer_time_mpi )",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_INCLUSIVE,
            "metric::execution() - metric::transfer_time_mpi()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}


void
POPTransferTest::add_max_total_time( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    cube::Metric* _met = cube->getMetric( "max_total_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximum total time",
            "max_total_time",
            "DOUBLE",
            "sec",
            "",
            POP_TRANSFER_EFF_METRIC_URL,
            "Maximum total time, max( execution )",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_INCLUSIVE,
            "metric::execution()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}
