/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>
#include "POPSerialisationTest.h"


using namespace mpianalysis;

POPSerialisationTest::POPSerialisationTest( cube::CubeProxy* cube ) : popcalculation::PerformanceTest( cube )
{
    setName(  " * * Serialisation Efficiency" );
    setWeight( 1 );   // need to be adjusted
    comp_time            = cube->getMetric( "comp" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal" );
    if ( max_total_time_ideal == nullptr || comp_time == nullptr )
    {
        adjustForTest( cube );
    }
    comp_time            = cube->getMetric( "comp" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal" );

    if ( max_total_time_ideal == nullptr || comp_time == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }
    cube::metric_pair metric;
    metric.first  = comp_time;
    metric.second = cube::CUBE_CALCULATE_EXCLUSIVE;
    lmetrics.push_back( metric );


    metric.first  = max_total_time_ideal;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_total_time_ideal.push_back( metric );
}



void
POPSerialisationTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                  const bool                  direct_calculation  )
{
    ( void )direct_calculation; // not used here
    if ( max_total_time_ideal == nullptr || comp_time == nullptr )
    {
        return;
    }
    setValue( analyze( cnodes ) );
}

double
POPSerialisationTest::analyze( const cube::list_of_cnodes& cnodes,
                               cube::LocationGroup*        _lg  ) const
{
    ( void )_lg;
    if ( max_total_time_ideal == nullptr || comp_time == nullptr )
    {
        return 0.;
    }

    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );

    cube::value_container inclusive_values2;
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( lmax_total_time_ideal,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );

    const std::vector<cube::LocationGroup*>& _lgs      = cube->getLocationGroups();
    double                                   ser_value = std::numeric_limits<double>::lowest();
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
        {
            continue;
        }
        double _v1 =  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
        double _v2 =  inclusive_values2[ ( *iter )->get_sys_id() ]->getDouble();

        double _ser = ( _v2 <= std::numeric_limits<double>::min() ) ? 0. : _v1 / _v2;
        ser_value = std::max(  ser_value,  _ser  );
    }
    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values2.begin(), inclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values2.begin(), exclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    return ser_value;
}




const std::string&
POPSerialisationTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPSerialisationTest::isActive() const
{
    return ( max_total_time_ideal != nullptr ) && ( comp_time != nullptr );
};

bool
POPSerialisationTest::isIssue() const
{
    return false;
};

void
POPSerialisationTest::adjustForTest( cube::CubeProxy* cube ) const
{
    if ( scout_metrics_available( cube ) )
    {
        add_comp_time( cube );
        add_transfer_time_mpi( cube );
        add_max_total_time_ideal( cube );
    }
}


void
POPSerialisationTest::add_transfer_time_mpi( cube::CubeProxy* ) const
{
    add_mpi_time( cube );
    add_mpi_io_time( cube );
    add_wait_time_mpi( cube );
    cube::Metric* _met = cube->getMetric( "transfer_time_mpi" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Transfer time in MPI",
            "transfer_time_mpi",
            "DOUBLE",
            "sec",
            "",
            POP_SER_EFF_METRIC_URL,
            "Transfer time in MPI, (mpi - wait_time_mpi - mpi_io)",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "metric::mpi() - metric::wait_time_mpi() - metric::mpi_io()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}



void
POPSerialisationTest::add_max_total_time_ideal( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    cube::Metric* _transfer_time_mpi = cube->getMetric( "transfer_time_mpi" );
    if ( _transfer_time_mpi == nullptr )
    {
        return;
    }
    cube::Metric* _met = cube->getMetric( "max_total_time_ideal" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal total time in ideal network",
            "max_total_time_ideal",
            "DOUBLE",
            "sec",
            "",
            POP_TRANSFER_EFF_METRIC_URL,
            "Maximal total time in ideal network, ( execution - transfer_time_mpi )",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_INCLUSIVE,
            "metric::execution() - metric::transfer_time_mpi()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}
