/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <limits>
#include <algorithm>

#include "POPImbalanceTest.h"


using namespace mpianalysis;

POPImbalanceTest::POPImbalanceTest( cube::CubeProxy* cube ) : popcalculation::PerformanceTest( cube )
{
    setName(  " * Load Balance Efficiency" );
    setWeight( 1 );   // need to be adjusted
    pop_comp = cube->getMetric( "comp" );
    if ( pop_comp == nullptr )
    {
        adjustForTest( cube );
    }
    pop_comp = cube->getMetric( "comp" );
    if ( pop_comp == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }
    cube::list_of_sysresources lsysres = getRootsOfSystemTree();
    cube::metric_pair          metric;
    metric.first  = pop_comp;
    metric.second = cube::CUBE_CALCULATE_EXCLUSIVE;
    lmetrics.push_back( metric );
}



double
POPImbalanceTest::analyze( const cube::list_of_cnodes& cnodes,
                           cube::LocationGroup*          ) const
{
    if ( pop_comp == nullptr )
    {
        return 0.;
    }
    cube::value_container inclusive_values;
    cube::value_container exclusive_values;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values,
                               exclusive_values );

    const std::vector<cube::LocationGroup*>& _lgs                         = cube->getLocationGroups();
    double                                   pop_comp_imbalance_value_sum = 0.;
    double                                   pop_comp_imbalance_value_max = std::numeric_limits<double>::lowest();
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
        {
            continue;
        }
        double _v =  inclusive_values[ ( *iter )->get_sys_id() ]->getDouble();
        pop_comp_imbalance_value_sum += _v;
        pop_comp_imbalance_value_max  = std::max(
            pop_comp_imbalance_value_max,
            inclusive_values[ ( *iter )->get_sys_id() ]->getDouble()
            );
    }
    size_t cpu_locs_num = get_number_of_cpu_locations();

    std::for_each( inclusive_values.begin(), inclusive_values.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values.begin(), exclusive_values.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    return ( pop_comp_imbalance_value_max <= std::numeric_limits<double>::min() ) ? 0. :  pop_comp_imbalance_value_sum / cpu_locs_num / pop_comp_imbalance_value_max;
}




void
POPImbalanceTest::applyCnode( const cube::list_of_cnodes& cnodes,
                              const bool                  direct_calculation  )
{
    ( void )direct_calculation; // not used here
    if ( pop_comp == nullptr )
    {
        return;
    }
    setValue(  analyze( cnodes ) );
}


const std::string&
POPImbalanceTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPImbalanceTest::isActive() const
{
    return pop_comp != nullptr;
};

bool
POPImbalanceTest::isIssue() const
{
    return false;
};

void
POPImbalanceTest::adjustForTest( cube::CubeProxy* cube ) const
{
    cube::Metric* _comp = cube->getMetric( "comp" );
    if ( _comp == nullptr )
    {
        add_comp_time( cube );
    }
}
