/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_POP_HYBRID_COMMUNICATION_EFFICENCY_TEST_H
#define CUBELIB_POP_HYBRID_COMMUNICATION_EFFICENCY_TEST_H

#include <string>
#include "PerformanceTest.h"
#include "POPHybridSerialisationTest.h"
#include "POPHybridTransferTest.h"

namespace hybanalysis
{
class POPHybridProcessEfficiencyTest;
class POPHybridCommunicationEfficiencyTest : public popcalculation::PerformanceTest
{
    friend class POPHybridProcessEfficiencyTest;
private:
    cube::Metric*               max_omp_serial_comp_time;
    cube::Metric*               max_runtime;
    POPHybridSerialisationTest* pop_ser;
    POPHybridTransferTest*      pop_transeff;
    bool                        scout_cubex;
    bool                        original_scout_cubex;


    cube::list_of_metrics lmax_omp_ser_comp_time;

    double
    calculateForScout( const cube::list_of_cnodes& cnodes  ) const;


protected:

    const std::string&
    getCommentText() const;

    inline
    virtual
    std::string
    getHelpUrl() const
    {
        std::string active_text =
            "MPI Communication Efficiency (CommE) uses the following formula for direct evaluation: "
            "CommE = max(OpenMP_time + serial_computation_time) / max(runtime). "
            "This metric identifies inefficient code when communication time dominates useful computation. "
            "CommE is derived from two sub-metrics:\n"
            "1. Serialisation Efficiency - measures time processes wait at communication points\n"
            "2. Transfer Efficiency - evaluates data transfer relative to network capacity\n"
            "The final CommE value is calculated as: CommE = Serialisation_Efficiency * Transfer_Efficiency.\n"
            "These sub-metrics require Scalasca trace analysis to identify communication bottlenecks.";

        std::string not_active_text =
            "MPI Communication Efficiency (CommE) requires Score-P/Scalasca measurements to be available. "
            "If generated by other tools, the essential Time metric may be missing, preventing "
            "proper POP analysis of communication patterns.";
        return isActive() ? active_text : not_active_text;
    }

    virtual
    void
    adjustForTest( cube::CubeProxy* cube ) const;


public:
    POPHybridCommunicationEfficiencyTest(
        cube::CubeProxy*,
        POPHybridSerialisationTest* pop_ser,
        POPHybridTransferTest*      pop_transef );

    virtual
    ~POPHybridCommunicationEfficiencyTest()
    {
    };

    virtual
    double
    analyze( const cube::list_of_cnodes& cnodes,
             cube::LocationGroup*        _lg = nullptr ) const;

// ------ overview tests ---------

    void
    applyCnode( const cube::list_of_cnodes& cnodes,
                const bool                  direct_calculation = false  );

    bool
    isActive() const;

    bool
    isIssue() const;

    virtual
    std::list<PerformanceTest*>
    getPrereqs()
    {
        std::list<PerformanceTest*> prereqs;
        if ( pop_ser != nullptr && pop_transeff != nullptr )
        {
            prereqs.push_back( pop_transeff );
            prereqs.push_back( pop_ser );
        }
        return prereqs;
    }
};
};
#endif // ADVISER_RATING_WIDGET_H
