/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_POP_HYBRID_THREAD_EFFICIENCY_TEST_ADD_H
#define CUBELIB_POP_HYBRID_THREAD_EFFICIENCY_TEST_ADD_H

#include <string>
#include "PerformanceTest.h"
#include "POPHybridOMPRegionEfficiencyTestAdd.h"
#include "POPHybridAmdahlTestAdd.h"

namespace hybaddanalysis
{
class POPHybridParallelEfficiencyTestAdd;
class POPHybridThreadEfficiencyTestAdd : public popcalculation::PerformanceTest
{
    friend class POPHybridParallelEfficiencyTestAdd;
private:
    POPHybridAmdahlTestAdd*              amdahl_eff;
    POPHybridOmpRegionEfficiencyTestAdd* omp_region_eff;

    cube::Metric* pop_ser_comp_comp;
    cube::Metric* max_runtime;
    cube::Metric* pop_avg_omp;
    cube::Metric* pop_avg_omp_comp;

    cube::list_of_metrics lavg_omp_metrics;
    cube::list_of_metrics lavg_omp_comp_metrics;
    cube::list_of_metrics lavg_ser_metrics;
    cube::list_of_metrics lmax_runtime_metrics;
    void
    calculate();

    void
    calculate( const cube::list_of_cnodes& cnodes );

protected:

    virtual
    const std::string&
    getCommentText() const;

    inline
    virtual
    std::string
    getHelpUrl() const
    {
        std::string active_text =
            "Thread Efficiency considers two sources of inefficiency: \n"
            "\tSerial computation on the master outside OpenMP, i.e. reflects Amdahl's law\n"
            "\tInefficiencies within threads, e.g. serialisation across threads\n"
            "These two can be measured with Amdahl's Efficeincy and OpenMP region Efficiency respectively. \n"
            "Thread Efficeincy can be computed directly or as a sum of these two sub-metrics minus one:\n"
            "\tThread Efficiency=\n"
            "\t( max(runtime) - avg(time in OpenMP) + avg(time in useful computation within OpenMP) - \n"
            "\t  avg(idling time of OpenMP threads)z\n"
            "\t)/(max(runtime)) \n "
            "\t= Amdahl's Efficiency + OpenMP Region Efficiency - 1\n"
            "Where average idling time of OpenMP threads considers that threads are idling \n"
            "if only master thread is working and can be computed by following formula \n"
            "\taverage idling time of OpenMP threads=\n"
            "\tSUM (FROM process=0 TO num of processes) \n"
            "\t\t(serial computation x (number of threads per process - 1)/(number of all available threads))\n"
            "Moreover, average time in OpenMP computed as weighted arithmetic mean. \n"
            "If number of threads is equal across processes average time in OpenMP can be computed as ordinary arithmetic mean. ";
        std::string not_active_text =
            "Thread Efficiency metric is a basic POP metric and is available for every Score-P/Scalasca measurement. \n"
            "If Cube Report was produced by another tool than Score-P/Scalasca, it might have missing metric Time.\n"
            "In this case POP analysis is not possible.";

        return isActive() ? active_text : not_active_text;
    }

    virtual
    void
    adjustForTest( cube::CubeProxy* cube ) const;


public:
    POPHybridThreadEfficiencyTestAdd(   cube::CubeProxy*,
                                        POPHybridAmdahlTestAdd*              amdahl_eff,
                                        POPHybridOmpRegionEfficiencyTestAdd* omp_region_eff
                                        );

    virtual
    ~POPHybridThreadEfficiencyTestAdd()
    {
    };
// ------ overview tests ---------

    void
    applyCnode( const cube::list_of_cnodes& cnodes,
                const bool                  direct_calculation = false   );


    bool
    isActive() const;

    bool
    isIssue() const;

    virtual
    std::list<PerformanceTest*>
    getPrereqs()
    {
        std::list<PerformanceTest*> prereqs;
        if ( amdahl_eff != nullptr && omp_region_eff != nullptr )
        {
            prereqs.push_back( amdahl_eff );
            prereqs.push_back( omp_region_eff );
        }
        return prereqs;
    }
};
};
#endif // ADVISER_RATING_WIDGET_H
