/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>

#include "POPHybridSerialisationTestAdd.h"


using namespace hybaddanalysis;

POPHybridSerialisationTestAdd::POPHybridSerialisationTestAdd( cube::CubeProxy* cube ) : popcalculation::PerformanceTest( cube )
{
    setName( " + + + Serialisation Efficiency" );
    setWeight( 1 );   // need to be adjusted
    max_omp_serial_comp_time = cube->getMetric( "max_omp_serial_comp_time" );
    max_total_time_ideal     = cube->getMetric( "max_total_time_ideal_hyb" );
    if ( max_omp_serial_comp_time == nullptr || max_total_time_ideal == nullptr )
    {
        adjustForTest( cube );
    }
    max_omp_serial_comp_time = cube->getMetric( "max_omp_serial_comp_time" );
    max_total_time_ideal     = cube->getMetric( "max_total_time_ideal_hyb" );
    if ( max_omp_serial_comp_time == nullptr || max_total_time_ideal == nullptr )
    {
        setWeight( 0.2 );
        setValue( 0. );
        return;
    }
    cube::metric_pair metric;
    metric.first  = max_total_time_ideal;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );

    metric.first  = max_omp_serial_comp_time;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_omp_serial_comp_time.push_back( metric );
}


void
POPHybridSerialisationTestAdd::applyCnode( const cube::list_of_cnodes& cnodes,
                                           const bool                  direct_calculation  )
{
    ( void )direct_calculation; // not used here
    setValue(  analyze( cnodes ) );
}


double
POPHybridSerialisationTestAdd::analyze( const cube::list_of_cnodes& cnodes,
                                        cube::LocationGroup*          ) const
{
    if ( max_omp_serial_comp_time == nullptr || max_total_time_ideal == nullptr )
    {
        return 0.;
    }
    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );

    cube::value_container inclusive_values2;
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( lmax_omp_serial_comp_time,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );

    const std::vector<cube::LocationGroup*>& _lgs      = cube->getLocationGroups();
    double                                   ser_value = std::numeric_limits<double>::lowest();
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
        {
            continue;
        }
        double max_total_time_ideal  =  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
        double avg_max_omp_ser_value =  inclusive_values2[ ( *iter )->get_sys_id() ]->getDouble();
        double _ser                  = ( max_total_time_ideal <= std::numeric_limits<double>::min() ) ? 0. : avg_max_omp_ser_value / max_total_time_ideal;
        ser_value = std::max(  ser_value,  _ser  );
    }

    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values2.begin(), inclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values2.begin(), exclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );


    return ser_value;
}





const std::string&
POPHybridSerialisationTestAdd::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPHybridSerialisationTestAdd::isActive() const
{
    return ( max_omp_serial_comp_time != nullptr ) && ( max_total_time_ideal != nullptr );
};

bool
POPHybridSerialisationTestAdd::isIssue() const
{
    return false;
};

void
POPHybridSerialisationTestAdd::adjustForTest( cube::CubeProxy* cube ) const
{
    if ( scout_metrics_available( cube ) )
    {
        add_max_omp_and_ser_execution( cube );
        add_max_total_time_ideal_hyb( cube );
    }
}


void
POPHybridSerialisationTestAdd::add_max_total_time_ideal_hyb( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    add_transfer_time_mpi( cube );
    cube::Metric* _transfer_time_mpi = cube->getMetric( "transfer_time_mpi" );
    if ( _transfer_time_mpi == nullptr )
    {
        return;
    }

    cube::Metric* _met = cube->getMetric( "max_total_time_ideal_hyb" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal total time in ideal network, hybrid",
            "max_total_time_ideal_hyb",
            "DOUBLE",
            "sec",
            "",
            POP_TRANSFER_EFF_METRIC_URL,
            "Maximal total time in ideal network, ( execution - transfer_time_mpi )",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::execution() - metric::transfer_time_mpi()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}



void
POPHybridSerialisationTestAdd::add_transfer_time_mpi( cube::CubeProxy* ) const
{
    add_mpi_time( cube );
    add_mpi_io_time( cube );
    add_wait_time_mpi( cube );

    cube::Metric* _met = cube->getMetric( "transfer_time_mpi" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Transfer time in MPI",
            "transfer_time_mpi",
            "DOUBLE",
            "sec",
            "",
            POP_SER_EFF_METRIC_URL,
            "Transfer time in MPI, (mpi - wait_time_mpi - mpi_io)",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "metric::mpi() - metric::wait_time_mpi() - metric::mpi_io()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}
