/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>
#include <limits>
#include "POPHybridAmdahlTestAdd.h"
#include "POPCalculation.h"


using namespace hybaddanalysis;

POPHybridAmdahlTestAdd::POPHybridAmdahlTestAdd( cube::CubeProxy* cube ) : popcalculation::PerformanceTest( cube )
{
    setName( " + + Amdahl Efficiency" );
    setWeight( 1 );   // need to be adjusted
    pop_ser_comp_comp = cube->getMetric( "ser_comp_time" );
    if ( pop_ser_comp_comp == nullptr )
    {
        adjustForTest( cube );
    }
    pop_ser_comp_comp = cube->getMetric( "ser_comp_time" );
    if ( pop_ser_comp_comp == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }

    max_runtime = cube->getMetric( "max_runtime" );


    cube::metric_pair metric;
    metric.first  = max_runtime;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_runtime_metrics.push_back( metric );

    metric.first  = pop_ser_comp_comp;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );
}


void
POPHybridAmdahlTestAdd::applyCnode( const cube::list_of_cnodes& cnodes,
                                    const bool                  direct_calculation  )
{
    ( void )direct_calculation; // not used here
    if ( pop_ser_comp_comp == nullptr )
    {
        return;
    }
    cube::value_container inclusive_values1; // ser_comp
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );
    cube::value_container inclusive_values2; // max_ruintime
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( lmax_runtime_metrics,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );
    double max_runtime_value = inclusive_values2[ 0 ]->getDouble();

    const std::vector<cube::LocationGroup*>& _lgs              = cube->getLocationGroups();
    double                                   _avg_ser_comp_sum = 0.;
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
        {
            continue;
        }
        double _v =  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
        _avg_ser_comp_sum +=   _v * popcalculation::POPCalculation::get_num_thread_children( ( *iter ) );
    }


    size_t cpu_locs_num     = get_number_of_cpu_locations();
    double avg_ser_comp_sum = _avg_ser_comp_sum / cpu_locs_num;


    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values2.begin(), inclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values2.begin(), exclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );


    setValue( ( max_runtime_value  - avg_ser_comp_sum ) / ( max_runtime_value ) );
}


const std::string&
POPHybridAmdahlTestAdd::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPHybridAmdahlTestAdd::isActive() const
{
    return pop_ser_comp_comp != nullptr;
};

bool
POPHybridAmdahlTestAdd::isIssue() const
{
    return false;
};

void
POPHybridAmdahlTestAdd::adjustForTest( cube::CubeProxy* cube ) const
{
    add_avg_comp( cube );
}



void
POPHybridAmdahlTestAdd::add_avg_comp( cube::CubeProxy* ) const
{
    cube::Metric* _omp_time = cube->getMetric( "omp_time" );
    if ( _omp_time->isInactive() )
    {
        return;
    }
    add_max_omp_and_ser_execution( cube );
}
