/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023                                                     **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>

#include "POPWallTimeTest.h"


using namespace popcalculation;

POPWallTimeTest::POPWallTimeTest( cube::CubeProxy* cube ) : PerformanceTest( cube )
{
    setName( "Wall-clock time" );
    setWeight( 1 );   // need to be adjusted
    pop_walltime = cube->getMetric( "time" );
    if ( pop_walltime == nullptr )
    {
        adjustForTest( cube );
    }
    pop_walltime = cube->getMetric( "time" );
    maxValue     = 1.;
    if ( pop_walltime == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }
    cube::list_of_sysresources lsysres = getRootsOfSystemTree();
    cube::metric_pair          metric;
    metric.first  = pop_walltime;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );
}

void
POPWallTimeTest::applyCnode( const cube::list_of_cnodes& lcnodes,
                             const bool                  direct_calculation  )
{
    ( void )direct_calculation; // not used here
    if ( pop_walltime == nullptr )
    {
        return;
    }
    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;

    cube->getSystemTreeValues( lmetrics,
                               lcnodes,
                               inclusive_values1,
                               exclusive_values1 );

    const std::vector<cube::Location*>& _locs     = cube->getLocations();
    double                              _wall_sum = 0.;
    double                              _wall_min = std::numeric_limits<double>::max();
    double                              _wall_max = -std::numeric_limits<double>::max();
    double                              _wall_n   = 0.;
    for ( std::vector<cube::Location*>::const_iterator iter = _locs.begin(); iter != _locs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_TYPE_CPU_THREAD )  // skipp location which are not processes
        {
            continue;
        }
        _wall_n++;
        double _v =  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
        _wall_sum += _v;
        _wall_max  = std::max( _wall_max, _v );
        _wall_min  = std::min( _wall_min, _v );
    }
    double wall_avg = _wall_sum / _wall_n;


    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );



    setValues( wall_avg,  _wall_min, _wall_max );
}

const std::string&
POPWallTimeTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPWallTimeTest::isActive() const
{
    return pop_walltime != nullptr;
};

bool
POPWallTimeTest::isIssue() const
{
    return false;
};


void
POPWallTimeTest::adjustForTest( cube::CubeProxy* ) const
{
}
