/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <future>
#include <thread>

#include "POPIOEfficiencyTest.h"

using namespace popcalculation;

POPIOEfficiencyTest::POPIOEfficiencyTest( POPPosixIOTime* _posix_io,
                                          POPMpiIOTime*   _mpi_io ) :
    PerformanceTest( nullptr ),
    posix_io( _posix_io ),
    mpi_io( _mpi_io )

{
    setName(  "I/O Efficiency" );
    setWeight( 1 );   // need to be adjusted
    if ( posix_io == nullptr || mpi_io == nullptr )
    {
        setWeight( 0 );
        setValue( 0. );
        return;
    }
}

void
POPIOEfficiencyTest::calculate()
{
    if ( posix_io == nullptr || mpi_io == nullptr )
    {
        return;
    }
    if ( isActive() )
    {
        double posix_io_value = posix_io->value();
        double mpi_io_value   = mpi_io->value();
        setValue( 1 - ( posix_io_value  +  mpi_io_value ) );
    }
    else
    {
        setValue( -1 );
    }
}


static
void
posix_task_full_( POPPosixIOTime*             _posix_io,
                  const cube::list_of_cnodes& cnodes )
{
    if ( _posix_io )
    {
        _posix_io->applyCnode( cnodes );
    }
}

static
void
mpi_task_full_( POPMpiIOTime*               _mpi_io,
                const cube::list_of_cnodes& cnodes )
{
    if ( _mpi_io )
    {
        _mpi_io->applyCnode( cnodes );
    }
}


void
POPIOEfficiencyTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                 const bool                  direct_calculation  )
{
    auto posix_task_result = std::async( std::launch::async, posix_task_full_, posix_io, cnodes );
    auto mpi_task_result   = std::async( std::launch::async, mpi_task_full_, mpi_io, cnodes );

    posix_task_result.get();
    mpi_task_result.get();

    calculate();
}


const std::string&
POPIOEfficiencyTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPIOEfficiencyTest::isActive() const
{
    return posix_io != nullptr && posix_io->isActive() &&
           mpi_io != nullptr && mpi_io->isActive();
};

bool
POPIOEfficiencyTest::isIssue() const
{
    return false;
};



void
POPIOEfficiencyTest::adjustForTest( cube::CubeProxy* cube ) const
{
    ( void )cube;
}
