/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <future>
#include <thread>
#include "BSPOPHybridOMPParallelEfficiencyTest.h"

using namespace bscanalysis;

BSPOPHybridOMPParallelEfficiencyTest::BSPOPHybridOMPParallelEfficiencyTest( cube::CubeProxy*                      cube,
                                                                            BSPOPHybridParallelEfficiencyTest*    _par_eff,
                                                                            BSPOPHybridMPIParallelEfficiencyTest* _mpi_eff
                                                                            ) :
    popcalculation::PerformanceTest( cube ),
    par_eff( _par_eff ),
    mpi_eff( _mpi_eff )

{
    setName( "OpenMP Parallel Efficiency" );
    setWeight( 1 );   // need to be adjusted

    if ( par_eff == nullptr || mpi_eff == nullptr  ||
         ( !par_eff->isActive() && !mpi_eff->isActive() )
         )
    {
        setWeight( 0.2 );
        setValue( 0. );
        return;
    }
}

void
BSPOPHybridOMPParallelEfficiencyTest::calculate()
{
    if ( par_eff == nullptr || mpi_eff == nullptr  )
    {
        return;
    }
    double par_eff_value = par_eff->value();
    double mpi_eff_value = mpi_eff->value();
    setValue(
        ( ( par_eff->isActive() ) ? par_eff_value : 1. ) /
        ( ( mpi_eff->isActive() ) && ( mpi_eff_value > std::numeric_limits<double>::min() ) ? mpi_eff_value : 1. )
        );
}

static
void
hyb_par_task_full_( BSPOPHybridParallelEfficiencyTest* _par_eff,
                    const cube::list_of_cnodes&        cnodes )
{
    if ( _par_eff )
    {
        _par_eff->applyCnode( cnodes );
    }
}

static
void
hyb_mpi_par_task_full_( BSPOPHybridMPIParallelEfficiencyTest* _mpi_eff,
                        const cube::list_of_cnodes&           cnodes )
{
    if ( _mpi_eff )
    {
        _mpi_eff->applyCnode( cnodes );
    }
}



void
BSPOPHybridOMPParallelEfficiencyTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                                  const bool                   )
{
    auto hyb_par_task_result = std::async( std::launch::async, hyb_par_task_full_, par_eff, cnodes );
    auto hyb_mpi_task_result = std::async( std::launch::async, hyb_mpi_par_task_full_, mpi_eff, cnodes );

    hyb_par_task_result.get();
    hyb_mpi_task_result.get();

    calculate();
}


const std::string&
BSPOPHybridOMPParallelEfficiencyTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
BSPOPHybridOMPParallelEfficiencyTest::isActive() const
{
    return par_eff != nullptr &&
           mpi_eff != nullptr;
};

bool
BSPOPHybridOMPParallelEfficiencyTest::isIssue() const
{
    return false;
};



void
BSPOPHybridOMPParallelEfficiencyTest::adjustForTest( cube::CubeProxy* cube ) const
{
    ( void )cube;
}
