/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>

#include "BSPOPHybridMPIParallelEfficiencyTest.h"

using namespace bscanalysis;

BSPOPHybridMPIParallelEfficiencyTest::BSPOPHybridMPIParallelEfficiencyTest( cube::CubeProxy* cube ) :
    popcalculation::PerformanceTest( cube )

{
    setName( "MPI Parallel Efficiency" );
    setWeight( 1 );   // need to be adjusted


    non_mpi_time = cube->getMetric( "non_mpi_time" );
    if ( non_mpi_time == nullptr )
    {
        adjustForTest( cube );
    }
    non_mpi_time = cube->getMetric( "non_mpi_time" );
    if ( non_mpi_time == nullptr  )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }


    cube::metric_pair metric;
    metric.first  = non_mpi_time;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );


    max_runtime = cube->getMetric( "max_runtime" );

    metric.first  = max_runtime;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    max_runtime_metrics.push_back( metric );
}

void
BSPOPHybridMPIParallelEfficiencyTest::calculate( const cube::list_of_cnodes& cnodes )
{
    if ( non_mpi_time == nullptr )
    {
        return;
    }
    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );
    cube::value_container inclusive_values2;
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( max_runtime_metrics,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );

    double                                   max_runtime_value = inclusive_values2[ 0 ]->getDouble();
    const std::vector<cube::LocationGroup*>& _lgs              = cube->getLocationGroups();
    double                                   _avg_non_mpi_sum  = 0.;
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
        {
            continue;
        }
        _avg_non_mpi_sum +=  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
    }

    size_t cpu_locs_num = get_number_of_cpu_locations();

    double avg_non_mpi_time = _avg_non_mpi_sum / cpu_locs_num;

    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values2.begin(), inclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values2.begin(), exclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );


    setValue(  avg_non_mpi_time / max_runtime_value );
}


void
BSPOPHybridMPIParallelEfficiencyTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                                  const bool )
{
    calculate( cnodes );
}


const std::string&
BSPOPHybridMPIParallelEfficiencyTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
BSPOPHybridMPIParallelEfficiencyTest::isActive() const
{
    return non_mpi_time != nullptr;
};

bool
BSPOPHybridMPIParallelEfficiencyTest::isIssue() const
{
    return false;
};



void
BSPOPHybridMPIParallelEfficiencyTest::adjustForTest( cube::CubeProxy* cube ) const
{
    ( void )cube;
}
