/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>
#include <limits>
#include "BSPOPHybridLoadBalanceTest.h"


using namespace bscanalysis;

BSPOPHybridLoadBalanceTest::BSPOPHybridLoadBalanceTest( cube::CubeProxy* cube,  BSPOPHybridCommunicationEfficiencyTest* _hyb_comm_eff ) : popcalculation::PerformanceTest( cube ), hyb_comm_eff( _hyb_comm_eff )
{
    setName( " * Hybrid Load Balance Efficiency" );
    setWeight( 1 );   // need to be adjusted
    pop_avg_comp = cube->getMetric( "avg_comp" );
    if ( pop_avg_comp == nullptr )
    {
        adjustForTest( cube );
    }
    pop_avg_comp = cube->getMetric( "avg_comp" );
    if ( pop_avg_comp == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }

    max_comp = cube->getMetric( "max_comp_time" );

    cube::metric_pair metric;
    metric.first  = pop_avg_comp;
    metric.second = cube::CUBE_CALCULATE_EXCLUSIVE;
    lmetrics.push_back( metric );

    metric.first  = max_comp;
    metric.second = cube::CUBE_CALCULATE_EXCLUSIVE;
    max_comp_metrics.push_back( metric );
}



void
BSPOPHybridLoadBalanceTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                        const bool )
{
    if ( pop_avg_comp == nullptr )
    {
        return;
    }
    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );
    cube::value_container inclusive_values2;
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( max_comp_metrics,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );
    double avg_comp_value = inclusive_values1[ 0 ]->getDouble();
    double max_comp_value = inclusive_values2[ 0 ]->getDouble();
    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values2.begin(), inclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values2.begin(), exclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );

    setValue( ( max_comp_value <= std::numeric_limits<double>::min() ) ? 0. :  avg_comp_value / max_comp_value );
}


const std::string&
BSPOPHybridLoadBalanceTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
BSPOPHybridLoadBalanceTest::isActive() const
{
    return pop_avg_comp != nullptr;
};

bool
BSPOPHybridLoadBalanceTest::isIssue() const
{
    return false;
};

void
BSPOPHybridLoadBalanceTest::adjustForTest( cube::CubeProxy* cube ) const
{
    add_avg_comp( cube );
}



void
BSPOPHybridLoadBalanceTest::add_avg_comp( cube::CubeProxy* ) const
{
    add_comp_time( cube );
    cube::Metric* _met = cube->getMetric( "avg_comp" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "AVG computation",
            "avg_comp",
            "DOUBLE",
            "",
            "",
            BSPOP_TRANSFER_EFF_METRIC_URL,
            "Calculates average computation time",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::comp()/${cube::#locations}",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}
