/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>
#include "PerformanceTest.h"
#include "CubeRegion.h"

using namespace popcalculation;

std::string PerformanceTest::no_comment = "-- no comment --";

void
PerformanceTest::adjustForTest( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "__time_hierarchy_ghost" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "__time_hierarchy_ghost",
            "__time_hierarchy_ghost",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#non_existent",
            "Dummy metric to execute init sequence",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "1",
            cubepl_time_init_str,
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }

    _met = cube->getMetric( "__service_parallel_marker" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "__service_parallel_marker",
            "__service_parallel_marker",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#non_existent",
            "Dummy metric to display ${without_wait_state} variable",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_comp}[${calculation::callpath::id}]",
            " {"
            "        ${i} = 0;"
            "        global(omp_comp);"
            "        while (${i} < ${cube::#callpaths})"
            "        {"
            "        ${omp_comp}[${i}]=0;"
            "        ${callpath} = ${i};"
            "        while( (${callpath} != -1) and ( not( (${cube::region::role}[${cube::callpath::calleeid}[${callpath}] ] eq \"parallel\" ) or (${cube::region::role}[${cube::callpath::calleeid}[${callpath}] ] eq \"task\" ) or (${cube::region::role}[${cube::callpath::calleeid}[${callpath}] ] eq \"task create\" )) ) )"
            "        {"
            "        ${callpath} = ${cube::callpath::parent::id}[${callpath}];"
            "        };"
            "        if (${callpath} != -1)"
            "        {"
            "        ${omp_comp}[${i}]=1;"
            "        };"
            "        ${i} = ${i} + 1;"
            "        };"
            "        return 0;"
            "}",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }

    // initiate creation of advisor metrics
    add_comp_time( cube );
}

void
PerformanceTest::finalizePrepsForTest( cube::CubeProxy* cube )
{
    cube::Metric* _met = cube->getMetric( "__time_hierarchy_deactivating_ghost" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "__time_hierarchy_deactivating_ghost",
            "__time_hierarchy_deactivating_ghost",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#non_existent",
            "Dummy metric to execute init sequence",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "1",
            cubepl_time_set_str,
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

bool
PerformanceTest::scout_metrics_available( cube::CubeProxy* cube ) const
{
    if ( cube->getMetric( "mpi_latesender" ) != nullptr )
    {
        return true;
    }
    if ( cube->getMetric( "mpi_latereceiver" ) != nullptr )
    {
        return true;
    }
    if ( cube->getMetric( "mpi_earlyreduce" ) != nullptr )
    {
        return true;
    }
    if ( cube->getMetric( "mpi_earlyscan" ) != nullptr )
    {
        return true;
    }
    if ( cube->getMetric( "mpi_latebroadcast" ) != nullptr )
    {
        return true;
    }
    if ( cube->getMetric( "mpi_wait_nxn" ) != nullptr )
    {
        return true;
    }
    if ( cube->getMetric( "mpi_barrier_wait" ) != nullptr )
    {
        return true;
    }
    if ( cube->getMetric( "mpi_finalize_wait" ) != nullptr )
    {
        return true;
    }
    return false;
}




void
PerformanceTest::add_mpi_comp_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _execution = cube->getMetric( "execution" );
    cube::Metric* _mpi       = cube->getMetric( "mpi" );
    if ( _execution == nullptr )
    {
        add_execution_time( cube );
    }
    if ( _mpi == nullptr )
    {
        add_mpi_time( cube );
    }
    cube::Metric* _comp = cube->getMetric( "mpi_comp" );
    if ( _comp == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "MPI Computation",
            "mpi_comp",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#comp",
            "Time spent OUTSIDE of MPI routines.",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::execution() - metric::mpi()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_comp_time( cube::CubeProxy* cube ) const
{
    add_max_time( cube );
    add_execution_time( cube );
    add_omp_time( cube );
    add_mpi_time( cube );
    add_mpi_indicator( cube );
    add_shmem_time( cube );
    add_pthread_time( cube );
    add_openacc_time( cube );
    add_hip_time( cube );
    add_opencl_time( cube );
    add_cuda_time( cube );
    add_libwrap_time( cube );
    add_total_io_time( cube );

    add_kernels_execution_time( cube );


    cube::Metric* _comp = cube->getMetric( "comp" );
    //bool          omp_as_ghost = false;
    if ( _comp == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "Computation",
            "comp",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#comp",
            "Time spent on computation",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::execution() - metric::mpi() - metric::shmem_time() - metric::omp_time() - metric::pthread_time() - metric::openacc_time() - metric::opencl_time() - metric::cuda_time() - metric::hip_time()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );

        //omp_as_ghost = true;
    }


    add_gpu_comp_time( cube );
    add_max_comp_time( cube );
    add_omp_execution( cube );
    add_avg_omp_time( cube );
    add_max_omp_time( cube );
    add_omp_comp_time( cube );
    add_omp_non_wait_time( cube );
}



void
PerformanceTest::add_gpu_comp_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "gpu_comp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "GPU computation time",
            "gpu_comp_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#gpu_comp_time",
            "Time spent on GPU computation",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            " {"
            "    ${m}=0;"
            "    if ( ${cube::location::type}[${calculation::sysres::id} ]  == 1 )"
            "    {"
            "    ${m} = metric::time(*,*) ;"
            "    };"
            "    return ${m};"
            " }",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_ser_comp_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "ser_comp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Serial computation time",
            "ser_comp_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#serial_comp_time",
            "Time spent on computation in serial part of calculation ",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "(1-${omp_comp}[${calculation::callpath::id}])*metric::comp()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
    add_max_omp_and_ser_execution( cube );
}



void
PerformanceTest::add_omp_comp_time( cube::CubeProxy* cube, bool as_ghost ) const
{
    cube::Metric* _omp_comp_time = cube->getMetric( "omp_comp_time" );
    cube::Metric* _comp          = nullptr; // cube->getMetric( "comp" );
    if ( _omp_comp_time == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "OMP computation time",
            "omp_comp_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#omp_comp_time",
            "Time spent on computation within OpenMP regions",
            _comp,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_comp}[${calculation::callpath::id}]*metric::comp()",
            "",
            "",
            "",
            "",
            true,
            ( as_ghost ) ? cube::CUBE_METRIC_GHOST : cube::CUBE_METRIC_NORMAL
            );
        _met->def_attr( "origin", "advisor" );
    }

    add_ser_comp_time(  cube );
    add_max_omp_comp_time( cube );
    add_avg_omp_comp_io_time( cube );
}



void
PerformanceTest::add_max_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "max_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal time, max_time",
            "max_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#execution",
            "aximal run time time, max( time )",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "metric::time()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_execution_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "execution" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Execution",
            "execution",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#execution",
            "Execution time (does not include time allocated for idle threads)",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${execution}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }

    _met = cube->getMetric( "max_runtime" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Max Runtime",
            "max_runtime",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#max_runtime",
            "Execution time (does not include time allocated for idle threads)",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::execution()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_max_comp_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "max_comp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Max Computation Time",
            "max_comp_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#max_comp_time",
            "Maximal Computation time over all locations",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::comp()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_mpi_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "mpi" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "MPI",
            "mpi",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#mpi",
            "Time spent in MPI calls",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${mpi}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
    add_parallel_mpi_time( cube );
    add_serial_mpi_time( cube );
    add_non_mpi_time( cube );
}

void
PerformanceTest::add_non_mpi_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "non_mpi_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Not MPI Time",
            "non_mpi_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#mpi",
            "Time spent in Non MPI calls",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "(1-${mpi}[${calculation::callpath::id}]) * ( metric::time(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
    add_max_non_mpi_time( cube );
}

void
PerformanceTest::add_max_non_mpi_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "max_non_mpi_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal Not MPI Time",
            "max_non_mpi_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#mpi",
            "Maximal Time spent in Non MPI calls",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::non_mpi_time()",
            "",
            "",
            "",
            "max(arg1,arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_mpi_indicator( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "mpi_indicator" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "MPI Indicator",
            "mpi_indicator",
            "DOUBLE",
            "",
            "",
            TIME_METRIC_URL "#mpi_indicator",
            "1 if MPI has been executed on this call path, 0 - if no",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            " ${mpi}[${calculation::callpath::id}] * ( metric::mpi(e,*) /metric::mpi(e,*) )",
            "",
            "max( arg1 , arg2)",
            "max( arg1 , arg2)",
            "max( arg1 , arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_mpi_io_time( cube::CubeProxy* cube ) const
{
    add_mpi_io_individual_time( cube );
    add_mpi_io_collective_time( cube );
    cube::Metric* _met = cube->getMetric( "mpi_io" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "MPI IO",
            "mpi_io",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#mpi_io",
            "Time spent in MPI file I/O calls",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::mpi_io_individual() + metric::mpi_io_collective()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_mpi_io_individual_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "mpi_io_individual" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "MPI Individual",
            "mpi_io_individual",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#mpi_io_individual",
            "Time spent in individual MPI file I/O calls",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "${mpi_file_individual}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_mpi_io_collective_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "mpi_io_collective" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "MPI Collective",
            "mpi_io_collective",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#mpi_io_collective",
            "Time spent in collective MPI file I/O calls",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "${mpi_file_collective}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_wait_time_mpi( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "wait_time_mpi" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Waiting time in MPI",
            "wait_time_mpi",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#mpi_waiting_time",
            "Waiting time in MPI, (mpi_latesender + mpi_latereceiver + mpi_earlyreduce + mpi_earlyscan + mpi_latebroadcast + mpi_wait_nxn + mpi_barrier_wait + mpi_finalize_wait)",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "metric::mpi_latesender() + metric::mpi_latereceiver() + metric::mpi_earlyreduce() + metric::mpi_earlyscan() + metric::mpi_latebroadcast() + metric::mpi_wait_nxn() + metric::mpi_barrier_wait() + metric::mpi_finalize_wait()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}




void
PerformanceTest::add_shmem_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "shmem_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "SHMEM",
            "shmem_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#shmem",
            "Time spent in SHMEM calls",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${shmem}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_omp_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "omp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "OpenMP",
            "omp_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#omp_time",
            "Time spent in the OpenMP run-time system and API",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            " metric::omp_management() +  "
            " ${omp}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e)) ",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_omp_non_wait_time( cube::CubeProxy* cube, bool as_ghost ) const
{
    add_parallel_execution_time( cube, as_ghost );
    cube::Metric* _met = cube->getMetric( "omp_non_wait_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "OMP non-wait time",
            "omp_non_wait_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#omp_comp_time",
            "Time spent on computation within OpenMP regions",
            nullptr, //cube->getMetric( "par_execution_time" ),
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_comp}[${calculation::callpath::id}]*(metric::comp())",
            "",
            "",
            "",
            "",
            true,
            ( as_ghost ) ? cube::CUBE_METRIC_GHOST : cube::CUBE_METRIC_NORMAL
            );
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_parallel_execution_time( cube::CubeProxy* cube, bool ) const
{
    add_execution_time( cube );
    cube::Metric* _met = cube->getMetric( "par_execution_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Parallel execution time",
            "par_execution_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#serial_comp_time",
            "Time spent on computation in parallel part of execution ",
            nullptr, //cube->getMetric( "execution" ),
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_comp}[${calculation::callpath::id}]*metric::execution()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );

        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_parallel_mpi_time( cube::CubeProxy* cube, bool ) const
{
    cube::Metric* _met = cube->getMetric( "par_mpi_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Parallel MPI time",
            "par_mpi_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#parallel_mpi_time",
            "Time spent oin MPI in parallel part of execution ",
            nullptr, //cube->getMetric( "execution" ),
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_comp}[${calculation::callpath::id}]*metric::mpi()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );

        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_serial_mpi_time( cube::CubeProxy* cube, bool ) const
{
    cube::Metric* _met = cube->getMetric( "ser_mpi_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Serial MPI time",
            "ser_mpi_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#serial_comp_time",
            "Time spent in MPI in serial part of execution ",
            nullptr, //cube->getMetric( "execution" ),
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "(1-${omp_comp}[${calculation::callpath::id}])*metric::mpi()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );

        _met->def_attr( "origin", "advisor" );
    }
    add_max_serial_mpi_time( cube );
}

void
PerformanceTest::add_max_serial_mpi_time( cube::CubeProxy* cube, bool ) const
{
    cube::Metric* _met = cube->getMetric( "max_ser_mpi_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Max Serial MPI time",
            "max_ser_mpi_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#serial_comp_time",
            "Maximal time spent in MPI in serial part of execution ",
            nullptr, //cube->getMetric( "execution" ),
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "(1-${omp_comp}[${calculation::callpath::id}])*metric::mpi()",
            "",
            "",
            "",
            "max(arg1,arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );

        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_omp_execution( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "omp_execution_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "OpenMP Execution",
            "omp_execution_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#omp_time",
            "Time spent in the OpenMP region",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_comp}[${calculation::callpath::id}] * ( metric::execution() ) ",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_omp_io_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "omp_io_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "OpenMP IO Time",
            "omp_io_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#omp_io_time",
            "Time spent during IO in the OpenMP region",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_time}[${calculation::callpath::id}] * ( metric::mpi_io() ) ",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_serial_io_time( cube::CubeProxy* cube ) const
{
    add_mpi_io_time( cube );
    add_posix_io_time( cube );
    cube::Metric* _met = cube->getMetric( "serial_io_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Serial IO Time",
            "serial_io_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#serial_io_time",
            "Time spent during IO in the OpenMP region",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "metric::mpi_io() + metric::posix_io() ",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_total_io_time( cube::CubeProxy* cube ) const
{
    add_omp_io_time( cube );
    add_serial_io_time( cube );
    cube::Metric* _met = cube->getMetric( "total_io_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Total IO Time",
            "total_io_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#total_io_time",
            "Total time spent during IO",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::serial_io() + metric::omp_io()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}
void
PerformanceTest::add_max_omp_and_ser_execution( cube::CubeProxy* cube ) const
{
    add_serial_io_time( cube );
    cube::Metric* _met = cube->getMetric( "max_omp_serial_comp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Max OpenMP Execution and Serial Execution",
            "max_omp_serial_comp_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#omp_time",
            "Maximal",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            " metric::ser_comp_time() + metric::omp_execution_time() + metric::serial_io_time()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_max_omp_time( cube::CubeProxy* ) const
{
    cube::Metric* _met = cube->getMetric( "max_omp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal OMP run time",
            "max_omp_time",
            "DOUBLE",
            "sec",
            "",
            POP_LB_EFF_METRIC_URL,
            "Maximal OpenMP time, max( comp )",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::omp_execution_time()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}
void
PerformanceTest::add_max_omp_comp_time( cube::CubeProxy* ) const
{
    cube::Metric* _met = cube->getMetric( "max_omp_comp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal OMP computation time",
            "max_omp_comp_time",
            "DOUBLE",
            "sec",
            "",
            POP_LB_EFF_METRIC_URL,
            "Maximal OpenMP Computation time, max( comp )",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::omp_comp_time()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_avg_omp_comp_io_time( cube::CubeProxy* cube ) const
{
    add_omp_io_time( cube );
    cube::Metric* _met = cube->getMetric( "avg_omp_comp_io_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Average OMP computation runtime",
            "avg_omp_comp_io_time",
            "DOUBLE",
            "sec",
            "",
            POP_LB_EFF_METRIC_URL,
            "Average computation OpenMP time, avg( omp comp )",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "( metric::omp_comp_time() + metric::omp_io() )/${cube::#locations}",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_avg_omp_time( cube::CubeProxy* ) const
{
    cube::Metric* _met = cube->getMetric( "avg_omp_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Average OMP runtime",
            "avg_omp_time",
            "DOUBLE",
            "sec",
            "",
            POP_LB_EFF_METRIC_URL,
            "Average OpenMP execution time, avg( omp runtime )",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::omp_execution_time()/${cube::#locations}",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_pthread_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "pthread_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "POSIX threads",
            "pthread_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#pthread_time",
            "Time spent in the POSIX threads API",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${pthread}[${calculation::callpath::id}] * (metric::time(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_posix_io_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "posix_io" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "POSIX threads",
            "posix_io",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#pthread_time",
            "Time spent in the POSIX IO",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${posix_io}[${calculation::callpath::id}] * (metric::time(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_opencl_time( cube::CubeProxy* cube ) const
{
    add_opencl_kernel_execution_time( cube );
    cube::Metric* _opencl_kernel_execution = cube->getMetric( "opencl_time" );
    if ( _opencl_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "OpenCL",
            "opencl_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#opencl_time",
            "Time spent in the OpenCL run-time system, API and on device",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${opencl}[${calculation::callpath::id}] * (metric::time(e) - metric::opencl_kernel_executions(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_opencl_kernel_execution_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _opencl_kernel_execution = cube->getMetric( "opencl_kernel_executions" );
    if ( _opencl_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "OpenCL kernels",
            "opencl_kernel_executions",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#opencl_kernel_executions",
            "Time spent executing OpenCL kernels",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${opencl_kernel_executions}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_openacc_time( cube::CubeProxy* cube ) const
{
    add_openacc_kernel_execution_time( cube );
    cube::Metric* _openacc_kernel_execution = cube->getMetric( "openacc_time" );
    if ( _openacc_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "OpenACC",
            "openacc_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#openacc_time",
            "Time spent in the OpenACC run-time system, API and on device",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${openacc}[${calculation::callpath::id}] * (metric::time(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_openacc_kernel_execution_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _openacc_kernel_execution = cube->getMetric( "openacc_kernel_executions" );
    if ( _openacc_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "OpenACC kernels",
            "openacc_kernel_executions",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#openacc_kernel_executions",
            "Time spent executing OpenACC kernels",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${openacc_kernel_executions}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_cuda_time( cube::CubeProxy* cube ) const
{
    add_cuda_kernel_execution_time( cube );
    cube::Metric* _met = cube->getMetric( "cuda_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "CUDA",
            "cuda_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#cuda_time",
            "Time spent in the CUDA run-time system, API and on device",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${cuda}[${calculation::callpath::id}]* ( metric::time(e) - metric::cuda_kernel_executions(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_cuda_kernel_execution_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _cuda_kernel_execution = cube->getMetric( "cuda_kernel_executions" );
    if ( _cuda_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "CUDA kernels",
            "cuda_kernel_executions",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#cuda_kernel_executions",
            "Time spent executing CUDA kernels",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${cuda_kernel_executions}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_libwrap_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "libwrap_time" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Wrapped libraries",
            "libwrap_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#libwrap_time",
            "Time spent in wrapped libraries",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${libwrap}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_hip_time( cube::CubeProxy* cube ) const
{
    add_hip_kernel_execution_time( cube );
    cube::Metric* _openacc_kernel_execution = cube->getMetric( "hip_time" );
    if ( _openacc_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "HIP",
            "hip_time",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#hip_time",
            "Time spent in the HIP API calls",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${hip}[${calculation::callpath::id}] * (metric::time(e) - metric::hip_kernel_executions(e) - metric::omp_idle_threads(e))",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}
void
PerformanceTest::add_hip_kernel_execution_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _hip_kernel_execution = cube->getMetric( "hip_kernel_executions" );
    if ( _hip_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "CUDA kernels",
            "hip_kernel_executions",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#hip_kernel_executions",
            "Time spent executing HIP kernels",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${hip_kernel_executions}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}

void
PerformanceTest::add_omp_target_kernel_execution_time( cube::CubeProxy* cube ) const
{
    cube::Metric* _omp_target_kernel_execution = cube->getMetric( "omp_target_kernel_executions" );
    if ( _omp_target_kernel_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "OMP Target kernels",
            "omp_target_kernel_executions",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#omp_target_kernel_executions",
            "Time spent executing OMP Target kernels",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${omp_target}[${calculation::callpath::id}] * ( metric::time(e) - metric::omp_idle_threads(e) )",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_kernels_execution_time( cube::CubeProxy* cube ) const
{
    add_omp_target_kernel_execution_time( cube );
    add_hip_kernel_execution_time( cube );
    add_cuda_kernel_execution_time( cube );
    add_opencl_kernel_execution_time( cube );
    add_openacc_kernel_execution_time( cube );


    cube::Metric* _all_kernels_execution = cube->getMetric( "all_kernels_executions" );
    if ( _all_kernels_execution == nullptr )
    {
        cube::Metric* _met = cube->defineMetric(
            "ALL kernels",
            "all_kernels_executions",
            "DOUBLE",
            "sec",
            "",
            TIME_METRIC_URL "#kernels_executions",
            "Time spent executing all kernels",
            nullptr,
            cube::CUBE_METRIC_POSTDERIVED,
            "metric::hip_kernel_executions() + metric::cuda_kernel_executions() + metric::opencl_kernel_executions() + metric::openacc_kernel_executions()+ metric::omp_target_kernel_executions()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_ipc( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "ipc" );
    if ( _met == nullptr )
    {
        if ( cube->getMetric( "tot_ins_without_wait" ) != nullptr  && cube->getMetric( "tot_cyc_without_wait" ) != nullptr )
        {
            _met = cube->defineMetric(
                "IPC",
                "ipc",
                "DOUBLE",
                "",
                "",
                POP_IPC_METRIC_URL,
                "Value of IPC (instructions per cycle) without busy-wait in MPI and OpenMP, computed as tot_ins_without_wait() / tot_cyc_without_wait()",
                nullptr,
                cube::CUBE_METRIC_POSTDERIVED,
                "metric::tot_ins_without_wait() / metric::tot_cyc_without_wait()",
                "",
                "",
                "",
                "",
                true,
                cube::CUBE_METRIC_GHOST
                );
            if ( _met != nullptr )
            {
                _met->setConvertible( false );
            }
            _met->def_attr( "origin", "advisor" );
        }
    }
}



void
PerformanceTest::add_tot_ins_without_wait( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "tot_ins_without_wait" );
    if ( _met == nullptr )
    {
        std::vector<std::string> name_alternatives = { "PAPI_TOT_INS", "instructions" };
        NamedMetric              named_metric      = get_metric_alternative( cube, name_alternatives );
        if ( named_metric.isNull() )
        {
            return;
        }
        const std::string display_name = named_metric.name() + " without busy-wait";
        const std::string description  = "Here is " + named_metric.name() + " without busy-wait in MPI and OpenMP.";

        _met = cube->defineMetric(
            display_name,
            "tot_ins_without_wait",
            "DOUBLE",
            "",
            "",
            POP_IPC_METRIC_URL,
            description,
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${without_wait_state}[${calculation::callpath::id}] * metric::" + named_metric.name() + "()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}



void
PerformanceTest::add_tot_cyc_without_wait( cube::CubeProxy* cube )  const
{
    cube::Metric* _met = cube->getMetric( "tot_cyc_without_wait" );
    if ( _met == nullptr )
    {
        std::vector<std::string> name_alternatives = { "PAPI_TOT_CYC", "cycles" };
        NamedMetric              named_metric      = get_metric_alternative( cube, name_alternatives );
        if ( named_metric.isNull() )
        {
            return;
        }
        const std::string display_name = named_metric.name() + " without busy-wait";
        const std::string description  = "Here is " + named_metric.name() + " without busy-wait in MPI and OpenMP.";

        _met = cube->defineMetric(
            display_name,
            "tot_cyc_without_wait",
            "DOUBLE",
            "",
            "",
            POP_IPC_METRIC_URL,
            description,
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${without_wait_state}[${calculation::callpath::id}] * metric::" + named_metric.name() + "()",
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}


void
PerformanceTest::add_stalled_resources( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "stalled_resources" );
    if ( _met == nullptr )
    {
        if ( cube->getMetric( "res_stl_without_wait" ) != nullptr && cube->getMetric( "tot_cyc_without_wait" ) != nullptr )
        {
            _met = cube->defineMetric(
                "Stalled resources",
                "stalled_resources",
                "DOUBLE",
                "",
                "",
                POP_STALLED_METRIC_URL,
                "Amount of stalled resources, computed as res_stl_without_wait() / tot_cyc_without_wait()",
                nullptr,
                cube::CUBE_METRIC_POSTDERIVED,
                " metric::res_stl_without_wait() / metric::tot_cyc_without_wait()",
                "",
                "",
                "",
                "",
                true,
                cube::CUBE_METRIC_GHOST
                );
            if ( _met != nullptr )
            {
                _met->setConvertible( false );
            }
            _met->def_attr( "origin", "advisor" );
        }
    }
}



void
PerformanceTest::add_res_stl_without_wait( cube::CubeProxy* cube ) const
{
    cube::Metric* _met = cube->getMetric( "res_stl_without_wait" );
    if ( _met == nullptr )
    {
        std::string metric_calc;
        std::string metric_desc;
        if ( cube->getMetric( "PAPI_RES_STL" ) != nullptr )
        {
            metric_calc = "metric::PAPI_RES_STL()";
            metric_desc = "PAPI_RES_STL";
        }
        else if ( cube->getMetric( "stalled_cycles_backend" ) != nullptr )
        {
            if ( cube->getMetric( "stalled_cycles_frontend" ) != nullptr )
            {
                metric_calc = "(metric::stalled_cycles_backend() + metric::stalled_cycles_frontend())";
                metric_desc = "(stalled_cycles_backend + stalled_cycles_frontend";
            }
            else
            {
                metric_calc = "metric::stalled_cycles_backend()";
                metric_desc = "stalled_cycles_backend";
            }
        }
        else if ( cube->getMetric( "stalled_cycles_frontend" ) != nullptr )
        {
            metric_calc = "metric::stalled_cycles_frontend()";
            metric_desc = "stalled_cycles_frontend";
        }
        else
        {
            return;
        }

        std::string display_name = metric_desc + " without busy wait";
        std::string description  = "Here is " + metric_desc + " without busy-wait in MPI and OpenMP.";

        _met = cube->defineMetric(
            display_name,
            "res_stl_without_wait",
            "DOUBLE",
            "",
            "",
            POP_IPC_METRIC_URL,
            description,
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE,
            "${without_wait_state}[${calculation::callpath::id}] * " + metric_calc,
            "",
            "",
            "",
            "",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}

PerformanceTest::NamedMetric
PerformanceTest::get_metric_alternative( cube::CubeProxy*,
                                         std::vector<std::string> const& name_alternatives ) const
{
    for ( const std::string& name : name_alternatives )
    {
        cube::Metric* metric = cube->getMetric( name );
        if ( metric != nullptr )
        {
            return NamedMetric( metric, name );
        }
    }
    return NamedMetric();
}




void
PerformanceTest::findRoot()
{
    const std::vector<cube::Cnode*>& cnodes = cube->getRootCnodes();
    if ( cnodes.size() == 1 )
    {
        root_cnode = cnodes[ 0 ];
        return;
    }
    for ( std::vector<cube::Cnode*>::const_iterator iter = cnodes.begin(); iter != cnodes.end(); ++iter )
    {
        if ( ( ( *iter )->get_callee()->get_name() == "main" ) ||
             ( ( *iter )->get_callee()->get_name() == "MAIN" ) )
        {
            root_cnode = *iter;
            return;
        }
    }
    root_cnode = nullptr;
}


size_t
PerformanceTest::get_number_of_cpu_locations() const
{
    const std::vector<cube::Location*>& _locs        = cube->getLocations();
    size_t                              cpu_locs_num = 0.;
    std::for_each(  _locs.begin(),
                    _locs.end(),
                    [ &cpu_locs_num ]( const cube::Location* _l )
    {
        cpu_locs_num += ( _l->get_type() == cube::CUBE_LOCATION_TYPE_CPU_THREAD ) ? 1 : 0;
    } );
    return cpu_locs_num;
}
size_t
PerformanceTest::get_number_of_gpu_location_groups() const
{
    const std::vector<cube::LocationGroup*>& _lgs        = cube->getLocationGroups();
    size_t                                   gpu_lgs_num = 0.;
    std::for_each(  _lgs.begin(),
                    _lgs.end(),
                    [ &gpu_lgs_num ]( const cube::LocationGroup* _l )
    {
        gpu_lgs_num += ( _l->get_type() == cube::CUBE_LOCATION_GROUP_TYPE_ACCELERATOR ) ? 1 : 0;
    } );
    return gpu_lgs_num;
}

size_t
PerformanceTest::get_number_of_gpu_locations() const
{
    const std::vector<cube::Location*>& _locs        = cube->getLocations();
    size_t                              gpu_locs_num = 0.;
    std::for_each(  _locs.begin(),
                    _locs.end(),
                    [ &gpu_locs_num ]( const cube::Location* _l )
    {
        gpu_locs_num += ( _l->get_type() == cube::CUBE_LOCATION_TYPE_ACCELERATOR_STREAM ) ? 1 : 0;
    } );
    return gpu_locs_num;
}

size_t
PerformanceTest::get_number_of_metric_locations() const
{
    const std::vector<cube::Location*>& _locs           = cube->getLocations();
    size_t                              metric_locs_num = 0.;
    std::for_each(  _locs.begin(),
                    _locs.end(),
                    [ &metric_locs_num ]( const cube::Location* _l )
    {
        metric_locs_num += ( _l->get_type() == cube::CUBE_LOCATION_TYPE_METRIC ) ? 1 : 0;
    } );
    return metric_locs_num;
}

size_t
PerformanceTest::get_number_of_all_locations() const
{
    return cube->getLocations().size();
}
