/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>

#include <cstdint>
#include <dirent.h> // <filesystem> library requires C++17
#include <dlfcn.h>
#include <iostream>
#include <set>

#include "CubeServerPlugin.h"
#include "CubePluginManager.h"

using namespace std;
using namespace cube;

PluginManager* PluginManager::single = nullptr;

PluginManager::PluginManager()
{
}

PluginManager*
PluginManager::getInstance()
{
    return single ? single : single = new PluginManager();
}

PluginManager::~PluginManager()
{
    for ( size_t i = 0; i < plugins.size(); i++ )
    {
        destroy_t* destroyPlugin = ( destroy_t* )dlsym( pluginHandles[ i ], "destroy" );
        // delete the plugin using the plugins destroy function (calling delete directly could cause a non-matching new and delete call)
        destroyPlugin( plugins[ i ] );

        // unload the library
        dlclose( pluginHandles[ i ] );
    }
}

void
PluginManager::loadPlugins( const vector<string>& pathes, uint64_t verbosity )
{
    std::set<std::string> pluginNames;

    // search plugins
    struct dirent* diread;
    DIR*           dir;
    for ( const string& searchPath : pathes )
    {
        if ( ( dir = opendir( searchPath.c_str() ) ) != nullptr )
        {
            while ( ( diread = readdir( dir ) ) != nullptr )
            {
                char* filename = diread->d_name;
                if ( ( string( filename ).find( ".so" ) != std::string::npos ) ||
                     ( string( filename ).find( ".dylib" ) != std::string::npos ) )
                {
                    string file = searchPath + "/" + string( filename ); // path to a potential plugin (dynamic library)

                    // check, if the dynamic library can be loaded
                    void* pluginHandle = nullptr;
#ifndef __MINGW32__
                    int flag = RTLD_NOLOAD | RTLD_LAZY;
#else  // mingw has only <glibc2.2 implemewntation of dlopen
                    int flag = RTLD_LAZY;
#endif
                    pluginHandle = dlopen( file.c_str(), flag ); // only check, don't load yet
                    if ( pluginHandle )                          // same plugin has already been loaded
                    {
                        //cerr << "Plugin has already been loaded: " << file << endl;
                        continue;
                    }
                    pluginHandle = dlopen( file.c_str(), RTLD_LAZY );
                    if ( pluginHandle )                          // check if shared library is a valid plugin of the type CubeServerPlugin
                    {
                        pluginHandles.push_back( pluginHandle ); // save to delete on exit

                        // reset errors
                        dlerror();

                        // load create/destroy symbols
                        create_t*   createPlugin = ( create_t* )dlsym( pluginHandle, "create" );
                        const char* dlsym_error  = dlerror();
                        if ( dlsym_error )
                        {
                            //cerr << "PluginManager:: cannot load symbol create: " << dlsym_error << endl;
                            continue;
                        }
                        destroy_t* destroyPlugin = ( destroy_t* )dlsym( pluginHandle, "destroy" );
                        dlsym_error = dlerror();
                        if ( dlsym_error )
                        {
                            //cerr << "PluginManager:: cannot load symbol destroy: " << dlsym_error << endl;
                            continue;
                        }

                        // create an instance of the plugin class
                        CubeServerPlugin* plugin = createPlugin();

                        // check if a plugin with the same name has already been loaded -> ignore further ones
                        if ( pluginNames.find( plugin->name() ) == pluginNames.end() )
                        {
                            pluginNames.insert( plugin->name() );
                        }
                        else
                        {
                            cerr << " PluginManager:: duplicate plugin ignored: " << file << endl;
                            destroyPlugin( plugin );
                            dlclose( pluginHandle );
                            continue;
                        }
                        plugins.push_back( plugin );
                        if ( verbosity > 0 )
                        {
                            cout << " PluginManager:: loaded: " << plugin->name() << " " << file << endl;
                        }
                    }
                    else
                    {
                        // cerr << "Cannot load plugin: " << " " << dlerror() << file << endl;
                    }
                }
            }
            closedir( dir );
        }
    }
    if ( plugins.empty() )
    {
        cout << "no server side plugins found" << endl;
    }
}

CubeServerPlugin*
PluginManager::getPlugin( const string& name )
{
    for ( auto* plugin : plugins )
    {
        if ( plugin->name() == name )
        {
            return plugin;
        }
    }
    return nullptr;
}
