/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023                                                     **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/*-------------------------------------------------------------------------*/
/**
 *  @file
 *  @ingroup CUBE_lib.network
 *  @brief   Definition of the class
 **/
/*-------------------------------------------------------------------------*/


#include <config.h>
#include <cassert>
#include <string>
#include "CubeOpenAlgorithmRequest.h"
#include "cube_network_types.h"
#include "CubeClientCallbackData.h"
#include "CubeIoProxy.h"
#include "CubeError.h"
#include "CubeProxy.h"
#include "CubeServerCallbackData.h"
#include "cube_network_types.h"
#include "CubeClientConnection.h"
#include "CubeServerConnection.h"


using namespace cube;
using namespace std;


/// --- Construction & destruction ---------------------------------------

NetworkRequestPtr
OpenAlgorithmRequest::create( Algorithm algo, const std::vector<string>& fileNames, const std::vector<std::string>& options )
{
    return std::shared_ptr<OpenAlgorithmRequest>( new OpenAlgorithmRequest(
                                                      algo, fileNames, options ) );
}


NetworkRequestPtr
OpenAlgorithmRequest::create()
{
    return std::shared_ptr<OpenAlgorithmRequest>( new OpenAlgorithmRequest() );
}


OpenAlgorithmRequest::OpenAlgorithmRequest()
{
}


OpenAlgorithmRequest::OpenAlgorithmRequest( Algorithm algo, const std::vector<string>& fileNames, const std::vector<string>& options )
    : mAlgo( algo ), mFileNames( fileNames ), mOptions( options )
{
}


OpenAlgorithmRequest::~OpenAlgorithmRequest()
{
}


/// --- Request identification -------------------------------------------

string
OpenAlgorithmRequest::getName() const
{
    return "OpenAlgorithm";
}


NetworkRequest::id_t
OpenAlgorithmRequest::getId() const
{
    return OPEN_ALGORITHM_REQUEST;
}

/// --- Connection interface ---------------------------------------------

// send ( Algorithm algo, const std::vector<string>& fileNames, const std::vector<std::string>& options )
void
OpenAlgorithmRequest::sendRequestPayload( ClientConnection& connection,
                                          ClientCallbackData* ) const
{
    connection << static_cast<uint8_t>( mAlgo );

    connection << static_cast<uint32_t> ( mFileNames.size() );
    for ( string name: mFileNames )
    {
        connection << name;
    }

    connection << static_cast<uint32_t> ( mOptions.size() );
    for ( string name: mOptions )
    {
        connection << name;
    }
}


void
OpenAlgorithmRequest::receiveRequestPayload( ServerConnection& connection,
                                             ServerCallbackData* )
{
    string   str;
    uint32_t size;

    uint8_t algo;
    connection >> algo;
    mAlgo = static_cast<Algorithm> ( algo );

    // read mFileNames
    connection >> size;
    for ( uint32_t i = 0; i < size; i++ )
    {
        connection >> str;
        mFileNames.push_back( str );
    }

    // read mOptions
    connection >> size;
    for ( uint32_t i = 0; i < size; i++ )
    {
        connection >> str;
        mOptions.push_back( str );
    }
}


void
OpenAlgorithmRequest::processRequest( ServerCallbackData* data )
{
    assert( data );
    data->setCube( new CubeIoProxy() );
    assert( data->getCube() );

    try
    {
        data->setCube( new CubeIoProxy( CubeProxy::generateCube( mAlgo, mFileNames, mOptions ) ) );
    }
    catch ( const std::exception& e )
    {
        throw RecoverableNetworkError( e.what() );
    }
}
