/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/**
 * \file CubeLastNRowsStrategy.cpp
 * \brief In this strategy, the last N accessed rows are kept in memory.
 */

#include "config.h"
#include <iostream>
#include <cstdlib>
#include <cstring>
#include "CubeLastNRowsStrategy.h"

#define CUBE_LAST_N_ROWS_NUMBER "CUBE_NUMBER_ROWS"

using namespace cube;

LastNRowsStrategy::LastNRowsStrategy( bool permissionToFreeAll,
                                      int  lastN )
    : m_permissionToFreeAll( permissionToFreeAll )
{
    char* _n = getenv( CUBE_LAST_N_ROWS_NUMBER );
    if ( _n == nullptr )
    {
        m_lastN = lastN;
    }
    else
    {
        try
        {
            m_lastN = std::abs( std::stoi( _n ) );
        }
        catch ( std::invalid_argument& e )
        {
            std::cerr << " Value for CUBE_NUMBER_ROW is invalid. Using default value 1000. Error: " << e.what() << std::endl;
            m_lastN = 1000;
        }
        catch ( const std::out_of_range& e )
        {
            std::cerr << "Warning: Value out of range for environment variable CUBE_NUMBER_ROW. Using default value 1000. Error: " << e.what() << std::endl;
            m_lastN = 1000;
        }
    }
    m_rowsInMemory.clear();
}



std::vector<cnode_id_t>
LastNRowsStrategy::initialize( rows_t* _rows )
{
    m_rowsInMemory.clear();

    // Determine the actual number of rows to initialize
    // This handles cases where _rows->size() is less than m_lastN
    size_t num_rows_to_initialize = std::min( ( size_t )m_lastN, _rows->size() );

    std::vector<cnode_id_t> to_return;
    // Pre-allocate memory for to_return to avoid reallocations
    to_return.reserve( num_rows_to_initialize );

    // Initialize m_rowsInMemory and to_return
    for ( cnode_id_t _cid = 0; _cid < ( cnode_id_t )num_rows_to_initialize; ++_cid )
    {
        m_rowsInMemory.push_back( _cid );
        to_return.push_back( _cid );
    }

    return to_return;
}



void
LastNRowsStrategy::addRow( const cube::cnode_id_t& rowId, bool&, std::vector<cnode_id_t>& rowsToRemove )
{
    rowsToRemove.clear(); // Clear the output vector for this call

    // Add the new rowId to the "most recently used" end of the deque
    m_rowsInMemory.push_back( rowId );

    // Enforce the "Last N" policy: if capacity is exceeded, remove the oldest row
    while ( m_rowsInMemory.size() > m_lastN )
    {
        // Add the ID of the oldest row to the list of rows to be removed by the caller
        rowsToRemove.push_back( m_rowsInMemory.front() );
        // Remove the oldest row from our internal memory
        m_rowsInMemory.pop_front();
    }
}




void
LastNRowsStrategy::removeRows( std::vector<cnode_id_t>&, std::vector<cnode_id_t>& rowsToRemove )
{
    // As per the original design, this strategy ignores external specific removal requests.
    // It only removes rows based on its internal "last N" policy.
    rowsToRemove.clear();   // Indicate that no rows were removed as a direct result of this call.
}


bool
LastNRowsStrategy::permissionToFreeAll()
{
    if ( m_permissionToFreeAll )
    {
        m_rowsInMemory.clear(); // Clear internal cache if permission is true
    }

    return m_permissionToFreeAll;
}




void
LastNRowsStrategy::forcedFreeAll()
{
    m_rowsInMemory.clear();
}



void
LastNRowsStrategy::needRows( std::vector<cnode_id_t>& rowsToAdd, std::vector<cnode_id_t>& rowsToRemoveFirst )
{
    // Clear the output vector for rows to be removed.
    rowsToRemoveFirst.clear();

    // 1. Add all newly requested rows to the end of our in-memory deque.
    // These are considered the "most recently accessed."
    for ( const cnode_id_t& rowId : rowsToAdd )
    {
        m_rowsInMemory.push_back( rowId );
    }

    // 2. Enforce the "Last N Rows" policy.
    // If the number of rows in memory exceeds m_lastN, remove the oldest ones
    // from the front of the deque until the size is m_lastN.
    while ( m_rowsInMemory.size() > m_lastN )
    {
        // This check is a safeguard, but typically `size() > m_lastN` implies `!empty()`
        // if `m_lastN` is non-negative.
        if ( !m_rowsInMemory.empty() )
        {
            rowsToRemoveFirst.push_back( m_rowsInMemory.front() ); // Identify oldest for removal
            m_rowsInMemory.pop_front();                            // Remove oldest from memory
        }
        else
        {
            // Should not be reached in normal operation if m_lastN >= 0.
            // If it is, it indicates a serious logical inconsistency.
            break;
        }
    }
}
