/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include <QtPlugin>
#include <QDebug>
#include <QPushButton>
#include <QVBoxLayout>
#include <QApplication>
#include <QStyle>
#include <QFileDialog>

#include <cstdlib>
#include <fstream>
#include <iostream>
#include <map>
#include <vector>
#include <string>

#include "Tau2Cube.h"
#include "ContextFreeServices.h"
#include "CubeProxy.h"
#include "Globals.h"
#include "RemoteFileDialog.h"

using namespace cube;
using namespace cubepluginapi;

void
Tau2Cube::version( int& major, int& minor, int& bugfix ) const
{
    major  = 1;
    minor  = 0;
    bugfix = 0;
}

QString
Tau2Cube::name() const
{
    return "Tau to Cube";
}

void
Tau2Cube::opened( ContextFreeServices* service )
{
    this->service = service;

    widget = this->service->getWidget();

    QHBoxLayout* layoutOuter = new QHBoxLayout();
    widget->setLayout( layoutOuter );



    QWidget*     inner  = new QWidget();
    QVBoxLayout* layout = new QVBoxLayout( inner );
    layoutOuter->addWidget( inner );
    inner->setSizePolicy( QSizePolicy::Maximum, QSizePolicy::Maximum );

    experiments = new QListWidget();

    remote = new QCheckBox( tr( "Open remote files using cube_server" ) );
    connect( remote, &QCheckBox::clicked, this, &Tau2Cube::clearList );

    QPushButton* addCube   = new QPushButton( tr( "Add TAU experiments" ) );
    QPushButton* clearList = new QPushButton( tr( "Clear list" ) );

    merge = new QPushButton( tr( "Show result" ) );
    connect( addCube, &QPushButton::clicked, this, &Tau2Cube::addTauDirectory );
    connect( clearList, &QPushButton::clicked, this, &Tau2Cube::clearList );
    connect( merge, &QPushButton::clicked, this, &Tau2Cube::startAction );

    layout->addWidget( experiments );
    layout->addWidget( remote );
    layout->addWidget( addCube );
    layout->addWidget( clearList );
    layout->addWidget( merge );

    layout->addItem( new QSpacerItem( 0, 10 ) );

    merge->setEnabled( false );
    merge->setStyleSheet( "padding:4px;text-align: center" );

    if ( !args.isEmpty() )
    {
        addExperiments( args );
        startAction();
    }
}

void
Tau2Cube::addTauDirectory()
{
    QString tauDirectory;

    if ( remote->isChecked() )
    {
        tauDirectory = cubegui::RemoteFileDialog::getDirectoryUrl( "", tr( "Select TAU Directory" ) );
    }
    else
    {
        tauDirectory = QFileDialog::getExistingDirectory( widget, tr( "Select TAU Directory" ),
                                                          tr( "" ),
                                                          QFileDialog::ShowDirsOnly
                                                          | QFileDialog::DontResolveSymlinks );
    }
    addExperiments( QStringList( tauDirectory ) );
}


void
Tau2Cube::addExperiments( const QStringList& list )
{
    experiments->addItems( list );
    merge->setEnabled( experiments->count() >= 1 );
}

void
Tau2Cube::clearList()
{
    url.clear();
    experiments->clear();
    merge->setEnabled( experiments->count() >= 1 );
}


void
Tau2Cube::startAction()
{
    std::vector<std::string> fileNames;
    for ( int i = 0; i < experiments->count(); i++ )
    {
        fileNames.push_back( ( url + experiments->item( i )->text() ).toStdString() );
    }
    try
    {
        CubeProxy* cube = CubeProxy::create( ALGORITHM_TAU, fileNames );
        service->openCube( cube );
    }
    catch ( const cube::Error& e )
    {
        QString message = QString( e.what() );
        service->setMessage( message, cubepluginapi::Error );
        return;
    }
}

void
Tau2Cube::closed()
{
}

QString
Tau2Cube::getHelpText() const
{
    return tr( "This plugin allows to open (multiple) TAU profile Directory in Cube GUI." );
}
