/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2022                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#ifndef CUBEGUIPLUGIN_DIMENSIONSELECTIONWIDGET_H
#define CUBEGUIPLUGIN_DIMENSIONSELECTIONWIDGET_H

#include <vector>
#include <QLabel>
#include <QPushButton>
#include <QSlider>

class SliderPopup;
class ValuePopupSlider;
class AxisOrderWidget;

/**
   Widget to select the dimensions and to assign an axis to each selected dimension
 */
class DimensionSelectionWidget : public QFrame
{
    Q_OBJECT
public:
    explicit
    DimensionSelectionWidget( const std::vector<long>& ndim,
                              const QStringList&       dimnames );
    std::vector<long>
    getSelectionVector();
    void
    setSelectionVector( std::vector<long> );

signals:
    void
    selectedDimensionsChanged();

private slots:
    void
    orderChanged();
    void
    selectionChanged();

private:
    std::vector<long>
    getCurrentSelection();

    void
    drawElement( QPainter& painter,
                 int       x,
                 int       y,
                 int       elemnr );

    std::vector<ValuePopupSlider*> slider;  // used for selection view
    std::vector<long>              dims;
    QStringList                    dimnames;
    AxisOrderWidget*               order;
};

/**
 * @brief The AxisOrderWidget class
 * allow the user to change the axis order of the displayed dimensions (ALL selected)
 */
class AxisOrderWidget : public QWidget
{
    Q_OBJECT
public:
    AxisOrderWidget( uint ndims );
    void
    paintEvent( QPaintEvent* );
    void
    setSelectionVector( const std::vector<long>& dims,
                        bool                     hasAxisInfo = false );

    const std::vector<long>&
    getSelectionOrderVector();

    inline int
    getCellheight() const
    {
        return QFontMetrics( font() ).lineSpacing() + 2 * insets;
    }
    virtual QSize
    minimumSizeHint() const override;

protected:
    void
    mousePressEvent( QMouseEvent* event ) override;
    void
    mouseMoveEvent( QMouseEvent* event ) override;
    void
    mouseReleaseEvent( QMouseEvent* event ) override;

signals:
    void
    orderChanged();

private:
    const static int insets = 2;       // margins around the label

    int               ndims;           // number of total dims
    int               shownDimensions; // number of dimensions which are shown
    std::vector<long> dims;            // vector which contains the index of the selected elements or -1 for complete dimension

    QPoint currentPos;                 // current position of mouse while dragging
    int    cellwidth;                  // width of each element
    int    cellheight;                 // height of each dimension cell
    int    labelwidth;                 // width of the legend
    int    move;                       // index of element to move
    int    moveY;

    void
    drawElement( QPainter& painter,
                 int       x,
                 int       y,
                 int       elemnr );
};


/**
   creates a widget which displays an integer value or the text "all" if value < 0
   if the widget is clicked, a slider widget appears to change the value
 */
class ValuePopupSlider : public QWidget
{
    Q_OBJECT
public:
    /**
       @param value the displayed value
       @param maximum the maximum value of the widget/slider
     */
    ValuePopupSlider( int value,
                      int max );
    int
    value() const;

public slots:
    void
    setValue( int );

private slots:
    void
    showSlider();

signals:
    void
    valueChanged( int );

private:
    int          value_;
    int          max_;
    QPushButton* push;
    SliderPopup* win;
};

/**
   part of ValuePopupSlider: a pop widget which contains a slider
 */
class SliderPopup : public QWidget
{
    Q_OBJECT
public:
    explicit
    SliderPopup( int min,
                 int max,
                 int value );
private:
    QSlider* slider_;
    friend class ValuePopupSlider;
};

#endif // DIMENSIONSELECTIONWIDGET_H
