/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBEGUIPLUGIN_SOURCECODEEDITORPLUGIN_H
#define CUBEGUIPLUGIN_SOURCECODEEDITORPLUGIN_H
#include "Compatibility.h"
#include <QtGui>
#include <QPlainTextEdit>
#include <QPushButton>
#include <QLineEdit>
#include <QStackedWidget>
#include <QScrollArea>
#include <QDialog>
#include <QLabel>
#include "TabInterface.h"
#include "CubePlugin.h"
#include "PluginServices.h"
#include "SourceCodeEditor.h"

namespace editor_plugin
{
enum class SourceType { CallSite, FunctionDefinition };

class SourceInfo
{
public:
    SourceInfo();
    SourceInfo( const QString& orig,
                const QString& file,
                int            start,
                int            end = 0 );
    bool
    isEmpty() const;
    const QString&
    originalLocation() const;
    const QString&
    fileName() const;
    int
    startLine() const;
    int
    endLine() const;
    void
    setOriginalLocation( const QString& newOriginalLocation );
    void
    invalidate();

private:
    QString originalLocation_;   // file name of the source code which corresponds to the selected region
    QString fileName_;           // originalLocation with corrected path
    int     startLine_ = -1;     // begin of the region
    int     endLine_   = -1;     // end of the region
};

class EditorPlugin : public QObject, cubepluginapi::TabInterface, cubepluginapi::CubePlugin, cubepluginapi::SettingsHandler
{
    Q_OBJECT
    Q_INTERFACES( cubepluginapi::CubePlugin )
    Q_PLUGIN_METADATA( IID "SourceCodeEditorPlugin" )

public:
    EditorPlugin()
    {
        deamonStarted = false;
    }

    // CubePlugin interface
    QString
    name() const override;

    virtual bool
    cubeOpened( cubepluginapi::PluginServices* service ) override;
    virtual void
    cubeClosed() override;

    void
    version( int& major,
             int& minor,
             int& bugfix ) const override;

    virtual QString
    getHelpText() const override;

    virtual QString
    getHelpURL() const override
    {
        return "SourceCodeViewerPlugin.html";
    }

    // SettingsHandler
    virtual void
    loadExperimentSettings( QSettings& ) override;

    virtual void
    saveExperimentSettings( QSettings& ) override;

    virtual void
    saveGlobalSettings( QSettings& settings ) override;
    virtual void
    loadGlobalSettings( QSettings& settings ) override;

    virtual QString
    settingName() override
    {
        return "SourceCodeViewer";
    }

    // TabInterface implementation
    virtual QWidget*
    widget() override;
    virtual QString
    label() const override;
    virtual void
    setActive( bool active ) override;
    virtual void
    detachEvent( QMainWindow*,
                 bool isDetached ) override;

    QIcon
    icon() const override;

    // end TabInterface implementation

    /** whether to show the call site or the function definition */
    void
    setSourceType( SourceType newSourceType );

private slots:
    void
    onChangeFont();
    void
    onSaveFile();
    void
    onSaveFileAs();
    void
    onToggleReadOnly( bool checked );
    void
    showSourceCode();
    void
    search( const QString& );
    void
    searchForward();
    void
    searchBackward();
    void
    closeSourceView();
    void
    treeItemSelected( cubepluginapi::TreeItem* );
    void
    openExternalEditor();
    void
    openDefinedExternalEditor();
    void
    openFileDialog();
    void
    deleteProcess();
    bool
    onChooseEditor();
    void
    startSearch();
    void
    contextMenuIsShown( cubepluginapi::DisplayType type,
                        cubepluginapi::TreeItem*   item );
    void
    toFront();
    void
    resetUserPath();

private:
    QStackedWidget* mainWidget;
    QWidget*        editorWidget;   // widget to display source
    QScrollArea*    openWidget;     // shown if source cannot be loaded
    // this is the editor to display and edit source code files
    SourceCodeEditor* textEdit;
    QWidget*          searchWidget; // widget below textEdit that is shown if search context menu item is selected
    // Selected Font
    QFont    fontSourceCode;
    QAction* copyPath;
    QAction* font;
    QAction* switchSource;
    // these are two buttons that get enabled and disabled when toggling the readOnly status
    QAction* textEditSaveAction, * textEditSaveAsAction;
    QAction* readOnlyAction;
    QAction* findAction;
    QAction* externalAction;      // open in external editor
    QAction* externalUserAction;  // open in user specific external editor
    QAction* resetUserPathAction; // removes previously stored user path (if errorneous)
    // "Find" - input field
    QLineEdit* findEdit;
    // these are two buttons for search forwards or backwards
    QPushButton* findNextButton, * findPrevButton;
    QPushButton* openFile;
    QLabel*      openFileLabel;

    cubepluginapi::PluginServices* service;
    cubepluginapi::TreeItem*       selectedItem; // currently selected item

    SourceType sourceType = SourceType::CallSite;

    SourceInfo         source;                  // function or call site info
    SourceInfo         functionSource;          // source information about the currently selected function
    SourceInfo         callSiteSource;          // call site source information about the currently selected item
    QString            origPath;                // original location of the sources
    QString            userPath;                // the new location of the sources
    QString            searchText;              // holds "search" text
    QList<QStringList> sourcePathes;            // default pathes to search for sources
    QTemporaryDir*     tempDir = nullptr;       // path to save source files from server

    QHash<QString, QStringList> externalEditors;
    QString                     externalEditor;
    bool                        deamonStarted;  // status of process of external editor

    //-------------------------------------------------------------------
    void
    showSourceCode( const QString& sourceFile,
                    int            startLine,
                    int            endLine );
    QString
    getSourceFile( const QString& sourceFile );
    void
    updateActions();
    void
    createWidgets();
    void
    setSourceInfo();
    void
    addPathReplacement( const QString& orig,
                        const QString& user );
    void
    removePathReplacement( const QString& orig );
};
} // namespace
#endif // SourceCodeEditorPlugin_H
