/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>

#include "JSCHybridTransferTest.h"


using namespace advisor;

JSCTransferTest::JSCTransferTest( cube::CubeProxy* cube ) : PerformanceTest( cube )
{
    setName( " -> Transfer Efficiency" );
    setWeight( 1 );   // need to be adjusted
    max_total_time       = cube->getMetric( "max_total_time_hyb" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal_hyb" );
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        adjustForTest( cube );
    }
    max_total_time       = cube->getMetric( "max_total_time_hyb" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal_hyb" );

    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }
    cube::metric_pair metric;
    metric.first  = max_total_time;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );


    metric.first  = max_total_time_ideal;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_total_time_ideal.push_back( metric );
}



void
JSCTransferTest::applyCnode( const cube::Cnode*, const cube::CalculationFlavour,
                             const bool )
{
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        return;
    }
}

void
JSCTransferTest::applyCnode( const cube::list_of_cnodes& cnodes,
                             const bool                    )
{
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        return;
    }
    setValue( analyze( cnodes ) );
}

double
JSCTransferTest::analyze( const cube::list_of_cnodes& cnodes,
                          cube::LocationGroup*          ) const
{
    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );

    cube::value_container inclusive_values2;
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( lmax_total_time_ideal,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );

    const std::vector<cube::LocationGroup*>& _lgs                         = cube->getLocationGroups();
    double                                   max_total_time_ideal_value_d = 0.;
    double                                   max_total_time_value_d       = 0.;
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        max_total_time_value_d       +=  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
        max_total_time_ideal_value_d +=  inclusive_values2[ ( *iter )->get_sys_id() ]->getDouble();
    }
    return max_total_time_ideal_value_d / max_total_time_value_d;
}


const std::string&
JSCTransferTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
JSCTransferTest::isActive() const
{
    return ( max_total_time_ideal != nullptr ) && ( max_total_time != nullptr );
};

bool
JSCTransferTest::isIssue() const
{
    return false;
};

void
JSCTransferTest::adjustForTest( cube::CubeProxy* cube ) const
{
    cube::Metric* _mpi_time = cube->getMetric( "mpi" );
    if ( _mpi_time->isInactive() )
    {
        return;
    }
    if ( scout_metrics_available( cube ) )
    {
        add_max_total_time_ideal( cube );
        add_max_total_time( cube );
    }
}


void
JSCTransferTest::add_max_total_time_ideal( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    cube::Metric* _met = cube->getMetric( "max_total_time_ideal_hyb" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            tr( "Maximal total time in ideal network, hybrid" ).toUtf8().data(),
            "max_total_time_ideal_hyb",
            "DOUBLE",
            tr( "sec" ).toUtf8().data(),
            "",
            POP_TRANSFER_EFF_METRIC_URL,
            tr( "Maximal total time in ideal network, ( execution - transfer_time_mpi )" ).toUtf8().data(),
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_INCLUSIVE,
            "metric::execution() - metric::jsc_transfer_time_mpi()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
        advisor_services->addMetric( _met, nullptr );
    }
}


void
JSCTransferTest::add_max_total_time( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    cube::Metric* _met = cube->getMetric( "max_total_time_hyb" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            tr( "Maximal total time, hybrid" ).toUtf8().data(),
            "max_total_time_hyb",
            "DOUBLE",
            tr( "sec" ).toUtf8().data(),
            "",
            POP_TRANSFER_EFF_METRIC_URL,
            tr( "Maximal total time, max( execution )" ).toUtf8().data(),
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_INCLUSIVE,
            "metric::execution()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
        advisor_services->addMetric( _met, nullptr );
    }
}
