/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <QApplication>
#include <QComboBox>
#include <QDebug>
#include <QFileDialog>
#include <QFrame>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QRect>
#include <QScrollArea>
#include <QStyle>
#include <QVBoxLayout>
#include <QWidget>
#include <QtPlugin>
#include "CubeAdvisorProgress.h"
#include "CubeAdvisorPlugin.h"
#include "CubeAdvice.h"
#include "PluginServices.h"
#include "CubeProxy.h"
#include "CubeTypes.h"
#include "CallTree.h"
#include "DefaultCallTree.h"
#include "FlatTree.h"
#include "CubeIdIndexMap.h"
#include "algebra4.h"
#include "Globals.h"

using namespace cube;
using namespace cubepluginapi;
using namespace advisor;
cubepluginapi::PluginServices* advisor_services;

CubeAdvisor::CubeAdvisor()
{
    _widget                  = nullptr;
    jsc_audit_analysis       = nullptr;
    direct_calculation_state = false;
}

CubeAdvisor::~CubeAdvisor()
{
    delete _widget;
    delete jsc_audit_analysis;

    DELETE_BUSY_WAIT_PROGRESS
}



void
CubeAdvisor::version( int& major, int& minor, int& bugfix ) const
{
    major  = 2;
    minor  = 0;
    bugfix = 0;
}

QString
CubeAdvisor::name() const
{
    return "Cube JSC Analysis";
}

bool
CubeAdvisor::cubeOpened( PluginServices* service )
{
    INIT_BUSY_WAIT_PROGRESS

    this->service    = service;
    advisor_services = service;

    startAnalysis            = false;
    initializationIsFinished = false;

    cube      = service->getCube();
    execution = cube->getMetric( "execution" );
    if ( execution == nullptr )
    {
        execution = cube->getMetric( "time" ); // fallback in case of not remapped file. we just use time metric as an execution time
    }
    if ( execution == nullptr )
    {
        return false;
    }

    _widget = new QWidget();

    service->addTab( SYSTEM, this, OTHER_PLUGIN_TAB );

    // start initialization in a separate thread
    future = service->createFuture();
    future->addCalculation( new InitialisationTask( this ) );
    connect( future, SIGNAL( calculationFinished() ), this, SLOT( initializationFinished() ) );
    future->startCalculation( false );

    QVBoxLayout* top_layout = new QVBoxLayout();
    _widget->setLayout( top_layout );

    createTests();

    top_layout->addWidget( advisor_progress_widget );

    recalculate_widget = new QWidget();

    QHBoxLayout* _recalculate_layout = new QHBoxLayout();
    recalculate_widget->setLayout( _recalculate_layout );


    copy_values_button = new QPushButton( tr( "Copy values" ) );
    copy_values_button->setIcon( QIcon( ":/images/advisor-metrics-copy.png" ) );
    copy_values_button->setToolTip( tr( "Copy values of selected metrics into clipboard" ) );
    recalculate_button      = new QPushButton( tr( "Recalculate" ) );
    automatic_recalculation = new QCheckBox( tr( "automatic" ) );
    direct_calculation      = new QCheckBox( tr( "direct calculation" ) );
    direct_calculation->setCheckState( Qt::Unchecked );
    _recalculate_layout->addWidget( copy_values_button );
    _recalculate_layout->addWidget( recalculate_button );
    _recalculate_layout->addWidget( automatic_recalculation );
    _recalculate_layout->addWidget( direct_calculation );
    top_layout->addWidget( recalculate_widget );
    recalculate_widget->show();

    connect( copy_values_button, SIGNAL( pressed() ), this, SLOT( copyMetrics() ) );
    connect( recalculate_button, SIGNAL( pressed() ), this, SLOT( recalculate() ) );
    connect( automatic_recalculation, SIGNAL( stateChanged( int ) ), this, SLOT( automatic_recalculate( int ) ) );
    connect( direct_calculation, SIGNAL( stateChanged( int ) ), this, SLOT( direct_calculate( int ) ) );


    QScrollArea* top_scroll = new QScrollArea();
    top_scroll->setWidgetResizable( true );
    // pop audit

    if (  jsc_audit_analysis->isActive() )
    {
        // jsc hybrid audit
        jsc_hybrid_rating = new CubeRatingWidget( this, tr( "JSC Hybrid Assessment " ), jsc_audit_analysis );
        top_scroll->setWidget(  jsc_hybrid_rating );
    }

    connect( service, SIGNAL( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ),
             this, SLOT( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ) );



    top_layout->addWidget( top_scroll );
    setActive( false );

    return true;
}



void
CubeAdvisor::contextMenuIsShown( cubegui::DisplayType type, cubegui::TreeItem* item )
{
    if ( ( item == nullptr ) || ( type != CALL )  )
    {
        return;
    }
    QAction* action = service->addContextMenuItem( type, "Analysis for candidates" );
    connect( action, SIGNAL( triggered( bool ) ), this, SLOT( analyseSubtree() ) );
    context_menu_item = item;
}

void
CubeAdvisor::initialization_run()
{
    value_container       inclusive_values;
    value_container       exclusive_values;
    IdIndexMap            metric_id_indices;
    list_of_cnodes        lcnodes;
    std::vector< Cnode* > cnodes = cube->getRootCnodes();
    for ( std::vector< Cnode* >::iterator iter = cnodes.begin(); iter != cnodes.end(); ++iter )
    {
        cnode_pair cnode;
        cnode.first  = *iter;
        cnode.second = cube::CUBE_CALCULATE_INCLUSIVE;
        lcnodes.push_back( cnode );
    }

    list_of_sysresources           lsysres;
    std::vector< SystemTreeNode* > sysress = cube->getRootSystemTreeNodes();
    for ( std::vector< SystemTreeNode* >::iterator iter = sysress.begin(); iter != sysress.end(); ++iter )
    {
        sysres_pair sres;
        sres.first  = *iter;
        sres.second = cube::CUBE_CALCULATE_INCLUSIVE;
        lsysres.push_back( sres );
    }




    root_callpaths = service->getTopLevelItems( service->getActiveTree( CALL )->getType() );

    cube->getMetricSubtreeValues(   lcnodes,
                                    lsysres,
                                    *execution,
                                    ( size_t )0,
                                    metric_id_indices,
                                    &inclusive_values,
                                    &exclusive_values );

    total_execution = inclusive_values[ 0 ]->getDouble();

    for ( value_container::iterator iter = inclusive_values.begin(); iter != inclusive_values.end(); ++iter )
    {
        delete *iter;
    }
    for ( value_container::iterator iter = exclusive_values.begin(); iter != exclusive_values.end(); ++iter )
    {
        delete *iter;
    }
}


void
CubeAdvisor::copyMetrics()
{
    const QList<TreeItem*>& items =  service->getSelections( service->getActiveTree( CALL )->getType() );
    /* handle special case: item is a loop in the call tree, which may be aggregated */
    cube::list_of_cnodes lcnodes;
    foreach( TreeItem * item, items )
    {
        cube::Cnode* cnode = static_cast<cube::Cnode*> ( item->getCubeObject() );
        if ( cnode == nullptr )
        {
            continue;
        }
        cube::cnode_pair pcnode;
        pcnode.first  = cnode;
        pcnode.second = ( item->isExpanded() ) ? cube::CUBE_CALCULATE_EXCLUSIVE : cube::CUBE_CALCULATE_INCLUSIVE;
        lcnodes.push_back( pcnode );
    }
    jsc_hybrid_rating->copyMetricsValues( lcnodes );
}

void
CubeAdvisor::initializationFinished()
{
    service->setGlobalValue( name() + "::initFinished", true );
    initializationIsFinished = true;
}




void
CubeAdvisor::createTests()
{
    jsc_audit_analysis = new JSCAuditPerformanceAnalysis( cube );
}


void
CubeAdvisor::calculateOverallTests()
{
    START_BUSY_WAIT
    const QList<TreeItem*>& items =  service->getSelections( service->getActiveTree( CALL )->getType() );
    cube::list_of_cnodes    lcnodes;
    foreach( TreeItem * item, items )
    {
        cube::Cnode* cnode = static_cast<cube::Cnode*> ( item->getCubeObject() );
        if ( cnode == nullptr )
        {
            continue;
        }
        cube::cnode_pair pcnode;
        pcnode.first  = cnode;
        pcnode.second = ( item->isExpanded() ) ? cube::CUBE_CALCULATE_EXCLUSIVE : cube::CUBE_CALCULATE_INCLUSIVE;
        lcnodes.push_back( pcnode );
    }
    jsc_hybrid_rating->apply( lcnodes, direct_calculation_state );
    END_BUSY_WAIT
}





void
CubeAdvisor::cubeClosed()
{
    markerList.clear();
    delete jsc_audit_analysis;
}

QString
CubeAdvisor::getHelpText() const
{
    return tr( "This plugin evaluates performance issues and helps to browse them one-by-one." );
}


QWidget*
CubeAdvisor::widget()
{
    return _widget;
}

QString
CubeAdvisor::label() const
{
    return tr( "Advisor" );
}



QIcon
CubeAdvisor::icon() const
{
    return QIcon( ":/images/advisor-icon.png" );
}



/**
 * @brief DemoPlugin::setActive is called when the tab gets activated or deactivated by selecting another tab
 * The tab related demo elements should only react on signals, if the tab is active. For that reason the
 * signals are disconnected if another tab becomes active.
 */
void
CubeAdvisor::setActive( bool active )
{
    if ( active )
    {
        service->connect( service, SIGNAL( treeItemIsSelected( cubepluginapi::TreeItem* ) ),
                          this, SLOT( treeItemIsSelected( cubepluginapi::TreeItem* ) ) );
        service->connect( service, SIGNAL( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ),
                          this, SLOT( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ) );
        // ??? treeItemIsSelected( nullptr );
    }
    else
    {
        service->disconnect( service, SIGNAL( treeItemIsSelected( cubepluginapi::TreeItem* ) ),
                             this, SLOT( treeItemIsSelected( cubepluginapi::TreeItem* ) ) );
        service->disconnect( service, SIGNAL( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ),
                             this, SLOT( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ) );
    }
}

void
CubeAdvisor::treeItemIsSelected( cubepluginapi::TreeItem* item )
{
    if ( ( item == nullptr ) || ( item->getDisplayType() != CALL )  )
    {
        return;
    }
    if ( item->isExpanded() /*do not calculate for expanded items*/ )
    {
        return;
    }
    recalculate_widget->setEnabled( true );
    recalculate_widget->show();
    if ( automatic_recalculation->isChecked() )
    {
        recalculate();
    }
}


void
CubeAdvisor::recalculate()
{
    if ( jsc_hybrid_rating->isCalculating() )
    {
        advisor_services->setMessage( tr( "Calculation is in progress..." ), cubegui::Warning );
        return;
    }
    /** Returns selected items in the given active call tree */
    list_of_cnodes   lcnodes;
    QList<TreeItem*> selected_items = service->getSelections( service->getActiveTree( CALL )->getType() );
    if ( selected_items.size() == 0 ) // no selection?  do nothing
    {
        return;
    }

    list_of_cnodes _cnodes;
    if ( service->getActiveTree( CALL )->getTreeType() == DEFAULTCALLTREE || service->getActiveTree( CALL )->getTreeType() == TASKTREE ) // calltree is activbe with the selection
    {
        cubegui::DefaultCallTree* _tree = dynamic_cast<cubegui::DefaultCallTree*>( selected_items[ 0 ]->getTree() );                     // reference to 0 is ok as size got cheecked before
        _cnodes = _tree->getNodes( selected_items );
    }
    if ( service->getActiveTree( CALL )->getTreeType() == FLATTREE )                                   // flattree is active
    {
        cubegui::FlatTree* _tree = dynamic_cast<cubegui::FlatTree*>( selected_items[ 0 ]->getTree() ); // reference to 0 is ok as size got cheecked before
        _cnodes = _tree->getNodes( selected_items );
    }
    lcnodes.insert( lcnodes.end(), _cnodes.begin(), _cnodes.end() );
    jsc_hybrid_rating->apply( lcnodes, direct_calculation_state );
}



void
CubeAdvisor::automatic_recalculate( int state )
{
    if ( state ==  Qt::Checked  )
    {
        recalculate();
    }
    recalculate_button->setEnabled( state != Qt::Checked );
}

void
CubeAdvisor::direct_calculate( int state )
{
    direct_calculation_state = ( state ==  Qt::Checked );
    recalculate();
}
