/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2020                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#include "config.h"

#include <cmath>
#include "ImprovedRainbowColorMap.h"

using std::floor;
using std::max;
using std::min;

const ImprovedRainbowColorMap::Schemes ImprovedRainbowColorMap::DEFAULT_SCHEME =
    ImprovedRainbowColorMap::ISO_L;

const QString ImprovedRainbowColorMap::COLOR_MAP_DESCRIPTION =
    tr( "Set of colormaps designed as an improved version of classic jet (rainbow) color scheme. "
        "These maps has been implemented as an extension to MATLAB, called \"Perceptually "
        "improved colormaps\". <br>" );

const QString ImprovedRainbowColorMap::SCHEMES_DESCRIPTION[ ImprovedRainbowColorMap::SCHEMES_COUNT ] = {
    tr( "Rainbow colormap with constant value of lightness, used for representation of interval data with "
        "external lighting." ),
    tr( "Rainbow colormap based on Lightness-Chroma-Hue color space, with hue values going around "
        "the circle. Used for representation of azimuth and phase data." ),
    tr( "Rainbow colormap based on Lightness-Chroma-Hue color space, with hue values going twice around "
        "the circle. Used for representation of azimuth and phase data." ),
    tr( "Rainbow colormap with linear difference in lightness. Used for representation of interval data"
        " without external lighting." ),
    tr( "Rainbow colormap with linear difference in lightness. Used for representation of interval data"
        " without external lighting." ),
    tr( "Rainbow colormap with cubic-law lightness. Used for representation of interval data"
        " without external lighting." ),
    tr( "Rainbow colormap with cubic-law lightness. Used for representation of interval data"
        " without external lighting." ),
    tr( "Rainbow colormap with sawtooth-shaped lightness." ),
    tr( "Diverging scheme with colours starting and ending at black. Used for representation of data"
        " with critical middle value and natural rainbow interpretation (\"cold\" and \"hot\")." ),
};

const QString ImprovedRainbowColorMap::SCHEMES_NAMES[ ImprovedRainbowColorMap::SCHEMES_COUNT ] = {
    tr( "IsoL - constant lightness" ),
    tr( "IsoAZ - full Hue circle" ),
    tr( "IsoAZ180 - twice Hue circle" ),
    tr( "LinearL - linear lightness" ),
    tr( "LinLhot - linear lightness" ),
    tr( "CubicYF - cubic-law lightness" ),
    tr( "CubicL - cubic-law lightness" ),
    tr( "Swtth - sawtooth-shaped lightness" ),
    tr( "Edge - diverging scheme" ),
};

ImprovedRainbowColorMap::ImprovedRainbowColorMap() :
    currentScheme( DEFAULT_SCHEME )
{
    widget = new ImprovedRainbowColorMapWidget( *this, currentScheme );

    // qRegisterMetaType<ImprovedRainbowColorMap::Schemes>("ImprovedRainbowColorMap::Schemes");
    QObject::connect( widget, SIGNAL( schemeChanged( ImprovedRainbowColorMap::Schemes ) ),
                      this, SLOT( schemeChanged( ImprovedRainbowColorMap::Schemes ) ) );
}

ImprovedRainbowColorMap::~ImprovedRainbowColorMap()
{
}

const QString&
ImprovedRainbowColorMap::getColorMapDescription() const
{
    return COLOR_MAP_DESCRIPTION;
}

ColorMapWidget*
ImprovedRainbowColorMap::getConfigurationPanel()
{
    return widget;
}

QColor
ImprovedRainbowColorMap::getColor( double value, double minValue,
                                   double maxValue, bool whiteForZero ) const
{
    pair<unsigned short, QColor> preprocess = ColorMapExtended::getColorBasicProcessing( value,
                                                                                         minValue, maxValue, whiteForZero );
    // preprocess returns value
    if ( preprocess.first == 2 )
    {
        return preprocess.second;
    }
    double               position;
    pair<double, double> minMax = adjustFilteringBorders( minValue, maxValue );
    minValue = minMax.first;
    maxValue = minMax.second;

    // usual interpolation
    if ( preprocess.first == 0 )
    {
        position = ( value - minValue ) / ( maxValue - minValue );
    }
    // case when maxValue == minValue
    else
    {
        position = preprocess.second.red() / 255.0;
    }

    if ( invertedCM )
    {
        position = 1.0 - position;
    }
    position = min( position, 1.0 );
    position = max( position, 0.0 );
    int array_position = floor( position * ( SCHEME_SIZE - 1 ) );

    return QColor::fromRgb( 255 * SCHEMES_DATA[ currentScheme ][ array_position ][ 0 ],
                            255 * SCHEMES_DATA[ currentScheme ][ array_position ][ 1 ],
                            255 * SCHEMES_DATA[ currentScheme ][ array_position ][ 2 ] );
}

QString
ImprovedRainbowColorMap::getMapName() const
{
    return tr( "Perceptually improved colormaps" );
}

void
ImprovedRainbowColorMap::saveGlobalSettings( QSettings& settings )
{
    ColorMapExtended::saveGlobalSettings( settings );
    settings.setValue( QString( "IRCM_selection" ), currentScheme );
}

void
ImprovedRainbowColorMap::loadGlobalSettings( QSettings& settings )
{
    ColorMapExtended::loadGlobalSettings( settings );
    int scheme = settings.value( QString( "IRCM_selection" ), 0 ).toInt();
    // incorrect value
    if ( scheme < 0 || scheme >= SCHEMES_END )
    {
        scheme = 0;
    }
    currentScheme = static_cast<Schemes>( scheme );
    widget->parentUpdated();
}

void
ImprovedRainbowColorMap::schemeChanged( ImprovedRainbowColorMap::Schemes newScheme )
{
    currentScheme = newScheme;
}

ImprovedRainbowColorMap::Schemes
ImprovedRainbowColorMap::getScheme()
{
    return currentScheme;
}

void
ImprovedRainbowColorMap::setScheme( Schemes scheme )
{
    currentScheme = scheme;
}

ImprovedRainbowColorMapWidget::ImprovedRainbowColorMapWidget( ImprovedRainbowColorMap&         map,
                                                              ImprovedRainbowColorMap::Schemes scheme ) :
    ColorMapWidget( map ), currentScheme( scheme )
{
    // populate combobox
    for ( int i = 0; i < getParent().SCHEMES_COUNT; ++i )
    {
        // enums are numbered 0, ..., SCHEMES_COUNT - 1
        schemeChoose.addItem( getParent().SCHEMES_NAMES[ i ], i );
    }
    schemeChoose.setCurrentIndex( scheme );
    schemeDescription.setWordWrap( true );
    updateDescription( scheme );

    layout.addWidget( &schemeChoose );
    layout.addWidget( &schemeDescription );
    addPlotToGUI( layout );
    enablePlotColorChange( false );
    enablePlotMiddleMark( false );
    this->setLayout( &layout );

    QObject::connect( &schemeChoose, SIGNAL( currentIndexChanged( int ) ),
                      this, SLOT( schemeChanged( int ) ) );
    setMinimumHeight( 200 );

    applyChanges();
}


ImprovedRainbowColorMapWidget::~ImprovedRainbowColorMapWidget()
{
}

void
ImprovedRainbowColorMapWidget::updateDescription( ImprovedRainbowColorMap::Schemes scheme )
{
    schemeDescription.setText( tr( "<b>Description</b>\n" ) + getParent().SCHEMES_DESCRIPTION[ scheme ] );
}

void
ImprovedRainbowColorMapWidget::schemeChanged( int index )
{
    ImprovedRainbowColorMap::Schemes scheme = static_cast<ImprovedRainbowColorMap::Schemes>( index );
    updateDescription( scheme );
    // propagate change
    emit schemeChanged( scheme );
    // repaint plot after changing color map
    colorMapUpdated();
}

ImprovedRainbowColorMap&
ImprovedRainbowColorMapWidget::getParent() const
{
    return dynamic_cast<ImprovedRainbowColorMap&>( parent );
}

void
ImprovedRainbowColorMapWidget::parentUpdated()
{
    schemeChoose.setCurrentIndex( getParent().getScheme() );
}

void
ImprovedRainbowColorMapWidget::applyChanges()
{
    ColorMapWidget::applyChanges();

    cachedSchemeChoose = getParent().getScheme();
}

void
ImprovedRainbowColorMapWidget::revertChanges()
{
    ColorMapWidget::revertChanges();

    getParent().setScheme( cachedSchemeChoose );
    /**
     * Revert GUI
     */
    schemeChoose.setCurrentIndex( cachedSchemeChoose );
}

/**
 * INTERPOLATION DATA
 * Taken from: http://www.mathworks.com/matlabcentral/fileexchange/28982-perceptually-improved-colormaps
 * Interpolated with Matlab for 256 points
 */

const double ImprovedRainbowColorMap::SCHEMES_DATA[ SCHEMES_COUNT ][ SCHEME_SIZE ][ 3 ] = {
    {
        { 9.1020000e-01, 2.2360000e-01, 8.9970000e-01 },
        { 8.9883141e-01, 2.2580469e-01, 9.0548507e-01 },
        { 8.8751033e-01, 2.2804206e-01, 9.1104330e-01 },
        { 8.7623732e-01, 2.3031181e-01, 9.1637924e-01 },
        { 8.6501294e-01, 2.3261362e-01, 9.2149741e-01 },
        { 8.5383776e-01, 2.3494716e-01, 9.2640236e-01 },
        { 8.4271234e-01, 2.3731212e-01, 9.3109862e-01 },
        { 8.3163725e-01, 2.3970818e-01, 9.3559072e-01 },
        { 8.2061304e-01, 2.4213502e-01, 9.3988321e-01 },
        { 8.0964028e-01, 2.4459234e-01, 9.4398062e-01 },
        { 7.9871954e-01, 2.4707979e-01, 9.4788749e-01 },
        { 7.8785136e-01, 2.4959708e-01, 9.5160836e-01 },
        { 7.7703633e-01, 2.5214388e-01, 9.5514775e-01 },
        { 7.6627500e-01, 2.5471987e-01, 9.5851021e-01 },
        { 7.5556794e-01, 2.5732474e-01, 9.6170028e-01 },
        { 7.4491570e-01, 2.5995817e-01, 9.6472249e-01 },
        { 7.3431885e-01, 2.6261984e-01, 9.6758138e-01 },
        { 7.2377796e-01, 2.6530943e-01, 9.7028148e-01 },
        { 7.1329358e-01, 2.6802663e-01, 9.7282734e-01 },
        { 7.0286628e-01, 2.7077111e-01, 9.7522348e-01 },
        { 6.9249663e-01, 2.7354256e-01, 9.7747445e-01 },
        { 6.8218518e-01, 2.7634066e-01, 9.7958478e-01 },
        { 6.7193250e-01, 2.7916510e-01, 9.8155900e-01 },
        { 6.6173915e-01, 2.8201555e-01, 9.8340167e-01 },
        { 6.5160570e-01, 2.8489170e-01, 9.8511730e-01 },
        { 6.4153270e-01, 2.8779323e-01, 9.8671044e-01 },
        { 6.3152073e-01, 2.9071982e-01, 9.8818563e-01 },
        { 6.2157034e-01, 2.9367116e-01, 9.8954740e-01 },
        { 6.1168209e-01, 2.9664692e-01, 9.9080029e-01 },
        { 6.0185656e-01, 2.9964679e-01, 9.9194884e-01 },
        { 5.9209430e-01, 3.0267045e-01, 9.9299758e-01 },
        { 5.8239587e-01, 3.0571758e-01, 9.9395104e-01 },
        { 5.7276185e-01, 3.0878787e-01, 9.9481378e-01 },
        { 5.6319279e-01, 3.1188100e-01, 9.9559031e-01 },
        { 5.5368925e-01, 3.1499664e-01, 9.9628519e-01 },
        { 5.4425180e-01, 3.1813448e-01, 9.9690293e-01 },
        { 5.3488100e-01, 3.2129421e-01, 9.9744810e-01 },
        { 5.2557741e-01, 3.2447551e-01, 9.9792521e-01 },
        { 5.1634160e-01, 3.2767805e-01, 9.9833881e-01 },
        { 5.0717414e-01, 3.3090152e-01, 9.9869343e-01 },
        { 4.9807557e-01, 3.3414560e-01, 9.9899361e-01 },
        { 4.8904647e-01, 3.3740998e-01, 9.9924388e-01 },
        { 4.8008741e-01, 3.4069433e-01, 9.9944879e-01 },
        { 4.7119893e-01, 3.4399835e-01, 9.9961287e-01 },
        { 4.6238161e-01, 3.4732170e-01, 9.9974065e-01 },
        { 4.5363600e-01, 3.5066407e-01, 9.9983668e-01 },
        { 4.4496268e-01, 3.5402515e-01, 9.9990549e-01 },
        { 4.3636221e-01, 3.5740462e-01, 9.9995161e-01 },
        { 4.2783514e-01, 3.6080216e-01, 9.9997958e-01 },
        { 4.1938204e-01, 3.6421744e-01, 9.9999395e-01 },
        { 4.1100347e-01, 3.6765016e-01, 9.9999924e-01 },
        { 4.0270000e-01, 3.7110000e-01, 1.0000000e+00 },
        { 3.9434756e-01, 3.7465362e-01, 9.9971294e-01 },
        { 3.8583050e-01, 3.7839161e-01, 9.9886213e-01 },
        { 3.7716207e-01, 3.8230449e-01, 9.9746307e-01 },
        { 3.6835547e-01, 3.8638283e-01, 9.9553131e-01 },
        { 3.5942393e-01, 3.9061714e-01, 9.9308237e-01 },
        { 3.5038067e-01, 3.9499798e-01, 9.9013177e-01 },
        { 3.4123892e-01, 3.9951588e-01, 9.8669504e-01 },
        { 3.3201189e-01, 4.0416139e-01, 9.8278771e-01 },
        { 3.2271280e-01, 4.0892504e-01, 9.7842530e-01 },
        { 3.1335489e-01, 4.1379738e-01, 9.7362334e-01 },
        { 3.0395137e-01, 4.1876894e-01, 9.6839735e-01 },
        { 2.9451546e-01, 4.2383027e-01, 9.6276287e-01 },
        { 2.8506039e-01, 4.2897190e-01, 9.5673541e-01 },
        { 2.7559937e-01, 4.3418439e-01, 9.5033051e-01 },
        { 2.6614563e-01, 4.3945826e-01, 9.4356369e-01 },
        { 2.5671240e-01, 4.4478405e-01, 9.3645048e-01 },
        { 2.4731289e-01, 4.5015232e-01, 9.2900640e-01 },
        { 2.3796032e-01, 4.5555359e-01, 9.2124698e-01 },
        { 2.2866792e-01, 4.6097841e-01, 9.1318775e-01 },
        { 2.1944891e-01, 4.6641733e-01, 9.0484422e-01 },
        { 2.1031651e-01, 4.7186087e-01, 8.9623194e-01 },
        { 2.0128395e-01, 4.7729958e-01, 8.8736642e-01 },
        { 1.9236444e-01, 4.8272400e-01, 8.7826320e-01 },
        { 1.8357120e-01, 4.8812467e-01, 8.6893779e-01 },
        { 1.7491747e-01, 4.9349213e-01, 8.5940572e-01 },
        { 1.6641646e-01, 4.9881692e-01, 8.4968253e-01 },
        { 1.5808139e-01, 5.0408959e-01, 8.3978373e-01 },
        { 1.4992549e-01, 5.0930067e-01, 8.2972486e-01 },
        { 1.4196197e-01, 5.1444070e-01, 8.1952143e-01 },
        { 1.3420407e-01, 5.1950022e-01, 8.0918898e-01 },
        { 1.2666499e-01, 5.2446977e-01, 7.9874303e-01 },
        { 1.1935797e-01, 5.2933990e-01, 7.8819911e-01 },
        { 1.1229623e-01, 5.3410115e-01, 7.7757275e-01 },
        { 1.0549298e-01, 5.3874404e-01, 7.6687946e-01 },
        { 9.8961449e-02, 5.4325913e-01, 7.5613479e-01 },
        { 9.2714862e-02, 5.4763696e-01, 7.4535424e-01 },
        { 8.6766439e-02, 5.5186806e-01, 7.3455336e-01 },
        { 8.1129400e-02, 5.5594298e-01, 7.2374766e-01 },
        { 7.5816970e-02, 5.5985225e-01, 7.1295268e-01 },
        { 7.0842369e-02, 5.6358642e-01, 7.0218393e-01 },
        { 6.6218819e-02, 5.6713603e-01, 6.9145695e-01 },
        { 6.1959543e-02, 5.7049161e-01, 6.8078726e-01 },
        { 5.8077763e-02, 5.7364371e-01, 6.7019038e-01 },
        { 5.4586700e-02, 5.7658287e-01, 6.5968186e-01 },
        { 5.1499577e-02, 5.7929963e-01, 6.4927720e-01 },
        { 4.8829615e-02, 5.8178453e-01, 6.3899194e-01 },
        { 4.6590037e-02, 5.8402810e-01, 6.2884160e-01 },
        { 4.4794064e-02, 5.8602090e-01, 6.1884171e-01 },
        { 4.3454919e-02, 5.8775345e-01, 6.0900779e-01 },
        { 4.2585824e-02, 5.8921631e-01, 5.9935538e-01 },
        { 4.2200000e-02, 5.9040000e-01, 5.8990000e-01 },
        { 4.2061549e-02, 5.9143492e-01, 5.8026958e-01 },
        { 4.1925760e-02, 5.9245753e-01, 5.7010114e-01 },
        { 4.1792637e-02, 5.9346749e-01, 5.5942386e-01 },
        { 4.1662181e-02, 5.9446449e-01, 5.4826688e-01 },
        { 4.1534398e-02, 5.9544819e-01, 5.3665938e-01 },
        { 4.1409289e-02, 5.9641825e-01, 5.2463052e-01 },
        { 4.1286858e-02, 5.9737435e-01, 5.1220946e-01 },
        { 4.1167108e-02, 5.9831616e-01, 4.9942537e-01 },
        { 4.1050043e-02, 5.9924335e-01, 4.8630740e-01 },
        { 4.0935665e-02, 6.0015558e-01, 4.7288472e-01 },
        { 4.0823978e-02, 6.0105252e-01, 4.5918650e-01 },
        { 4.0714986e-02, 6.0193385e-01, 4.4524189e-01 },
        { 4.0608690e-02, 6.0279923e-01, 4.3108006e-01 },
        { 4.0505095e-02, 6.0364834e-01, 4.1673017e-01 },
        { 4.0404204e-02, 6.0448083e-01, 4.0222139e-01 },
        { 4.0306020e-02, 6.0529639e-01, 3.8758287e-01 },
        { 4.0210547e-02, 6.0609468e-01, 3.7284379e-01 },
        { 4.0117786e-02, 6.0687536e-01, 3.5803330e-01 },
        { 4.0027742e-02, 6.0763812e-01, 3.4318056e-01 },
        { 3.9940419e-02, 6.0838261e-01, 3.2831475e-01 },
        { 3.9855818e-02, 6.0910852e-01, 3.1346502e-01 },
        { 3.9773943e-02, 6.0981550e-01, 2.9866054e-01 },
        { 3.9694798e-02, 6.1050322e-01, 2.8393046e-01 },
        { 3.9618386e-02, 6.1117136e-01, 2.6930396e-01 },
        { 3.9544710e-02, 6.1181959e-01, 2.5481019e-01 },
        { 3.9473773e-02, 6.1244757e-01, 2.4047831e-01 },
        { 3.9405578e-02, 6.1305498e-01, 2.2633750e-01 },
        { 3.9340129e-02, 6.1364147e-01, 2.1241692e-01 },
        { 3.9277429e-02, 6.1420673e-01, 1.9874572e-01 },
        { 3.9217481e-02, 6.1475043e-01, 1.8535306e-01 },
        { 3.9160288e-02, 6.1527222e-01, 1.7226813e-01 },
        { 3.9105854e-02, 6.1577178e-01, 1.5952006e-01 },
        { 3.9054181e-02, 6.1624878e-01, 1.4713804e-01 },
        { 3.9005273e-02, 6.1670289e-01, 1.3515122e-01 },
        { 3.8959134e-02, 6.1713378e-01, 1.2358876e-01 },
        { 3.8915766e-02, 6.1754112e-01, 1.1247983e-01 },
        { 3.8875172e-02, 6.1792457e-01, 1.0185359e-01 },
        { 3.8837357e-02, 6.1828381e-01, 9.1739202e-02 },
        { 3.8802322e-02, 6.1861851e-01, 8.2165832e-02 },
        { 3.8770072e-02, 6.1892833e-01, 7.3162641e-02 },
        { 3.8740609e-02, 6.1921294e-01, 6.4758793e-02 },
        { 3.8713936e-02, 6.1947202e-01, 5.6983451e-02 },
        { 3.8690058e-02, 6.1970523e-01, 4.9865776e-02 },
        { 3.8668977e-02, 6.1991224e-01, 4.3434933e-02 },
        { 3.8650696e-02, 6.2009272e-01, 3.7720084e-02 },
        { 3.8635219e-02, 6.2024635e-01, 3.2750393e-02 },
        { 3.8622549e-02, 6.2037278e-01, 2.8555022e-02 },
        { 3.8612689e-02, 6.2047169e-01, 2.5163134e-02 },
        { 3.8605642e-02, 6.2054275e-01, 2.2603893e-02 },
        { 3.8601411e-02, 6.2058563e-01, 2.0906460e-02 },
        { 3.8600000e-02, 6.2060000e-01, 2.0100000e-02 },
        { 3.8994716e-02, 6.2055844e-01, 1.9752028e-02 },
        { 4.0162835e-02, 6.2043470e-01, 1.9410626e-02 },
        { 4.2080312e-02, 6.2023017e-01, 1.9075807e-02 },
        { 4.4723103e-02, 6.1994626e-01, 1.8747584e-02 },
        { 4.8067165e-02, 6.1958435e-01, 1.8425971e-02 },
        { 5.2088453e-02, 6.1914585e-01, 1.8110980e-02 },
        { 5.6762922e-02, 6.1863215e-01, 1.7802623e-02 },
        { 6.2066530e-02, 6.1804465e-01, 1.7500914e-02 },
        { 6.7975232e-02, 6.1738476e-01, 1.7205867e-02 },
        { 7.4464983e-02, 6.1665385e-01, 1.6917493e-02 },
        { 8.1511739e-02, 6.1585335e-01, 1.6635806e-02 },
        { 8.9091457e-02, 6.1498463e-01, 1.6360818e-02 },
        { 9.7180092e-02, 6.1404911e-01, 1.6092543e-02 },
        { 1.0575360e-01, 6.1304817e-01, 1.5830994e-02 },
        { 1.1478794e-01, 6.1198321e-01, 1.5576183e-02 },
        { 1.2425906e-01, 6.1085564e-01, 1.5328124e-02 },
        { 1.3414292e-01, 6.0966684e-01, 1.5086829e-02 },
        { 1.4441548e-01, 6.0841823e-01, 1.4852311e-02 },
        { 1.5505270e-01, 6.0711118e-01, 1.4624584e-02 },
        { 1.6603052e-01, 6.0574711e-01, 1.4403659e-02 },
        { 1.7732490e-01, 6.0432741e-01, 1.4189551e-02 },
        { 1.8891181e-01, 6.0285348e-01, 1.3982272e-02 },
        { 2.0076719e-01, 6.0132671e-01, 1.3781835e-02 },
        { 2.1286700e-01, 5.9974850e-01, 1.3588253e-02 },
        { 2.2518720e-01, 5.9812025e-01, 1.3401539e-02 },
        { 2.3770375e-01, 5.9644336e-01, 1.3221706e-02 },
        { 2.5039259e-01, 5.9471923e-01, 1.3048767e-02 },
        { 2.6322969e-01, 5.9294924e-01, 1.2882734e-02 },
        { 2.7619100e-01, 5.9113481e-01, 1.2723621e-02 },
        { 2.8925248e-01, 5.8927732e-01, 1.2571440e-02 },
        { 3.0239008e-01, 5.8737818e-01, 1.2426205e-02 },
        { 3.1557976e-01, 5.8543878e-01, 1.2287929e-02 },
        { 3.2879747e-01, 5.8346052e-01, 1.2156624e-02 },
        { 3.4201918e-01, 5.8144480e-01, 1.2032303e-02 },
        { 3.5522083e-01, 5.7939301e-01, 1.1914980e-02 },
        { 3.6837839e-01, 5.7730656e-01, 1.1804667e-02 },
        { 3.8146780e-01, 5.7518684e-01, 1.1701377e-02 },
        { 3.9446503e-01, 5.7303524e-01, 1.1605123e-02 },
        { 4.0734602e-01, 5.7085317e-01, 1.1515919e-02 },
        { 4.2008675e-01, 5.6864202e-01, 1.1433776e-02 },
        { 4.3266315e-01, 5.6640319e-01, 1.1358708e-02 },
        { 4.4505120e-01, 5.6413808e-01, 1.1290728e-02 },
        { 4.5722683e-01, 5.6184809e-01, 1.1229849e-02 },
        { 4.6916602e-01, 5.5953461e-01, 1.1176084e-02 },
        { 4.8084471e-01, 5.5719904e-01, 1.1129445e-02 },
        { 4.9223886e-01, 5.5484278e-01, 1.1089947e-02 },
        { 5.0332443e-01, 5.5246722e-01, 1.1057600e-02 },
        { 5.1407738e-01, 5.5007377e-01, 1.1032420e-02 },
        { 5.2447365e-01, 5.4766381e-01, 1.1014417e-02 },
        { 5.3448920e-01, 5.4523876e-01, 1.1003606e-02 },
        { 5.4410000e-01, 5.4280000e-01, 1.1000000e-02 },
        { 5.5349184e-01, 5.4025524e-01, 1.1085428e-02 },
        { 5.6286652e-01, 5.3751340e-01, 1.1339556e-02 },
        { 5.7222399e-01, 5.3457769e-01, 1.1759149e-02 },
        { 5.8156418e-01, 5.3145134e-01, 1.2340974e-02 },
        { 5.9088703e-01, 5.2813757e-01, 1.3081797e-02 },
        { 6.0019250e-01, 5.2463959e-01, 1.3978384e-02 },
        { 6.0948051e-01, 5.2096064e-01, 1.5027500e-02 },
        { 6.1875102e-01, 5.1710392e-01, 1.6225912e-02 },
        { 6.2800396e-01, 5.1307266e-01, 1.7570385e-02 },
        { 6.3723928e-01, 5.0887008e-01, 1.9057685e-02 },
        { 6.4645692e-01, 5.0449941e-01, 2.0684579e-02 },
        { 6.5565682e-01, 4.9996385e-01, 2.2447832e-02 },
        { 6.6483893e-01, 4.9526663e-01, 2.4344211e-02 },
        { 6.7400318e-01, 4.9041097e-01, 2.6370480e-02 },
        { 6.8314952e-01, 4.8540010e-01, 2.8523407e-02 },
        { 6.9227789e-01, 4.8023722e-01, 3.0799757e-02 },
        { 7.0138823e-01, 4.7492557e-01, 3.3196296e-02 },
        { 7.1048048e-01, 4.6946836e-01, 3.5709790e-02 },
        { 7.1955459e-01, 4.6386881e-01, 3.8337005e-02 },
        { 7.2861050e-01, 4.5813015e-01, 4.1074707e-02 },
        { 7.3764815e-01, 4.5225559e-01, 4.3919662e-02 },
        { 7.4666749e-01, 4.4624836e-01, 4.6868636e-02 },
        { 7.5566844e-01, 4.4011166e-01, 4.9918394e-02 },
        { 7.6465097e-01, 4.3384874e-01, 5.3065703e-02 },
        { 7.7361500e-01, 4.2746280e-01, 5.6307329e-02 },
        { 7.8256049e-01, 4.2095706e-01, 5.9640037e-02 },
        { 7.9148737e-01, 4.1433475e-01, 6.3060594e-02 },
        { 8.0039558e-01, 4.0759908e-01, 6.6565766e-02 },
        { 8.0928507e-01, 4.0075329e-01, 7.0152318e-02 },
        { 8.1815578e-01, 3.9380057e-01, 7.3817016e-02 },
        { 8.2700766e-01, 3.8674417e-01, 7.7556627e-02 },
        { 8.3584063e-01, 3.7958730e-01, 8.1367916e-02 },
        { 8.4465466e-01, 3.7233317e-01, 8.5247649e-02 },
        { 8.5344967e-01, 3.6498501e-01, 8.9192593e-02 },
        { 8.6222561e-01, 3.5754604e-01, 9.3199512e-02 },
        { 8.7098243e-01, 3.5001947e-01, 9.7265174e-02 },
        { 8.7972006e-01, 3.4240854e-01, 1.0138634e-01 },
        { 8.8843844e-01, 3.3471646e-01, 1.0555979e-01 },
        { 8.9713753e-01, 3.2694645e-01, 1.0978227e-01 },
        { 9.0581725e-01, 3.1910173e-01, 1.1405056e-01 },
        { 9.1447756e-01, 3.1118552e-01, 1.1836142e-01 },
        { 9.2311840e-01, 3.0320104e-01, 1.2271162e-01 },
        { 9.3173970e-01, 2.9515151e-01, 1.2709793e-01 },
        { 9.4034141e-01, 2.8704016e-01, 1.3151710e-01 },
        { 9.4892347e-01, 2.7887019e-01, 1.3596591e-01 },
        { 9.5748583e-01, 2.7064484e-01, 1.4044112e-01 },
        { 9.6602843e-01, 2.6236733e-01, 1.4493949e-01 },
        { 9.7455120e-01, 2.5404086e-01, 1.4945781e-01 },
        { 9.8305409e-01, 2.4566867e-01, 1.5399282e-01 },
        { 9.9153704e-01, 2.3725398e-01, 1.5854129e-01 },
        { 1.0000000e+00, 2.2880000e-01, 1.6310000e-01 }
    },
    {
        { 1.0000000e+00, 2.6270000e-01, 1.0000000e+00 },
        { 9.9795592e-01, 2.6651725e-01, 1.0000000e+00 },
        { 9.9564290e-01, 2.7033451e-01, 1.0000000e+00 },
        { 9.9308212e-01, 2.7415176e-01, 1.0000000e+00 },
        { 9.9029479e-01, 2.7796902e-01, 1.0000000e+00 },
        { 9.8730209e-01, 2.8178627e-01, 1.0000000e+00 },
        { 9.8412520e-01, 2.8560353e-01, 1.0000000e+00 },
        { 9.8078532e-01, 2.8942078e-01, 1.0000000e+00 },
        { 9.7730364e-01, 2.9323804e-01, 1.0000000e+00 },
        { 9.7362118e-01, 2.9702746e-01, 1.0000000e+00 },
        { 9.6958606e-01, 3.0074597e-01, 1.0000000e+00 },
        { 9.6522664e-01, 3.0442830e-01, 1.0000000e+00 },
        { 9.6057823e-01, 3.0811183e-01, 1.0000000e+00 },
        { 9.5567614e-01, 3.1183396e-01, 1.0000000e+00 },
        { 9.5055567e-01, 3.1563210e-01, 1.0000000e+00 },
        { 9.4525214e-01, 3.1954363e-01, 1.0000000e+00 },
        { 9.3980085e-01, 3.2360596e-01, 1.0000000e+00 },
        { 9.3419404e-01, 3.2785412e-01, 1.0000000e+00 },
        { 9.2825529e-01, 3.3228819e-01, 1.0000000e+00 },
        { 9.2199648e-01, 3.3688210e-01, 1.0000000e+00 },
        { 9.1545894e-01, 3.4160918e-01, 1.0000000e+00 },
        { 9.0868397e-01, 3.4644278e-01, 1.0000000e+00 },
        { 9.0171289e-01, 3.5135622e-01, 1.0000000e+00 },
        { 8.9458701e-01, 3.5632284e-01, 1.0000000e+00 },
        { 8.8734763e-01, 3.6131599e-01, 1.0000000e+00 },
        { 8.8002142e-01, 3.6631344e-01, 1.0000000e+00 },
        { 8.7244972e-01, 3.7135706e-01, 1.0000000e+00 },
        { 8.6461667e-01, 3.7646306e-01, 1.0000000e+00 },
        { 8.5657161e-01, 3.8163002e-01, 1.0000000e+00 },
        { 8.4836388e-01, 3.8685653e-01, 1.0000000e+00 },
        { 8.4004285e-01, 3.9214118e-01, 1.0000000e+00 },
        { 8.3165785e-01, 3.9748254e-01, 1.0000000e+00 },
        { 8.2325825e-01, 4.0287922e-01, 1.0000000e+00 },
        { 8.1489402e-01, 4.0832941e-01, 1.0000000e+00 },
        { 8.0666428e-01, 4.1379065e-01, 1.0000000e+00 },
        { 7.9854318e-01, 4.1925838e-01, 1.0000000e+00 },
        { 7.9042865e-01, 4.2476618e-01, 1.0000000e+00 },
        { 7.8221863e-01, 4.3034764e-01, 1.0000000e+00 },
        { 7.7381105e-01, 4.3603633e-01, 1.0000000e+00 },
        { 7.6510385e-01, 4.4186583e-01, 1.0000000e+00 },
        { 7.5599496e-01, 4.4786972e-01, 1.0000000e+00 },
        { 7.4638231e-01, 4.5408158e-01, 1.0000000e+00 },
        { 7.3629865e-01, 4.6050657e-01, 9.9994330e-01 },
        { 7.2587687e-01, 4.6711545e-01, 9.9974650e-01 },
        { 7.1504742e-01, 4.7391233e-01, 9.9942227e-01 },
        { 7.0373787e-01, 4.8090182e-01, 9.9898457e-01 },
        { 6.9187579e-01, 4.8808855e-01, 9.9844734e-01 },
        { 6.7938877e-01, 4.9547714e-01, 9.9782455e-01 },
        { 6.6620438e-01, 5.0307223e-01, 9.9713015e-01 },
        { 6.5225019e-01, 5.1087844e-01, 9.9637809e-01 },
        { 6.3728977e-01, 5.1892766e-01, 9.9546675e-01 },
        { 6.2081602e-01, 5.2730191e-01, 9.9403546e-01 },
        { 6.0301026e-01, 5.3597462e-01, 9.9212856e-01 },
        { 5.8410989e-01, 5.4491014e-01, 9.8982681e-01 },
        { 5.6435235e-01, 5.5407281e-01, 9.8721094e-01 },
        { 5.4397505e-01, 5.6342698e-01, 9.8436168e-01 },
        { 5.2321543e-01, 5.7293699e-01, 9.8135977e-01 },
        { 5.0231089e-01, 5.8256719e-01, 9.7828594e-01 },
        { 4.8136387e-01, 5.9232445e-01, 9.7516846e-01 },
        { 4.5948953e-01, 6.0249882e-01, 9.7164716e-01 },
        { 4.3672070e-01, 6.1304096e-01, 9.6770989e-01 },
        { 4.1335221e-01, 6.2381434e-01, 9.6344557e-01 },
        { 3.8967892e-01, 6.3468241e-01, 9.5894308e-01 },
        { 3.6599569e-01, 6.4550864e-01, 9.5429134e-01 },
        { 3.4259738e-01, 6.5615649e-01, 9.4957924e-01 },
        { 3.1977883e-01, 6.6648942e-01, 9.4489569e-01 },
        { 2.9780451e-01, 6.7638314e-01, 9.4032646e-01 },
        { 2.7590234e-01, 6.8614065e-01, 9.3582207e-01 },
        { 2.5383094e-01, 6.9587816e-01, 9.3130837e-01 },
        { 2.3187556e-01, 7.0550422e-01, 9.2675221e-01 },
        { 2.1032145e-01, 7.1492740e-01, 9.2212041e-01 },
        { 1.8945384e-01, 7.2405623e-01, 9.1737982e-01 },
        { 1.6955800e-01, 7.3279927e-01, 9.1249728e-01 },
        { 1.5091918e-01, 7.4106508e-01, 9.0743963e-01 },
        { 1.3382261e-01, 7.4876221e-01, 9.0217371e-01 },
        { 1.1738726e-01, 7.5615171e-01, 8.9663404e-01 },
        { 1.0072420e-01, 7.6350973e-01, 8.9080553e-01 },
        { 8.4348057e-02, 7.7069507e-01, 8.8473980e-01 },
        { 6.8774805e-02, 7.7756618e-01, 8.7848854e-01 },
        { 5.4520416e-02, 7.8398147e-01, 8.7210342e-01 },
        { 4.2100864e-02, 7.8979938e-01, 8.6563612e-01 },
        { 3.2032121e-02, 7.9487834e-01, 8.5913830e-01 },
        { 2.4830160e-02, 7.9907677e-01, 8.5266165e-01 },
        { 2.0070577e-02, 8.0248039e-01, 8.4631392e-01 },
        { 1.5691086e-02, 8.0559231e-01, 8.4018807e-01 },
        { 1.1671974e-02, 8.0844248e-01, 8.3414776e-01 },
        { 8.1025593e-03, 8.1103499e-01, 8.2804770e-01 },
        { 5.0721641e-03, 8.1337391e-01, 8.2174257e-01 },
        { 2.6701087e-03, 8.1546333e-01, 8.1508707e-01 },
        { 9.8571348e-04, 8.1730732e-01, 8.0793588e-01 },
        { 1.0829903e-04, 8.1890996e-01, 8.0014371e-01 },
        { 0.0000000e+00, 8.2029083e-01, 7.9151577e-01 },
        { 0.0000000e+00, 8.2154246e-01, 7.8174870e-01 },
        { 0.0000000e+00, 8.2268573e-01, 7.7095573e-01 },
        { 0.0000000e+00, 8.2372798e-01, 7.5930589e-01 },
        { 0.0000000e+00, 8.2467659e-01, 7.4696818e-01 },
        { 0.0000000e+00, 8.2553892e-01, 7.3411159e-01 },
        { 0.0000000e+00, 8.2632233e-01, 7.2090513e-01 },
        { 0.0000000e+00, 8.2703418e-01, 7.0751780e-01 },
        { 0.0000000e+00, 8.2768107e-01, 6.9408245e-01 },
        { 0.0000000e+00, 8.2825919e-01, 6.8014459e-01 },
        { 0.0000000e+00, 8.2877754e-01, 6.6559897e-01 },
        { 0.0000000e+00, 8.2925037e-01, 6.5055114e-01 },
        { 0.0000000e+00, 8.2969195e-01, 6.3510660e-01 },
        { 0.0000000e+00, 8.3011652e-01, 6.1937091e-01 },
        { 0.0000000e+00, 8.3053832e-01, 6.0344958e-01 },
        { 0.0000000e+00, 8.3097162e-01, 5.8744813e-01 },
        { 0.0000000e+00, 8.3143059e-01, 5.7147059e-01 },
        { 0.0000000e+00, 8.3190471e-01, 5.5522347e-01 },
        { 0.0000000e+00, 8.3237882e-01, 5.3853570e-01 },
        { 0.0000000e+00, 8.3285294e-01, 5.2156230e-01 },
        { 0.0000000e+00, 8.3332706e-01, 5.0445829e-01 },
        { 0.0000000e+00, 8.3380118e-01, 4.8737869e-01 },
        { 0.0000000e+00, 8.3427529e-01, 4.7047854e-01 },
        { 0.0000000e+00, 8.3474941e-01, 4.5391284e-01 },
        { 0.0000000e+00, 8.3522353e-01, 4.3783664e-01 },
        { 0.0000000e+00, 8.3573406e-01, 4.2210545e-01 },
        { 0.0000000e+00, 8.3632307e-01, 4.0632732e-01 },
        { 0.0000000e+00, 8.3695006e-01, 3.9063551e-01 },
        { 0.0000000e+00, 8.3757298e-01, 3.7517432e-01 },
        { 0.0000000e+00, 8.3814979e-01, 3.6008805e-01 },
        { 0.0000000e+00, 8.3863844e-01, 3.4552098e-01 },
        { 0.0000000e+00, 8.3899691e-01, 3.3161741e-01 },
        { 0.0000000e+00, 8.3918313e-01, 3.1852164e-01 },
        { 0.0000000e+00, 8.3920000e-01, 3.0627052e-01 },
        { 0.0000000e+00, 8.3920000e-01, 2.9451873e-01 },
        { 0.0000000e+00, 8.3920000e-01, 2.8321494e-01 },
        { 0.0000000e+00, 8.3920000e-01, 2.7234680e-01 },
        { 0.0000000e+00, 8.3920000e-01, 2.6190194e-01 },
        { 0.0000000e+00, 8.3920000e-01, 2.5186798e-01 },
        { 0.0000000e+00, 8.3920000e-01, 2.4223257e-01 },
        { 0.0000000e+00, 8.3920000e-01, 2.3298333e-01 },
        { 2.0462864e-04, 8.3918809e-01, 2.2411888e-01 },
        { 2.5055938e-03, 8.3905241e-01, 2.1571186e-01 },
        { 7.0595651e-03, 8.3877879e-01, 2.0771023e-01 },
        { 1.3486421e-02, 8.3838396e-01, 2.0002994e-01 },
        { 2.1406039e-02, 8.3788465e-01, 1.9258694e-01 },
        { 3.0438297e-02, 8.3729758e-01, 1.8529718e-01 },
        { 4.0203074e-02, 8.3663950e-01, 1.7807661e-01 },
        { 5.0320248e-02, 8.3592713e-01, 1.7084119e-01 },
        { 6.0449868e-02, 8.3517147e-01, 1.6350903e-01 },
        { 7.2167482e-02, 8.3412123e-01, 1.5612592e-01 },
        { 8.6154569e-02, 8.3268378e-01, 1.4876176e-01 },
        { 1.0200312e-01, 8.3092734e-01, 1.4144073e-01 },
        { 1.1930514e-01, 8.2892013e-01, 1.3418702e-01 },
        { 1.3765260e-01, 8.2673035e-01, 1.2702479e-01 },
        { 1.5663752e-01, 8.2442623e-01, 1.1997823e-01 },
        { 1.7585187e-01, 8.2207599e-01, 1.1307151e-01 },
        { 1.9488766e-01, 8.1974783e-01, 1.0632881e-01 },
        { 2.1419654e-01, 8.1736628e-01, 9.9670013e-02 },
        { 2.3457852e-01, 8.1479201e-01, 9.3008699e-02 },
        { 2.5580109e-01, 8.1205490e-01, 8.6405310e-02 },
        { 2.7762776e-01, 8.0918549e-01, 7.9920806e-02 },
        { 2.9982208e-01, 8.0621431e-01, 7.3616150e-02 },
        { 3.2214755e-01, 8.0317191e-01, 6.7552304e-02 },
        { 3.4436770e-01, 8.0008881e-01, 6.1790231e-02 },
        { 3.6624605e-01, 7.9699557e-01, 5.6390891e-02 },
        { 3.8789080e-01, 7.9388247e-01, 5.1323283e-02 },
        { 4.1001780e-01, 7.9066391e-01, 4.6380117e-02 },
        { 4.3241999e-01, 7.8735703e-01, 4.1585896e-02 },
        { 4.5483221e-01, 7.8398564e-01, 3.6980216e-02 },
        { 4.7698930e-01, 7.8057352e-01, 3.2602671e-02 },
        { 4.9862612e-01, 7.7714446e-01, 2.8492856e-02 },
        { 5.1947750e-01, 7.7372225e-01, 2.4690365e-02 },
        { 5.3927830e-01, 7.7033069e-01, 2.1234793e-02 },
        { 5.5791671e-01, 7.6700923e-01, 1.8091165e-02 },
        { 5.7615543e-01, 7.6384445e-01, 1.4862335e-02 },
        { 5.9402275e-01, 7.6078008e-01, 1.1609383e-02 },
        { 6.1137800e-01, 7.5773748e-01, 8.4831739e-03 },
        { 6.2808048e-01, 7.5463801e-01, 5.6345704e-03 },
        { 6.4398953e-01, 7.5140302e-01, 3.2144370e-03 },
        { 6.5896446e-01, 7.4795388e-01, 1.3736378e-03 },
        { 6.7286461e-01, 7.4421194e-01, 2.6303657e-04 },
        { 6.8555776e-01, 7.4010015e-01, 4.8931118e-06 },
        { 6.9716618e-01, 7.3559595e-01, 1.1314231e-04 },
        { 7.0791838e-01, 7.3071229e-01, 3.5436167e-04 },
        { 7.1800301e-01, 7.2545680e-01, 7.1662665e-04 },
        { 7.2760870e-01, 7.1983712e-01, 1.1880127e-03 },
        { 7.3692412e-01, 7.1386088e-01, 1.7565952e-03 },
        { 7.4613790e-01, 7.0753570e-01, 2.4104497e-03 },
        { 7.5543870e-01, 7.0086922e-01, 3.1376515e-03 },
        { 7.6501495e-01, 6.9386876e-01, 3.9266979e-03 },
        { 7.7480809e-01, 6.8625861e-01, 5.1749511e-03 },
        { 7.8462644e-01, 6.7787142e-01, 7.1486921e-03 },
        { 7.9442235e-01, 6.6882363e-01, 9.7158627e-03 },
        { 8.0414820e-01, 6.5923170e-01, 1.2744404e-02 },
        { 8.1375634e-01, 6.4921208e-01, 1.6102258e-02 },
        { 8.2319913e-01, 6.3888123e-01, 1.9657367e-02 },
        { 8.3242895e-01, 6.2835558e-01, 2.3277671e-02 },
        { 8.4139815e-01, 6.1775160e-01, 2.6831113e-02 },
        { 8.5012429e-01, 6.0691951e-01, 3.0352998e-02 },
        { 8.5871076e-01, 5.9546394e-01, 3.4109893e-02 },
        { 8.6714929e-01, 5.8353226e-01, 3.8090746e-02 },
        { 8.7542809e-01, 5.7128746e-01, 4.2275565e-02 },
        { 8.8353536e-01, 5.5889252e-01, 4.6644360e-02 },
        { 8.9145931e-01, 5.4651042e-01, 5.1177136e-02 },
        { 8.9918814e-01, 5.3430416e-01, 5.5853904e-02 },
        { 9.0671006e-01, 5.2243672e-01, 6.0654671e-02 },
        { 9.1406937e-01, 5.1095584e-01, 6.5592769e-02 },
        { 9.2146343e-01, 4.9948025e-01, 7.0789112e-02 },
        { 9.2882048e-01, 4.8802895e-01, 7.6226919e-02 },
        { 9.3603679e-01, 4.7667850e-01, 8.1871604e-02 },
        { 9.4300863e-01, 4.6550548e-01, 8.7688579e-02 },
        { 9.4963229e-01, 4.5458646e-01, 9.3643255e-02 },
        { 9.5580403e-01, 4.4399800e-01, 9.9701044e-02 },
        { 9.6142012e-01, 4.3381669e-01, 1.0582736e-01 },
        { 9.6643577e-01, 4.2406784e-01, 1.1199586e-01 },
        { 9.7140866e-01, 4.1427245e-01, 1.1827640e-01 },
        { 9.7636674e-01, 4.0445268e-01, 1.2468871e-01 },
        { 9.8113848e-01, 3.9480865e-01, 1.3122725e-01 },
        { 9.8555236e-01, 3.8554050e-01, 1.3788649e-01 },
        { 9.8943686e-01, 3.7684837e-01, 1.4466089e-01 },
        { 9.9262046e-01, 3.6893240e-01, 1.5154493e-01 },
        { 9.9493162e-01, 3.6199272e-01, 1.5853306e-01 },
        { 9.9620830e-01, 3.5622229e-01, 1.6561743e-01 },
        { 9.9700908e-01, 3.5123927e-01, 1.7264666e-01 },
        { 9.9773504e-01, 3.4669644e-01, 1.7961456e-01 },
        { 9.9837689e-01, 3.4254202e-01, 1.8663639e-01 },
        { 9.9892534e-01, 3.3872425e-01, 1.9382744e-01 },
        { 9.9937110e-01, 3.3519137e-01, 2.0130298e-01 },
        { 9.9970488e-01, 3.3189159e-01, 2.0917828e-01 },
        { 9.9991739e-01, 3.2877315e-01, 2.1756863e-01 },
        { 9.9999934e-01, 3.2578429e-01, 2.2658930e-01 },
        { 1.0000000e+00, 3.2295324e-01, 2.3642219e-01 },
        { 1.0000000e+00, 3.2034521e-01, 2.4714960e-01 },
        { 1.0000000e+00, 3.1791222e-01, 2.5867442e-01 },
        { 1.0000000e+00, 3.1560539e-01, 2.7089844e-01 },
        { 1.0000000e+00, 3.1337584e-01, 2.8372344e-01 },
        { 1.0000000e+00, 3.1117466e-01, 2.9705122e-01 },
        { 1.0000000e+00, 3.0895299e-01, 3.1078357e-01 },
        { 1.0000000e+00, 3.0666193e-01, 3.2482226e-01 },
        { 1.0000000e+00, 3.0425558e-01, 3.3922043e-01 },
        { 1.0000000e+00, 3.0174196e-01, 3.5436010e-01 },
        { 1.0000000e+00, 2.9917247e-01, 3.7019059e-01 },
        { 1.0000000e+00, 2.9659959e-01, 3.8662715e-01 },
        { 1.0000000e+00, 2.9407581e-01, 4.0358502e-01 },
        { 1.0000000e+00, 2.9165361e-01, 4.2097945e-01 },
        { 1.0000000e+00, 2.8938549e-01, 4.3872568e-01 },
        { 1.0000000e+00, 2.8732392e-01, 4.5673895e-01 },
        { 1.0000000e+00, 2.8550155e-01, 4.7497353e-01 },
        { 1.0000000e+00, 2.8380507e-01, 4.9365192e-01 },
        { 1.0000000e+00, 2.8220251e-01, 5.1280391e-01 },
        { 1.0000000e+00, 2.8068822e-01, 5.3240477e-01 },
        { 1.0000000e+00, 2.7925657e-01, 5.5242976e-01 },
        { 1.0000000e+00, 2.7790192e-01, 5.7285412e-01 },
        { 1.0000000e+00, 2.7661864e-01, 5.9365314e-01 },
        { 1.0000000e+00, 2.7540107e-01, 6.1480205e-01 },
        { 1.0000000e+00, 2.7424309e-01, 6.3628026e-01 },
        { 1.0000000e+00, 2.7312807e-01, 6.5817580e-01 },
        { 1.0000000e+00, 2.7205343e-01, 6.8052457e-01 },
        { 1.0000000e+00, 2.7102358e-01, 7.0331523e-01 },
        { 1.0000000e+00, 2.7004290e-01, 7.2653644e-01 },
        { 1.0000000e+00, 2.6911580e-01, 7.5017687e-01 },
        { 1.0000000e+00, 2.6824669e-01, 7.7422517e-01 },
        { 1.0000000e+00, 2.6743995e-01, 7.9866999e-01 },
        { 1.0000000e+00, 2.6670000e-01, 8.2350000e-01 }
    },
    {
        { 8.6580000e-01, 5.1330000e-01, 6.2370000e-01 },
        { 8.6149999e-01, 5.1436329e-01, 6.3673607e-01 },
        { 8.5644891e-01, 5.1571039e-01, 6.4955656e-01 },
        { 8.5071311e-01, 5.1731241e-01, 6.6215457e-01 },
        { 8.4435893e-01, 5.1914050e-01, 6.7452321e-01 },
        { 8.3745273e-01, 5.2116576e-01, 6.8665557e-01 },
        { 8.3006087e-01, 5.2335933e-01, 6.9854477e-01 },
        { 8.2224969e-01, 5.2569233e-01, 7.1018391e-01 },
        { 8.1408555e-01, 5.2813589e-01, 7.2156610e-01 },
        { 8.0537438e-01, 5.3079681e-01, 7.3278443e-01 },
        { 7.9562595e-01, 5.3392298e-01, 7.4400268e-01 },
        { 7.8494085e-01, 5.3744276e-01, 7.5511944e-01 },
        { 7.7344231e-01, 5.4127253e-01, 7.6602393e-01 },
        { 7.6125353e-01, 5.4532864e-01, 7.7660539e-01 },
        { 7.4849773e-01, 5.4952745e-01, 7.8675301e-01 },
        { 7.3529813e-01, 5.5378533e-01, 7.9635602e-01 },
        { 7.2177796e-01, 5.5801864e-01, 8.0530364e-01 },
        { 7.0794449e-01, 5.6220129e-01, 8.1363191e-01 },
        { 6.9332914e-01, 5.6655028e-01, 8.2192958e-01 },
        { 6.7793938e-01, 5.7105588e-01, 8.3011175e-01 },
        { 6.6186002e-01, 5.7567059e-01, 8.3799130e-01 },
        { 6.4517587e-01, 5.8034689e-01, 8.4538116e-01 },
        { 6.2797172e-01, 5.8503727e-01, 8.5209422e-01 },
        { 6.1033239e-01, 5.8969422e-01, 8.5794340e-01 },
        { 5.9234266e-01, 5.9427022e-01, 8.6274160e-01 },
        { 5.7406671e-01, 5.9872955e-01, 8.6637254e-01 },
        { 5.5528457e-01, 6.0319153e-01, 8.6965388e-01 },
        { 5.3592795e-01, 6.0767311e-01, 8.7279582e-01 },
        { 5.1601441e-01, 6.1214023e-01, 8.7570328e-01 },
        { 4.9556157e-01, 6.1655882e-01, 8.7828121e-01 },
        { 4.7458700e-01, 6.2089484e-01, 8.8043453e-01 },
        { 4.5310832e-01, 6.2511422e-01, 8.8206819e-01 },
        { 4.3114310e-01, 6.2918291e-01, 8.8308712e-01 },
        { 4.0870517e-01, 6.3306696e-01, 8.8339489e-01 },
        { 3.8538285e-01, 6.3676063e-01, 8.8277383e-01 },
        { 3.6105557e-01, 6.4029974e-01, 8.8122192e-01 },
        { 3.3596153e-01, 6.4371953e-01, 8.7888995e-01 },
        { 3.1033899e-01, 6.4705521e-01, 8.7592869e-01 },
        { 2.8442617e-01, 6.5034203e-01, 8.7248894e-01 },
        { 2.5846129e-01, 6.5361521e-01, 8.6872145e-01 },
        { 2.3268259e-01, 6.5690998e-01, 8.6477703e-01 },
        { 2.0732829e-01, 6.6026158e-01, 8.6080644e-01 },
        { 1.8032999e-01, 6.6385889e-01, 8.5658417e-01 },
        { 1.4933793e-01, 6.6784285e-01, 8.5160924e-01 },
        { 1.1648182e-01, 6.7200870e-01, 8.4594862e-01 },
        { 8.3947374e-02, 6.7614760e-01, 8.3967735e-01 },
        { 5.3920327e-02, 6.8005073e-01, 8.3287045e-01 },
        { 2.8586401e-02, 6.8350926e-01, 8.2560295e-01 },
        { 1.0131323e-02, 6.8631438e-01, 8.1794989e-01 },
        { 7.4081663e-04, 6.8825726e-01, 8.0998628e-01 },
        { 0.0000000e+00, 6.8936394e-01, 8.0158705e-01 },
        { 0.0000000e+00, 6.9033484e-01, 7.9217150e-01 },
        { 0.0000000e+00, 6.9122002e-01, 7.8185222e-01 },
        { 0.0000000e+00, 6.9200255e-01, 7.7080444e-01 },
        { 0.0000000e+00, 6.9266548e-01, 7.5920339e-01 },
        { 0.0000000e+00, 6.9319188e-01, 7.4722429e-01 },
        { 0.0000000e+00, 6.9356481e-01, 7.3504238e-01 },
        { 0.0000000e+00, 6.9376731e-01, 7.2283288e-01 },
        { 1.1697519e-03, 6.9377596e-01, 7.1069204e-01 },
        { 1.2501380e-02, 6.9353968e-01, 6.9811366e-01 },
        { 3.3710848e-02, 6.9308741e-01, 6.8508308e-01 },
        { 6.2121091e-02, 6.9246402e-01, 6.7173476e-01 },
        { 9.5055043e-02, 6.9171435e-01, 6.5820317e-01 },
        { 1.2983564e-01, 6.9088328e-01, 6.4462277e-01 },
        { 1.6378582e-01, 6.9001566e-01, 6.3112802e-01 },
        { 1.9422851e-01, 6.8915636e-01, 6.1785339e-01 },
        { 2.1862736e-01, 6.8834808e-01, 6.0492304e-01 },
        { 2.3989410e-01, 6.8754698e-01, 5.9209358e-01 },
        { 2.6027296e-01, 6.8671078e-01, 5.7925643e-01 },
        { 2.7986659e-01, 6.8582852e-01, 5.6647400e-01 },
        { 2.9877760e-01, 6.8488924e-01, 5.5380868e-01 },
        { 3.1710865e-01, 6.8388200e-01, 5.4132287e-01 },
        { 3.3496236e-01, 6.8279585e-01, 5.2907897e-01 },
        { 3.5244137e-01, 6.8161983e-01, 5.1713938e-01 },
        { 3.6964831e-01, 6.8034299e-01, 5.0556651e-01 },
        { 3.8651598e-01, 6.7890142e-01, 4.9417972e-01 },
        { 4.0292212e-01, 6.7724783e-01, 4.8280190e-01 },
        { 4.1891922e-01, 6.7540803e-01, 4.7155119e-01 },
        { 4.3455996e-01, 6.7340787e-01, 4.6054603e-01 },
        { 4.4989701e-01, 6.7127323e-01, 4.4990487e-01 },
        { 4.6498305e-01, 6.6902996e-01, 4.3974613e-01 },
        { 4.7987076e-01, 6.6670394e-01, 4.3018826e-01 },
        { 4.9461281e-01, 6.6432102e-01, 4.2134969e-01 },
        { 5.0922747e-01, 6.6186076e-01, 4.1304303e-01 },
        { 5.2365282e-01, 6.5922873e-01, 4.0467317e-01 },
        { 5.3788232e-01, 6.5643746e-01, 3.9642863e-01 },
        { 5.5191296e-01, 6.5350500e-01, 3.8853614e-01 },
        { 5.6574175e-01, 6.5044940e-01, 3.8122243e-01 },
        { 5.7936567e-01, 6.4728871e-01, 3.7471421e-01 },
        { 5.9278175e-01, 6.4404098e-01, 3.6923821e-01 },
        { 6.0598697e-01, 6.4072427e-01, 3.6502116e-01 },
        { 6.1899608e-01, 6.3733794e-01, 3.6208382e-01 },
        { 6.3189218e-01, 6.3379717e-01, 3.5943091e-01 },
        { 6.4465314e-01, 6.3010910e-01, 3.5695077e-01 },
        { 6.5723986e-01, 6.2629750e-01, 3.5470640e-01 },
        { 6.6961321e-01, 6.2238614e-01, 3.5276077e-01 },
        { 6.8173407e-01, 6.1839879e-01, 3.5117687e-01 },
        { 6.9356332e-01, 6.1435922e-01, 3.5001768e-01 },
        { 7.0506184e-01, 6.1029121e-01, 3.4934618e-01 },
        { 7.1620568e-01, 6.0621144e-01, 3.4925869e-01 },
        { 7.2720011e-01, 6.0202834e-01, 3.5030477e-01 },
        { 7.3806415e-01, 5.9773423e-01, 3.5250859e-01 },
        { 7.4872316e-01, 5.9336469e-01, 3.5567328e-01 },
        { 7.5910251e-01, 5.8895535e-01, 3.5960197e-01 },
        { 7.6912754e-01, 5.8454180e-01, 3.6409777e-01 },
        { 7.7872363e-01, 5.8015966e-01, 3.6896380e-01 },
        { 7.8781613e-01, 5.7584453e-01, 3.7400319e-01 },
        { 7.9633191e-01, 5.7163147e-01, 3.7902194e-01 },
        { 8.0459017e-01, 5.6741400e-01, 3.8460640e-01 },
        { 8.1277710e-01, 5.6313326e-01, 3.9119813e-01 },
        { 8.2075198e-01, 5.5884819e-01, 3.9863830e-01 },
        { 8.2837409e-01, 5.5461773e-01, 4.0676804e-01 },
        { 8.3550271e-01, 5.5050083e-01, 4.1542851e-01 },
        { 8.4199712e-01, 5.4655642e-01, 4.2446085e-01 },
        { 8.4771660e-01, 5.4284345e-01, 4.3370620e-01 },
        { 8.5252044e-01, 5.3942086e-01, 4.4300572e-01 },
        { 8.5669196e-01, 5.3615760e-01, 4.5254825e-01 },
        { 8.6084911e-01, 5.3280393e-01, 4.6280207e-01 },
        { 8.6485399e-01, 5.2947265e-01, 4.7361966e-01 },
        { 8.6855293e-01, 5.2628387e-01, 4.8484065e-01 },
        { 8.7179231e-01, 5.2335770e-01, 4.9630465e-01 },
        { 8.7441847e-01, 5.2081424e-01, 5.0785127e-01 },
        { 8.7627777e-01, 5.1877361e-01, 5.1932012e-01 },
        { 8.7721655e-01, 5.1735590e-01, 5.3055083e-01 },
        { 8.7721528e-01, 5.1653743e-01, 5.4144050e-01 },
        { 8.7673228e-01, 5.1582681e-01, 5.5215246e-01 },
        { 8.7585992e-01, 5.1518092e-01, 5.6276971e-01 },
        { 8.7464199e-01, 5.1461130e-01, 5.7335795e-01 },
        { 8.7312230e-01, 5.1412947e-01, 5.8398290e-01 },
        { 8.7134463e-01, 5.1374697e-01, 5.9471024e-01 },
        { 8.6935279e-01, 5.1347531e-01, 6.0560570e-01 },
        { 8.6719058e-01, 5.1332604e-01, 6.1673497e-01 },
        { 8.6479132e-01, 5.1335558e-01, 6.2818166e-01 },
        { 8.6116436e-01, 5.1397853e-01, 6.4011646e-01 },
        { 8.5623071e-01, 5.1520532e-01, 6.5244242e-01 },
        { 8.5019859e-01, 5.1692623e-01, 6.6500390e-01 },
        { 8.4327626e-01, 5.1903155e-01, 6.7764525e-01 },
        { 8.3567198e-01, 5.2141155e-01, 6.9021080e-01 },
        { 8.2759397e-01, 5.2395650e-01, 7.0254492e-01 },
        { 8.1925050e-01, 5.2655668e-01, 7.1449195e-01 },
        { 8.1083320e-01, 5.2911133e-01, 7.2590627e-01 },
        { 8.0171976e-01, 5.3195491e-01, 7.3713195e-01 },
        { 7.9159414e-01, 5.3524337e-01, 7.4832405e-01 },
        { 7.8057954e-01, 5.3889307e-01, 7.5937179e-01 },
        { 7.6879919e-01, 5.4282037e-01, 7.7016438e-01 },
        { 7.5637630e-01, 5.4694165e-01, 7.8059104e-01 },
        { 7.4343410e-01, 5.5117325e-01, 7.9054099e-01 },
        { 7.3009579e-01, 5.5543154e-01, 7.9990344e-01 },
        { 7.1648460e-01, 5.5963288e-01, 8.0856762e-01 },
        { 7.0238421e-01, 5.6386316e-01, 8.1684568e-01 },
        { 6.8745902e-01, 5.6827841e-01, 8.2512084e-01 },
        { 6.7179227e-01, 5.7283188e-01, 8.3320806e-01 },
        { 6.5546873e-01, 5.7747607e-01, 8.4092025e-01 },
        { 6.3857323e-01, 5.8216346e-01, 8.4807031e-01 },
        { 6.2119057e-01, 5.8684653e-01, 8.5447116e-01 },
        { 6.0340554e-01, 5.9147778e-01, 8.5993570e-01 },
        { 5.8530294e-01, 5.9600969e-01, 8.6427684e-01 },
        { 5.6686548e-01, 6.0045227e-01, 8.6765321e-01 },
        { 5.4785887e-01, 6.0492588e-01, 8.7089187e-01 },
        { 5.2828458e-01, 6.0940590e-01, 8.7395432e-01 },
        { 5.0816019e-01, 6.1385828e-01, 8.7674549e-01 },
        { 4.8750330e-01, 6.1824895e-01, 8.7917033e-01 },
        { 4.6633151e-01, 6.2254385e-01, 8.8113377e-01 },
        { 4.4466240e-01, 6.2670894e-01, 8.8254075e-01 },
        { 4.2251357e-01, 6.3071015e-01, 8.8329620e-01 },
        { 3.9981159e-01, 6.3451735e-01, 8.8327452e-01 },
        { 3.7607199e-01, 6.3814701e-01, 8.8227524e-01 },
        { 3.5141964e-01, 6.4163575e-01, 8.8040349e-01 },
        { 3.2609276e-01, 6.4501880e-01, 8.7781004e-01 },
        { 3.0032959e-01, 6.4833139e-01, 8.7464567e-01 },
        { 2.7436835e-01, 6.5160875e-01, 8.7106116e-01 },
        { 2.4844728e-01, 6.5488611e-01, 8.6720730e-01 },
        { 2.2280461e-01, 6.5819870e-01, 8.6323486e-01 },
        { 1.9746281e-01, 6.6159690e-01, 8.5926211e-01 },
        { 1.6869359e-01, 6.6536617e-01, 8.5474452e-01 },
        { 1.3672068e-01, 6.6944532e-01, 8.4949525e-01 },
        { 1.0372980e-01, 6.7362550e-01, 8.4358935e-01 },
        { 7.1906673e-02, 6.7769791e-01, 8.3710184e-01 },
        { 4.3437033e-02, 6.8145371e-01, 8.3010775e-01 },
        { 2.0506602e-02, 6.8468408e-01, 8.2268210e-01 },
        { 5.3011057e-03, 6.8718020e-01, 8.1489993e-01 },
        { 0.0000000e+00, 6.8873381e-01, 8.0683574e-01 },
        { 0.0000000e+00, 6.8974886e-01, 7.9806070e-01 },
        { 0.0000000e+00, 6.9068858e-01, 7.8827454e-01 },
        { 0.0000000e+00, 6.9153604e-01, 7.7765247e-01 },
        { 0.0000000e+00, 6.9227428e-01, 7.6636973e-01 },
        { 0.0000000e+00, 6.9288638e-01, 7.5460156e-01 },
        { 0.0000000e+00, 6.9335538e-01, 7.4252316e-01 },
        { 0.0000000e+00, 6.9366434e-01, 7.3030978e-01 },
        { 0.0000000e+00, 6.9379633e-01, 7.1813664e-01 },
        { 4.2136786e-03, 6.9371298e-01, 7.0588521e-01 },
        { 1.9686557e-02, 6.9338777e-01, 6.9311583e-01 },
        { 4.4000992e-02, 6.9286394e-01, 6.7994630e-01 },
        { 7.4479920e-02, 6.9218634e-01, 6.6651109e-01 },
        { 1.0844627e-01, 6.9139984e-01, 6.5294466e-01 },
        { 1.4322299e-01, 6.9054930e-01, 6.3938146e-01 },
        { 1.7613300e-01, 6.8967958e-01, 6.2595597e-01 },
        { 2.0449924e-01, 6.8883554e-01, 6.1280264e-01 },
        { 2.2697152e-01, 6.8804144e-01, 5.9996169e-01 },
        { 2.4788239e-01, 6.8722806e-01, 5.8712185e-01 },
        { 2.6794511e-01, 6.8637532e-01, 5.7429848e-01 },
        { 2.8726233e-01, 6.8547229e-01, 5.6155398e-01 },
        { 3.0593666e-01, 6.8450801e-01, 5.4895074e-01 },
        { 3.2407076e-01, 6.8347152e-01, 5.3655118e-01 },
        { 3.4176725e-01, 6.8235188e-01, 5.2441767e-01 },
        { 3.5912877e-01, 6.8113813e-01, 5.1261264e-01 },
        { 3.7623575e-01, 6.7981173e-01, 5.0116507e-01 },
        { 3.9291863e-01, 6.7828506e-01, 4.8976788e-01 },
        { 4.0916018e-01, 6.7655632e-01, 4.7842521e-01 },
        { 4.2501309e-01, 6.7465138e-01, 4.6725550e-01 },
        { 4.4053002e-01, 6.7259609e-01, 4.5637720e-01 },
        { 4.5576365e-01, 6.7041634e-01, 4.4590873e-01 },
        { 4.7076667e-01, 6.6813797e-01, 4.3596854e-01 },
        { 4.8559175e-01, 6.6578685e-01, 4.2667505e-01 },
        { 5.0029061e-01, 6.6338747e-01, 4.1813728e-01 },
        { 5.1483452e-01, 6.6086195e-01, 4.0980268e-01 },
        { 5.2918441e-01, 6.5816614e-01, 4.0145443e-01 },
        { 5.4333729e-01, 6.5531807e-01, 3.9331928e-01 },
        { 5.5729015e-01, 6.5233580e-01, 3.8562393e-01 },
        { 5.7103999e-01, 6.4923738e-01, 3.7859513e-01 },
        { 5.8458382e-01, 6.4604086e-01, 3.7245958e-01 },
        { 5.9791862e-01, 6.4276429e-01, 3.6744402e-01 },
        { 6.1104142e-01, 6.3942571e-01, 3.6377517e-01 },
        { 6.2400165e-01, 6.3598631e-01, 3.6104041e-01 },
        { 6.3685008e-01, 6.3238570e-01, 3.5844690e-01 },
        { 6.4954824e-01, 6.2864699e-01, 3.5605056e-01 },
        { 6.6205700e-01, 6.2479396e-01, 3.5391436e-01 },
        { 6.7433726e-01, 6.2085036e-01, 3.5210128e-01 },
        { 6.8634987e-01, 6.1683998e-01, 3.5067431e-01 },
        { 6.9805574e-01, 6.1278658e-01, 3.4969643e-01 },
        { 7.0941573e-01, 6.0871394e-01, 3.4923062e-01 },
        { 7.2047230e-01, 6.0460762e-01, 3.4951373e-01 },
        { 7.3142511e-01, 6.0037732e-01, 3.5103132e-01 },
        { 7.4221864e-01, 5.9604979e-01, 3.5363045e-01 },
        { 7.5277824e-01, 5.9166062e-01, 3.5711423e-01 },
        { 7.6302929e-01, 5.8724542e-01, 3.6128580e-01 },
        { 7.7289713e-01, 5.8283981e-01, 3.6594827e-01 },
        { 7.8230714e-01, 5.7847938e-01, 3.7090476e-01 },
        { 7.9118467e-01, 5.7419975e-01, 3.7595841e-01 },
        { 7.9952815e-01, 5.7001017e-01, 3.8105405e-01 },
        { 8.0777549e-01, 5.6576121e-01, 3.8704726e-01 },
        { 8.1589703e-01, 5.6147180e-01, 3.9398627e-01 },
        { 8.2375204e-01, 5.5720088e-01, 4.0171221e-01 },
        { 8.3119982e-01, 5.5300739e-01, 4.1006624e-01 },
        { 8.3809963e-01, 5.4895026e-01, 4.1888950e-01 },
        { 8.4431076e-01, 5.4508845e-01, 4.2802314e-01 },
        { 8.4969249e-01, 5.4148090e-01, 4.3730831e-01 },
        { 8.5412944e-01, 5.3817979e-01, 4.4660186e-01 },
        { 8.5816866e-01, 5.3505375e-01, 4.5619553e-01 },
        { 8.6202800e-01, 5.3204026e-01, 4.6623954e-01 },
        { 8.6563414e-01, 5.2915140e-01, 4.7671316e-01 },
        { 8.6891374e-01, 5.2639927e-01, 4.8759566e-01 },
        { 8.7179346e-01, 5.2379597e-01, 4.9886632e-01 },
        { 8.7419997e-01, 5.2135359e-01, 5.1050442e-01 },
        { 8.7605993e-01, 5.1908424e-01, 5.2248922e-01 },
        { 8.7730000e-01, 5.1700000e-01, 5.3480000e-01 }
    },
    {
        { 1.4300000e-02, 1.4300000e-02, 1.4300000e-02 },
        { 2.4408389e-02, 1.6879212e-02, 1.9714331e-02 },
        { 3.4347461e-02, 1.9440210e-02, 2.5289986e-02 },
        { 4.4085135e-02, 2.1982745e-02, 3.1023019e-02 },
        { 5.3589329e-02, 2.4506568e-02, 3.6909484e-02 },
        { 6.2827962e-02, 2.7011429e-02, 4.2945432e-02 },
        { 7.1768951e-02, 2.9497079e-02, 4.9126919e-02 },
        { 8.0380216e-02, 3.1963269e-02, 5.5449996e-02 },
        { 8.8629673e-02, 3.4409750e-02, 6.1910718e-02 },
        { 9.6485242e-02, 3.6836271e-02, 6.8505137e-02 },
        { 1.0391484e-01, 3.9242584e-02, 7.5229308e-02 },
        { 1.1088639e-01, 4.1628440e-02, 8.2079282e-02 },
        { 1.1736780e-01, 4.3993588e-02, 8.9051114e-02 },
        { 1.2332700e-01, 4.6337781e-02, 9.6140856e-02 },
        { 1.2873190e-01, 4.8660767e-02, 1.0334456e-01 },
        { 1.3355042e-01, 5.0962299e-02, 1.1065829e-01 },
        { 1.3775048e-01, 5.3242126e-02, 1.1807808e-01 },
        { 1.4130000e-01, 5.5500000e-02, 1.2560000e-01 },
        { 1.4449369e-01, 5.7722199e-02, 1.3329697e-01 },
        { 1.4763545e-01, 5.9899075e-02, 1.4123387e-01 },
        { 1.5070702e-01, 6.2036492e-02, 1.4939471e-01 },
        { 1.5369010e-01, 6.4140312e-02, 1.5776349e-01 },
        { 1.5656641e-01, 6.6216397e-02, 1.6632421e-01 },
        { 1.5931768e-01, 6.8270609e-02, 1.7506086e-01 },
        { 1.6192562e-01, 7.0308811e-02, 1.8395745e-01 },
        { 1.6437195e-01, 7.2336865e-02, 1.9299798e-01 },
        { 1.6663839e-01, 7.4360634e-02, 2.0216645e-01 },
        { 1.6870666e-01, 7.6385981e-02, 2.1144686e-01 },
        { 1.7055847e-01, 7.8418766e-02, 2.2082320e-01 },
        { 1.7217554e-01, 8.0464853e-02, 2.3027949e-01 },
        { 1.7353959e-01, 8.2530105e-02, 2.3979971e-01 },
        { 1.7463235e-01, 8.4620383e-02, 2.4936787e-01 },
        { 1.7543552e-01, 8.6741550e-02, 2.5896798e-01 },
        { 1.7593083e-01, 8.8899468e-02, 2.6858402e-01 },
        { 1.7610000e-01, 9.1100000e-02, 2.7820000e-01 },
        { 1.7607989e-01, 9.3289268e-02, 2.8823568e-01 },
        { 1.7602016e-01, 9.5420275e-02, 2.9902596e-01 },
        { 1.7592176e-01, 9.7509208e-02, 3.1042756e-01 },
        { 1.7578559e-01, 9.9572250e-02, 3.2229719e-01 },
        { 1.7561259e-01, 1.0162559e-01, 3.3449158e-01 },
        { 1.7540367e-01, 1.0368541e-01, 3.4686743e-01 },
        { 1.7515977e-01, 1.0576789e-01, 3.5928148e-01 },
        { 1.7488181e-01, 1.0788922e-01, 3.7159042e-01 },
        { 1.7457072e-01, 1.1006559e-01, 3.8365099e-01 },
        { 1.7422741e-01, 1.1231318e-01, 3.9531990e-01 },
        { 1.7385281e-01, 1.1464817e-01, 4.0645387e-01 },
        { 1.7344786e-01, 1.1708675e-01, 4.1690960e-01 },
        { 1.7301346e-01, 1.1964511e-01, 4.2654383e-01 },
        { 1.7255055e-01, 1.2233943e-01, 4.3521327e-01 },
        { 1.7206006e-01, 1.2518589e-01, 4.4277463e-01 },
        { 1.7154290e-01, 1.2820069e-01, 4.4908464e-01 },
        { 1.7100000e-01, 1.3140000e-01, 4.5400000e-01 },
        { 1.7003061e-01, 1.3495750e-01, 4.5809091e-01 },
        { 1.6829000e-01, 1.3900203e-01, 4.6201999e-01 },
        { 1.6586311e-01, 1.4348254e-01, 4.6578264e-01 },
        { 1.6283489e-01, 1.4834798e-01, 4.6937424e-01 },
        { 1.5929030e-01, 1.5354731e-01, 4.7279017e-01 },
        { 1.5531429e-01, 1.5902947e-01, 4.7602582e-01 },
        { 1.5099180e-01, 1.6474342e-01, 4.7907657e-01 },
        { 1.4640779e-01, 1.7063810e-01, 4.8193783e-01 },
        { 1.4164721e-01, 1.7666246e-01, 4.8460496e-01 },
        { 1.3679500e-01, 1.8276546e-01, 4.8707335e-01 },
        { 1.3193613e-01, 1.8889604e-01, 4.8933840e-01 },
        { 1.2715553e-01, 1.9500316e-01, 4.9139548e-01 },
        { 1.2253816e-01, 2.0103576e-01, 4.9323999e-01 },
        { 1.1816897e-01, 2.0694279e-01, 4.9486731e-01 },
        { 1.1413292e-01, 2.1267321e-01, 4.9627283e-01 },
        { 1.1051494e-01, 2.1817596e-01, 4.9745193e-01 },
        { 1.0740000e-01, 2.2340000e-01, 4.9840000e-01 },
        { 1.0466238e-01, 2.2842170e-01, 4.9921990e-01 },
        { 1.0210351e-01, 2.3335849e-01, 5.0001540e-01 },
        { 9.9699005e-02, 2.3822094e-01, 5.0078324e-01 },
        { 9.7424511e-02, 2.4301961e-01, 5.0152014e-01 },
        { 9.5255659e-02, 2.4776504e-01, 5.0222284e-01 },
        { 9.3168082e-02, 2.5246779e-01, 5.0288809e-01 },
        { 9.1137412e-02, 2.5713841e-01, 5.0351262e-01 },
        { 8.9139285e-02, 2.6178747e-01, 5.0409316e-01 },
        { 8.7149331e-02, 2.6642551e-01, 5.0462644e-01 },
        { 8.5143186e-02, 2.7106310e-01, 5.0510921e-01 },
        { 8.3096483e-02, 2.7571078e-01, 5.0553820e-01 },
        { 8.0984854e-02, 2.8037912e-01, 5.0591015e-01 },
        { 7.8783933e-02, 2.8507866e-01, 5.0622179e-01 },
        { 7.6469354e-02, 2.8981996e-01, 5.0646986e-01 },
        { 7.4016750e-02, 2.9461358e-01, 5.0665110e-01 },
        { 7.1401754e-02, 2.9947008e-01, 5.0676223e-01 },
        { 6.8600000e-02, 3.0440000e-01, 5.0680000e-01 },
        { 6.5357202e-02, 3.0940359e-01, 5.0626230e-01 },
        { 6.1504249e-02, 3.1446899e-01, 5.0471435e-01 },
        { 5.7144536e-02, 3.1958859e-01, 5.0225383e-01 },
        { 5.2381458e-02, 3.2475481e-01, 4.9897846e-01 },
        { 4.7318411e-02, 3.2996004e-01, 4.9498593e-01 },
        { 4.2058789e-02, 3.3519668e-01, 4.9037394e-01 },
        { 3.6705987e-02, 3.4045714e-01, 4.8524019e-01 },
        { 3.1363401e-02, 3.4573381e-01, 4.7968238e-01 },
        { 2.6134425e-02, 3.5101911e-01, 4.7379821e-01 },
        { 2.1122454e-02, 3.5630543e-01, 4.6768539e-01 },
        { 1.6430885e-02, 3.6158517e-01, 4.6144161e-01 },
        { 1.2163110e-02, 3.6685073e-01, 4.5516457e-01 },
        { 8.4225268e-03, 3.7209452e-01, 4.4895197e-01 },
        { 5.3125290e-03, 3.7730895e-01, 4.4290151e-01 },
        { 2.9365118e-03, 3.8248640e-01, 4.3711090e-01 },
        { 1.3978705e-03, 3.8761928e-01, 4.3167783e-01 },
        { 8.0000000e-04, 3.9270000e-01, 4.2670000e-01 },
        { 7.0962277e-04, 3.9774240e-01, 4.2200961e-01 },
        { 6.2454641e-04, 4.0276585e-01, 4.1736683e-01 },
        { 5.4479370e-04, 4.0777101e-01, 4.1276131e-01 },
        { 4.7038742e-04, 4.1275856e-01, 4.0818269e-01 },
        { 4.0135038e-04, 4.1772917e-01, 4.0362063e-01 },
        { 3.3770534e-04, 4.2268352e-01, 3.9906477e-01 },
        { 2.7947511e-04, 4.2762227e-01, 3.9450478e-01 },
        { 2.2668247e-04, 4.3254610e-01, 3.8993029e-01 },
        { 1.7935020e-04, 4.3745568e-01, 3.8533095e-01 },
        { 1.3750109e-04, 4.4235168e-01, 3.8069643e-01 },
        { 1.0115793e-04, 4.4723478e-01, 3.7601636e-01 },
        { 7.0343511e-05, 4.5210564e-01, 3.7128041e-01 },
        { 4.5080613e-05, 4.5696495e-01, 3.6647821e-01 },
        { 2.5392025e-05, 4.6181336e-01, 3.6159942e-01 },
        { 1.1300536e-05, 4.6665156e-01, 3.5663369e-01 },
        { 2.8289319e-06, 4.7148022e-01, 3.5157066e-01 },
        { 0.0000000e+00, 4.7630000e-01, 3.4640000e-01 },
        { 0.0000000e+00, 4.8110213e-01, 3.4107160e-01 },
        { 0.0000000e+00, 4.8587930e-01, 3.3555818e-01 },
        { 0.0000000e+00, 4.9063438e-01, 3.2988363e-01 },
        { 0.0000000e+00, 4.9537026e-01, 3.2407185e-01 },
        { 0.0000000e+00, 5.0008983e-01, 3.1814672e-01 },
        { 0.0000000e+00, 5.0479595e-01, 3.1213214e-01 },
        { 0.0000000e+00, 5.0949151e-01, 3.0605199e-01 },
        { 0.0000000e+00, 5.1417939e-01, 2.9993016e-01 },
        { 0.0000000e+00, 5.1886247e-01, 2.9379054e-01 },
        { 0.0000000e+00, 5.2354363e-01, 2.8765703e-01 },
        { 0.0000000e+00, 5.2822575e-01, 2.8155351e-01 },
        { 0.0000000e+00, 5.3291172e-01, 2.7550386e-01 },
        { 0.0000000e+00, 5.3760441e-01, 2.6953199e-01 },
        { 0.0000000e+00, 5.4230670e-01, 2.6366178e-01 },
        { 0.0000000e+00, 5.4702148e-01, 2.5791712e-01 },
        { 0.0000000e+00, 5.5175162e-01, 2.5232189e-01 },
        { 0.0000000e+00, 5.5650000e-01, 2.4690000e-01 },
        { 0.0000000e+00, 5.6130703e-01, 2.4171132e-01 },
        { 0.0000000e+00, 5.6619964e-01, 2.3677163e-01 },
        { 0.0000000e+00, 5.7116055e-01, 2.3203863e-01 },
        { 0.0000000e+00, 5.7617246e-01, 2.2747005e-01 },
        { 0.0000000e+00, 5.8121809e-01, 2.2302359e-01 },
        { 0.0000000e+00, 5.8628012e-01, 2.1865698e-01 },
        { 0.0000000e+00, 5.9134128e-01, 2.1432792e-01 },
        { 0.0000000e+00, 5.9638427e-01, 2.0999413e-01 },
        { 0.0000000e+00, 6.0139179e-01, 2.0561332e-01 },
        { 0.0000000e+00, 6.0634655e-01, 2.0114321e-01 },
        { 0.0000000e+00, 6.1123126e-01, 1.9654152e-01 },
        { 0.0000000e+00, 6.1602863e-01, 1.9176595e-01 },
        { 0.0000000e+00, 6.2072135e-01, 1.8677422e-01 },
        { 0.0000000e+00, 6.2529215e-01, 1.8152405e-01 },
        { 0.0000000e+00, 6.2972372e-01, 1.7597314e-01 },
        { 0.0000000e+00, 6.3399876e-01, 1.7007922e-01 },
        { 0.0000000e+00, 6.3810000e-01, 1.6380000e-01 },
        { 1.5344828e-03, 6.4204184e-01, 1.5642098e-01 },
        { 5.9417673e-03, 6.4585858e-01, 1.4743280e-01 },
        { 1.2927608e-02, 6.4956271e-01, 1.3710089e-01 },
        { 2.2197758e-02, 6.5316674e-01, 1.2569067e-01 },
        { 3.3457971e-02, 6.5668318e-01, 1.1346756e-01 },
        { 4.6414003e-02, 6.6012453e-01, 1.0069700e-01 },
        { 6.0771607e-02, 6.6350330e-01, 8.7644405e-02 },
        { 7.6236537e-02, 6.6683199e-01, 7.4575200e-02 },
        { 9.2514547e-02, 6.7012311e-01, 6.1754811e-02 },
        { 1.0931139e-01, 6.7338915e-01, 4.9448661e-02 },
        { 1.2633282e-01, 6.7664264e-01, 3.7922176e-02 },
        { 1.4328460e-01, 6.7989607e-01, 2.7440781e-02 },
        { 1.5987247e-01, 6.8316194e-01, 1.8269899e-02 },
        { 1.7580219e-01, 6.8645276e-01, 1.0674955e-02 },
        { 1.9077952e-01, 6.8978105e-01, 4.9213741e-03 },
        { 2.0451020e-01, 6.9315929e-01, 1.2745810e-03 },
        { 2.1670000e-01, 6.9660000e-01, 0.0000000e+00 },
        { 2.2773880e-01, 7.0016579e-01, 0.0000000e+00 },
        { 2.3826651e-01, 7.0388902e-01, 0.0000000e+00 },
        { 2.4836395e-01, 7.0773687e-01, 0.0000000e+00 },
        { 2.5811191e-01, 7.1167650e-01, 0.0000000e+00 },
        { 2.6759121e-01, 7.1567505e-01, 0.0000000e+00 },
        { 2.7688265e-01, 7.1969968e-01, 0.0000000e+00 },
        { 2.8606703e-01, 7.2371757e-01, 0.0000000e+00 },
        { 2.9522517e-01, 7.2769586e-01, 0.0000000e+00 },
        { 3.0443786e-01, 7.3160172e-01, 0.0000000e+00 },
        { 3.1378591e-01, 7.3540230e-01, 0.0000000e+00 },
        { 3.2335012e-01, 7.3906477e-01, 0.0000000e+00 },
        { 3.3321131e-01, 7.4255627e-01, 0.0000000e+00 },
        { 3.4345027e-01, 7.4584398e-01, 0.0000000e+00 },
        { 3.5414782e-01, 7.4889505e-01, 0.0000000e+00 },
        { 3.6538475e-01, 7.5167664e-01, 0.0000000e+00 },
        { 3.7724188e-01, 7.5415590e-01, 0.0000000e+00 },
        { 3.8980000e-01, 7.5630000e-01, 0.0000000e+00 },
        { 4.0357383e-01, 7.5819475e-01, 0.0000000e+00 },
        { 4.1888514e-01, 7.5995672e-01, 0.0000000e+00 },
        { 4.3552532e-01, 7.6159921e-01, 0.0000000e+00 },
        { 4.5328577e-01, 7.6313554e-01, 0.0000000e+00 },
        { 4.7195788e-01, 7.6457900e-01, 0.0000000e+00 },
        { 4.9133304e-01, 7.6594291e-01, 0.0000000e+00 },
        { 5.1120265e-01, 7.6724056e-01, 0.0000000e+00 },
        { 5.3135810e-01, 7.6848525e-01, 0.0000000e+00 },
        { 5.5159078e-01, 7.6969030e-01, 0.0000000e+00 },
        { 5.7169210e-01, 7.7086901e-01, 0.0000000e+00 },
        { 5.9145343e-01, 7.7203469e-01, 0.0000000e+00 },
        { 6.1066619e-01, 7.7320063e-01, 0.0000000e+00 },
        { 6.2912175e-01, 7.7438014e-01, 0.0000000e+00 },
        { 6.4661152e-01, 7.7558653e-01, 0.0000000e+00 },
        { 6.6292689e-01, 7.7683310e-01, 0.0000000e+00 },
        { 6.7785925e-01, 7.7813315e-01, 0.0000000e+00 },
        { 6.9120000e-01, 7.7950000e-01, 0.0000000e+00 },
        { 7.0343957e-01, 7.8089157e-01, 3.4306432e-03 },
        { 7.1522450e-01, 7.8226084e-01, 1.3258943e-02 },
        { 7.2658030e-01, 7.8361364e-01, 2.8789456e-02 },
        { 7.3753251e-01, 7.8495583e-01, 4.9326736e-02 },
        { 7.4810667e-01, 7.8629326e-01, 7.4175339e-02 },
        { 7.5832828e-01, 7.8763178e-01, 1.0263982e-01 },
        { 7.6822290e-01, 7.8897724e-01, 1.3402474e-01 },
        { 7.7781603e-01, 7.9033547e-01, 1.6763464e-01 },
        { 7.8713322e-01, 7.9171235e-01, 2.0277410e-01 },
        { 7.9619999e-01, 7.9311370e-01, 2.3874765e-01 },
        { 8.0504186e-01, 7.9454539e-01, 2.7485986e-01 },
        { 8.1368437e-01, 7.9601325e-01, 3.1041528e-01 },
        { 8.2215305e-01, 7.9752315e-01, 3.4471847e-01 },
        { 8.3047343e-01, 7.9908092e-01, 3.7707398e-01 },
        { 8.3867102e-01, 8.0069242e-01, 4.0678636e-01 },
        { 8.4677137e-01, 8.0236350e-01, 4.3316019e-01 },
        { 8.5480000e-01, 8.0410000e-01, 4.5550000e-01 },
        { 8.6304856e-01, 8.0588277e-01, 4.7534556e-01 },
        { 8.7171288e-01, 8.0769335e-01, 4.9470249e-01 },
        { 8.8067477e-01, 8.0953868e-01, 5.1357415e-01 },
        { 8.8981606e-01, 8.1142569e-01, 5.3196388e-01 },
        { 8.9901853e-01, 8.1336129e-01, 5.4987504e-01 },
        { 9.0816401e-01, 8.1535242e-01, 5.6731097e-01 },
        { 9.1713430e-01, 8.1740599e-01, 5.8427503e-01 },
        { 9.2581121e-01, 8.1952894e-01, 6.0077058e-01 },
        { 9.3407655e-01, 8.2172819e-01, 6.1680095e-01 },
        { 9.4181214e-01, 8.2401066e-01, 6.3236950e-01 },
        { 9.4889978e-01, 8.2638328e-01, 6.4747959e-01 },
        { 9.5522127e-01, 8.2885298e-01, 6.6213457e-01 },
        { 9.6065844e-01, 8.3142668e-01, 6.7633778e-01 },
        { 9.6509309e-01, 8.3411130e-01, 6.9009257e-01 },
        { 9.6840702e-01, 8.3691378e-01, 7.0340231e-01 },
        { 9.7048206e-01, 8.3984104e-01, 7.1627033e-01 },
        { 9.7120000e-01, 8.4290000e-01, 7.2870000e-01 },
        { 9.7119959e-01, 8.4616270e-01, 7.4081388e-01 },
        { 9.7119674e-01, 8.4968967e-01, 7.5272198e-01 },
        { 9.7118901e-01, 8.5347063e-01, 7.6440877e-01 },
        { 9.7117395e-01, 8.5749525e-01, 7.7585875e-01 },
        { 9.7114911e-01, 8.6175324e-01, 7.8705641e-01 },
        { 9.7111207e-01, 8.6623428e-01, 7.9798622e-01 },
        { 9.7106037e-01, 8.7092807e-01, 8.0863269e-01 },
        { 9.7099157e-01, 8.7582431e-01, 8.1898028e-01 },
        { 9.7090324e-01, 8.8091269e-01, 8.2901350e-01 },
        { 9.7079292e-01, 8.8618290e-01, 8.3871683e-01 },
        { 9.7065817e-01, 8.9162464e-01, 8.4807474e-01 },
        { 9.7049656e-01, 8.9722760e-01, 8.5707174e-01 },
        { 9.7030564e-01, 9.0298148e-01, 8.6569231e-01 },
        { 9.7008296e-01, 9.0887596e-01, 8.7392094e-01 },
        { 9.6982609e-01, 9.1490075e-01, 8.8174210e-01 },
        { 9.6953259e-01, 9.2104553e-01, 8.8914029e-01 },
        { 9.6920000e-01, 9.2730000e-01, 8.9610000e-01 }
    },
    {
        { 2.2500000e-02, 1.2100000e-02, 1.2100000e-02 },
        { 3.3588316e-02, 1.3326623e-02, 1.4584795e-02 },
        { 4.4557136e-02, 1.4471809e-02, 1.6861755e-02 },
        { 5.5403445e-02, 1.5536475e-02, 1.8938266e-02 },
        { 6.6124229e-02, 1.6521535e-02, 2.0821718e-02 },
        { 7.6716474e-02, 1.7427906e-02, 2.2519499e-02 },
        { 8.7177164e-02, 1.8256503e-02, 2.4038999e-02 },
        { 9.7503286e-02, 1.9008243e-02, 2.5387604e-02 },
        { 1.0769182e-01, 1.9684042e-02, 2.6572705e-02 },
        { 1.1773976e-01, 2.0284816e-02, 2.7601689e-02 },
        { 1.2764409e-01, 2.0811480e-02, 2.8481946e-02 },
        { 1.3740179e-01, 2.1264950e-02, 2.9220863e-02 },
        { 1.4700985e-01, 2.1646143e-02, 2.9825829e-02 },
        { 1.5646525e-01, 2.1955974e-02, 3.0304234e-02 },
        { 1.6576498e-01, 2.2195359e-02, 3.0663464e-02 },
        { 1.7490603e-01, 2.2365215e-02, 3.0910910e-02 },
        { 1.8388537e-01, 2.2466456e-02, 3.1053959e-02 },
        { 1.9270000e-01, 2.2500000e-02, 3.1100000e-02 },
        { 2.0133444e-01, 2.2417830e-02, 3.0789823e-02 },
        { 2.0978486e-01, 2.2181569e-02, 2.9909933e-02 },
        { 2.1806573e-01, 2.1806588e-02, 2.8536291e-02 },
        { 2.2619154e-01, 2.1308261e-02, 2.6744861e-02 },
        { 2.3417674e-01, 2.0701961e-02, 2.4611602e-02 },
        { 2.4203581e-01, 2.0003059e-02, 2.2212477e-02 },
        { 2.4978323e-01, 1.9226928e-02, 1.9623448e-02 },
        { 2.5743346e-01, 1.8388941e-02, 1.6920476e-02 },
        { 2.6500098e-01, 1.7504471e-02, 1.4179524e-02 },
        { 2.7250025e-01, 1.6588889e-02, 1.1476552e-02 },
        { 2.7994576e-01, 1.5657569e-02, 8.8875229e-03 },
        { 2.8735196e-01, 1.4725882e-02, 6.4883981e-03 },
        { 2.9473334e-01, 1.3809203e-02, 4.3551394e-03 },
        { 3.0210436e-01, 1.2922902e-02, 2.5637085e-03 },
        { 3.0947950e-01, 1.2082353e-02, 1.1900672e-03 },
        { 3.1687322e-01, 1.1302928e-02, 3.1017708e-04 },
        { 3.2430000e-01, 1.0600000e-02, 0.0000000e+00 },
        { 3.3171340e-01, 9.9100375e-03, 3.8806329e-05 },
        { 3.3906203e-01, 9.1673926e-03, 1.5339571e-04 },
        { 3.4635292e-01, 8.3842627e-03, 3.4102372e-04 },
        { 3.5359310e-01, 7.5728450e-03, 5.9894595e-04 },
        { 3.6078960e-01, 6.7453369e-03, 9.2441798e-04 },
        { 3.6794945e-01, 5.9139355e-03, 1.3146954e-03 },
        { 3.7507968e-01, 5.0908384e-03, 1.7670338e-03 },
        { 3.8218733e-01, 4.2882426e-03, 2.2786888e-03 },
        { 3.8927942e-01, 3.5183456e-03, 2.8469158e-03 },
        { 3.9636299e-01, 2.7933446e-03, 3.4689707e-03 },
        { 4.0344507e-01, 2.1254370e-03, 4.1421088e-03 },
        { 4.1053269e-01, 1.5268200e-03, 4.8635859e-03 },
        { 4.1763289e-01, 1.0096909e-03, 5.6306574e-03 },
        { 4.2475268e-01, 5.8624710e-04, 6.4405790e-03 },
        { 4.3189911e-01, 2.6868580e-04, 7.2906062e-03 },
        { 4.3907921e-01, 6.9204333e-05, 8.1779947e-03 },
        { 4.4630000e-01, 0.0000000e+00, 9.1000000e-03 },
        { 4.5354805e-01, 0.0000000e+00, 1.0372923e-02 },
        { 4.6080510e-01, 0.0000000e+00, 1.2273345e-02 },
        { 4.6807093e-01, 0.0000000e+00, 1.4738945e-02 },
        { 4.7534535e-01, 0.0000000e+00, 1.7707397e-02 },
        { 4.8262814e-01, 0.0000000e+00, 2.1116380e-02 },
        { 4.8991911e-01, 0.0000000e+00, 2.4903571e-02 },
        { 4.9721803e-01, 0.0000000e+00, 2.9006646e-02 },
        { 5.0452472e-01, 0.0000000e+00, 3.3363283e-02 },
        { 5.1183896e-01, 0.0000000e+00, 3.7911159e-02 },
        { 5.1916054e-01, 0.0000000e+00, 4.2587951e-02 },
        { 5.2648926e-01, 0.0000000e+00, 4.7331335e-02 },
        { 5.3382492e-01, 0.0000000e+00, 5.2078989e-02 },
        { 5.4116730e-01, 0.0000000e+00, 5.6768589e-02 },
        { 5.4851621e-01, 0.0000000e+00, 6.1337814e-02 },
        { 5.5587143e-01, 0.0000000e+00, 6.5724339e-02 },
        { 5.6323276e-01, 0.0000000e+00, 6.9865842e-02 },
        { 5.7060000e-01, 0.0000000e+00, 7.3700000e-02 },
        { 5.7798001e-01, 0.0000000e+00, 7.7371689e-02 },
        { 5.8537819e-01, 0.0000000e+00, 8.1062421e-02 },
        { 5.9279215e-01, 0.0000000e+00, 8.4764833e-02 },
        { 6.0021948e-01, 0.0000000e+00, 8.8471557e-02 },
        { 6.0765779e-01, 0.0000000e+00, 9.2175228e-02 },
        { 6.1510467e-01, 0.0000000e+00, 9.5868481e-02 },
        { 6.2255773e-01, 0.0000000e+00, 9.9543949e-02 },
        { 6.3001457e-01, 0.0000000e+00, 1.0319427e-01 },
        { 6.3747278e-01, 0.0000000e+00, 1.0681207e-01 },
        { 6.4492997e-01, 0.0000000e+00, 1.1038999e-01 },
        { 6.5238373e-01, 0.0000000e+00, 1.1392066e-01 },
        { 6.5983167e-01, 0.0000000e+00, 1.1739672e-01 },
        { 6.6727138e-01, 0.0000000e+00, 1.2081079e-01 },
        { 6.7470047e-01, 0.0000000e+00, 1.2415553e-01 },
        { 6.8211654e-01, 0.0000000e+00, 1.2742355e-01 },
        { 6.8951718e-01, 0.0000000e+00, 1.3060750e-01 },
        { 6.9690000e-01, 0.0000000e+00, 1.3370000e-01 },
        { 7.0446630e-01, 0.0000000e+00, 1.3685050e-01 },
        { 7.1236987e-01, 0.0000000e+00, 1.4018140e-01 },
        { 7.2053705e-01, 0.0000000e+00, 1.4364476e-01 },
        { 7.2889416e-01, 0.0000000e+00, 1.4719265e-01 },
        { 7.3736753e-01, 0.0000000e+00, 1.5077715e-01 },
        { 7.4588349e-01, 0.0000000e+00, 1.5435033e-01 },
        { 7.5436837e-01, 0.0000000e+00, 1.5786426e-01 },
        { 7.6274849e-01, 0.0000000e+00, 1.6127100e-01 },
        { 7.7095019e-01, 0.0000000e+00, 1.6452262e-01 },
        { 7.7889980e-01, 0.0000000e+00, 1.6757121e-01 },
        { 7.8652364e-01, 0.0000000e+00, 1.7036882e-01 },
        { 7.9374804e-01, 0.0000000e+00, 1.7286753e-01 },
        { 8.0049933e-01, 0.0000000e+00, 1.7501941e-01 },
        { 8.0670385e-01, 0.0000000e+00, 1.7677652e-01 },
        { 8.1228791e-01, 0.0000000e+00, 1.7809095e-01 },
        { 8.1717785e-01, 0.0000000e+00, 1.7891475e-01 },
        { 8.2130000e-01, 0.0000000e+00, 1.7920000e-01 },
        { 8.2492769e-01, 0.0000000e+00, 1.7822822e-01 },
        { 8.2838684e-01, 0.0000000e+00, 1.7544969e-01 },
        { 8.3168268e-01, 0.0000000e+00, 1.7106961e-01 },
        { 8.3482043e-01, 0.0000000e+00, 1.6529319e-01 },
        { 8.3780531e-01, 0.0000000e+00, 1.5832562e-01 },
        { 8.4064254e-01, 0.0000000e+00, 1.5037211e-01 },
        { 8.4333733e-01, 0.0000000e+00, 1.4163788e-01 },
        { 8.4589491e-01, 0.0000000e+00, 1.3232811e-01 },
        { 8.4832050e-01, 0.0000000e+00, 1.2264801e-01 },
        { 8.5061932e-01, 0.0000000e+00, 1.1280280e-01 },
        { 8.5279659e-01, 0.0000000e+00, 1.0299766e-01 },
        { 8.5485753e-01, 0.0000000e+00, 9.3437816e-02 },
        { 8.5680737e-01, 0.0000000e+00, 8.4328458e-02 },
        { 8.5865131e-01, 0.0000000e+00, 7.5874793e-02 },
        { 8.6039458e-01, 0.0000000e+00, 6.8282026e-02 },
        { 8.6204240e-01, 0.0000000e+00, 6.1755360e-02 },
        { 8.6360000e-01, 0.0000000e+00, 5.6500000e-02 },
        { 8.6512583e-01, 1.9008022e-03, 5.1904888e-02 },
        { 8.6666570e-01, 7.3490289e-03, 4.7251172e-02 },
        { 8.6820588e-01, 1.5963411e-02, 4.2582361e-02 },
        { 8.6973260e-01, 2.7362677e-02, 3.7941967e-02 },
        { 8.7123211e-01, 4.1165560e-02, 3.3373501e-02 },
        { 8.7269068e-01, 5.6990788e-02, 2.8920471e-02 },
        { 8.7409455e-01, 7.4457093e-02, 2.4626390e-02 },
        { 8.7542998e-01, 9.3183204e-02, 2.0534767e-02 },
        { 8.7668320e-01, 1.1278785e-01, 1.6689112e-02 },
        { 8.7784048e-01, 1.3288977e-01, 1.3132937e-02 },
        { 8.7888806e-01, 1.5310768e-01, 9.9097506e-03 },
        { 8.7981220e-01, 1.7306032e-01, 7.0630647e-03 },
        { 8.8059914e-01, 1.9236642e-01, 4.6363892e-03 },
        { 8.8123514e-01, 2.1064470e-01, 2.6732346e-03 },
        { 8.8170645e-01, 2.2751391e-01, 1.2171112e-03 },
        { 8.8199932e-01, 2.4259277e-01, 3.1152954e-04 },
        { 8.8210000e-01, 2.5550000e-01, 0.0000000e+00 },
        { 8.8204832e-01, 2.6699727e-01, 0.0000000e+00 },
        { 8.8189745e-01, 2.7811844e-01, 0.0000000e+00 },
        { 8.8165367e-01, 2.8889056e-01, 0.0000000e+00 },
        { 8.8132326e-01, 2.9934069e-01, 0.0000000e+00 },
        { 8.8091249e-01, 3.0949587e-01, 0.0000000e+00 },
        { 8.8042765e-01, 3.1938317e-01, 0.0000000e+00 },
        { 8.7987500e-01, 3.2902964e-01, 0.0000000e+00 },
        { 8.7926082e-01, 3.3846232e-01, 0.0000000e+00 },
        { 8.7859139e-01, 3.4770828e-01, 0.0000000e+00 },
        { 8.7787298e-01, 3.5679455e-01, 0.0000000e+00 },
        { 8.7711188e-01, 3.6574821e-01, 0.0000000e+00 },
        { 8.7631435e-01, 3.7459629e-01, 0.0000000e+00 },
        { 8.7548667e-01, 3.8336586e-01, 0.0000000e+00 },
        { 8.7463512e-01, 3.9208396e-01, 0.0000000e+00 },
        { 8.7376597e-01, 4.0077765e-01, 0.0000000e+00 },
        { 8.7288551e-01, 4.0947398e-01, 0.0000000e+00 },
        { 8.7200000e-01, 4.1820000e-01, 0.0000000e+00 },
        { 8.7102998e-01, 4.2689609e-01, 0.0000000e+00 },
        { 8.6990022e-01, 4.3548787e-01, 0.0000000e+00 },
        { 8.6862341e-01, 4.4398026e-01, 0.0000000e+00 },
        { 8.6721221e-01, 4.5237821e-01, 0.0000000e+00 },
        { 8.6567927e-01, 4.6068664e-01, 0.0000000e+00 },
        { 8.6403728e-01, 4.6891048e-01, 0.0000000e+00 },
        { 8.6229889e-01, 4.7705467e-01, 0.0000000e+00 },
        { 8.6047678e-01, 4.8512412e-01, 0.0000000e+00 },
        { 8.5858360e-01, 4.9312378e-01, 0.0000000e+00 },
        { 8.5663203e-01, 5.0105857e-01, 0.0000000e+00 },
        { 8.5463473e-01, 5.0893343e-01, 0.0000000e+00 },
        { 8.5260438e-01, 5.1675328e-01, 0.0000000e+00 },
        { 8.5055362e-01, 5.2452305e-01, 0.0000000e+00 },
        { 8.4849514e-01, 5.3224768e-01, 0.0000000e+00 },
        { 8.4644160e-01, 5.3993209e-01, 0.0000000e+00 },
        { 8.4440566e-01, 5.4758122e-01, 0.0000000e+00 },
        { 8.4240000e-01, 5.5520000e-01, 0.0000000e+00 },
        { 8.4037302e-01, 5.6278132e-01, 0.0000000e+00 },
        { 8.3827007e-01, 5.7031483e-01, 0.0000000e+00 },
        { 8.3609922e-01, 5.7780056e-01, 0.0000000e+00 },
        { 8.3386855e-01, 5.8523853e-01, 0.0000000e+00 },
        { 8.3158613e-01, 5.9262877e-01, 0.0000000e+00 },
        { 8.2926004e-01, 5.9997131e-01, 0.0000000e+00 },
        { 8.2689836e-01, 6.0726619e-01, 0.0000000e+00 },
        { 8.2450915e-01, 6.1451344e-01, 0.0000000e+00 },
        { 8.2210049e-01, 6.2171308e-01, 0.0000000e+00 },
        { 8.1968046e-01, 6.2886514e-01, 0.0000000e+00 },
        { 8.1725713e-01, 6.3596966e-01, 0.0000000e+00 },
        { 8.1483857e-01, 6.4302667e-01, 0.0000000e+00 },
        { 8.1243286e-01, 6.5003618e-01, 0.0000000e+00 },
        { 8.1004807e-01, 6.5699825e-01, 0.0000000e+00 },
        { 8.0769229e-01, 6.6391288e-01, 0.0000000e+00 },
        { 8.0537357e-01, 6.7078012e-01, 0.0000000e+00 },
        { 8.0310000e-01, 6.7760000e-01, 0.0000000e+00 },
        { 8.0073741e-01, 6.8451157e-01, 0.0000000e+00 },
        { 7.9817569e-01, 6.9161972e-01, 0.0000000e+00 },
        { 7.9545904e-01, 6.9887322e-01, 0.0000000e+00 },
        { 7.9263163e-01, 7.0622086e-01, 0.0000000e+00 },
        { 7.8973765e-01, 7.1361141e-01, 0.0000000e+00 },
        { 7.8682128e-01, 7.2099365e-01, 0.0000000e+00 },
        { 7.8392670e-01, 7.2831635e-01, 0.0000000e+00 },
        { 7.8109810e-01, 7.3552830e-01, 0.0000000e+00 },
        { 7.7837967e-01, 7.4257827e-01, 0.0000000e+00 },
        { 7.7581558e-01, 7.4941504e-01, 0.0000000e+00 },
        { 7.7345001e-01, 7.5598738e-01, 0.0000000e+00 },
        { 7.7132716e-01, 7.6224407e-01, 0.0000000e+00 },
        { 7.6949121e-01, 7.6813389e-01, 0.0000000e+00 },
        { 7.6798633e-01, 7.7360561e-01, 0.0000000e+00 },
        { 7.6685671e-01, 7.7860802e-01, 0.0000000e+00 },
        { 7.6614654e-01, 7.8308989e-01, 0.0000000e+00 },
        { 7.6590000e-01, 7.8700000e-01, 0.0000000e+00 },
        { 7.6621926e-01, 7.9045787e-01, 0.0000000e+00 },
        { 7.6714143e-01, 7.9363831e-01, 0.0000000e+00 },
        { 7.6861309e-01, 7.9657301e-01, 0.0000000e+00 },
        { 7.7058082e-01, 7.9929365e-01, 0.0000000e+00 },
        { 7.7299122e-01, 8.0183191e-01, 0.0000000e+00 },
        { 7.7579086e-01, 8.0421949e-01, 0.0000000e+00 },
        { 7.7892633e-01, 8.0648807e-01, 0.0000000e+00 },
        { 7.8234421e-01, 8.0866933e-01, 0.0000000e+00 },
        { 7.8599108e-01, 8.1079498e-01, 0.0000000e+00 },
        { 7.8981354e-01, 8.1289668e-01, 0.0000000e+00 },
        { 7.9375815e-01, 8.1500613e-01, 0.0000000e+00 },
        { 7.9777152e-01, 8.1715502e-01, 0.0000000e+00 },
        { 8.0180022e-01, 8.1937502e-01, 0.0000000e+00 },
        { 8.0579083e-01, 8.2169784e-01, 0.0000000e+00 },
        { 8.0968994e-01, 8.2415515e-01, 0.0000000e+00 },
        { 8.1344414e-01, 8.2677864e-01, 0.0000000e+00 },
        { 8.1700000e-01, 8.2960000e-01, 0.0000000e+00 },
        { 8.2051431e-01, 8.3260085e-01, 2.9609996e-03 },
        { 8.2416911e-01, 8.3573297e-01, 1.1459544e-02 },
        { 8.2794889e-01, 8.3898334e-01, 2.4918953e-02 },
        { 8.3183813e-01, 8.4233891e-01, 4.2762545e-02 },
        { 8.3582132e-01, 8.4578664e-01, 6.4413639e-02 },
        { 8.3988294e-01, 8.4931347e-01, 8.9295554e-02 },
        { 8.4400748e-01, 8.5290638e-01, 1.1683161e-01 },
        { 8.4817942e-01, 8.5655232e-01, 1.4644512e-01 },
        { 8.5238324e-01, 8.6023824e-01, 1.7755941e-01 },
        { 8.5660344e-01, 8.6395110e-01, 2.0959780e-01 },
        { 8.6082450e-01, 8.6767787e-01, 2.4198360e-01 },
        { 8.6503090e-01, 8.7140549e-01, 2.7414014e-01 },
        { 8.6920713e-01, 8.7512093e-01, 3.0549073e-01 },
        { 8.7333766e-01, 8.7881115e-01, 3.3545869e-01 },
        { 8.7740700e-01, 8.8246309e-01, 3.6346734e-01 },
        { 8.8139962e-01, 8.8606372e-01, 3.8894001e-01 },
        { 8.8530000e-01, 8.8960000e-01, 4.1130000e-01 },
        { 8.8914036e-01, 8.9309805e-01, 4.3178063e-01 },
        { 8.9296314e-01, 8.9659269e-01, 4.5199327e-01 },
        { 8.9676820e-01, 9.0008393e-01, 4.7192832e-01 },
        { 9.0055541e-01, 9.0357174e-01, 4.9157620e-01 },
        { 9.0432462e-01, 9.0705613e-01, 5.1092729e-01 },
        { 9.0807569e-01, 9.1053710e-01, 5.2997203e-01 },
        { 9.1180848e-01, 9.1401463e-01, 5.4870080e-01 },
        { 9.1552285e-01, 9.1748872e-01, 5.6710401e-01 },
        { 9.1921865e-01, 9.2095937e-01, 5.8517208e-01 },
        { 9.2289576e-01, 9.2442657e-01, 6.0289540e-01 },
        { 9.2655402e-01, 9.2789032e-01, 6.2026439e-01 },
        { 9.3019330e-01, 9.3135061e-01, 6.3726945e-01 },
        { 9.3381345e-01, 9.3480743e-01, 6.5390098e-01 },
        { 9.3741434e-01, 9.3826079e-01, 6.7014940e-01 },
        { 9.4099582e-01, 9.4171068e-01, 6.8600510e-01 },
        { 9.4455776e-01, 9.4515708e-01, 7.0145850e-01 },
        { 9.4810000e-01, 9.4860000e-01, 7.1650000e-01 }
    },
    {
        { 5.1510000e-01, 4.8200000e-02, 6.6970000e-01 },
        { 5.1589821e-01, 5.6090406e-02, 6.7850700e-01 },
        { 5.1660263e-01, 6.3937811e-02, 6.8723890e-01 },
        { 5.1721911e-01, 7.1741790e-02, 6.9589435e-01 },
        { 5.1775353e-01, 7.9501919e-02, 7.0447203e-01 },
        { 5.1821174e-01, 8.7217773e-02, 7.1297061e-01 },
        { 5.1859961e-01, 9.4888928e-02, 7.2138877e-01 },
        { 5.1892300e-01, 1.0251496e-01, 7.2972516e-01 },
        { 5.1918777e-01, 1.1009544e-01, 7.3797847e-01 },
        { 5.1939978e-01, 1.1762995e-01, 7.4614736e-01 },
        { 5.1956489e-01, 1.2511806e-01, 7.5423051e-01 },
        { 5.1968897e-01, 1.3255934e-01, 7.6222658e-01 },
        { 5.1977788e-01, 1.3995338e-01, 7.7013424e-01 },
        { 5.1983747e-01, 1.4729975e-01, 7.7795218e-01 },
        { 5.1987362e-01, 1.5459802e-01, 7.8567905e-01 },
        { 5.1989218e-01, 1.6184777e-01, 7.9331353e-01 },
        { 5.1989902e-01, 1.6904857e-01, 8.0085429e-01 },
        { 5.1990000e-01, 1.7620000e-01, 8.0830000e-01 },
        { 5.1971935e-01, 1.8332232e-01, 8.1572076e-01 },
        { 5.1919533e-01, 1.9043182e-01, 8.2317257e-01 },
        { 5.1835479e-01, 1.9752213e-01, 8.3063297e-01 },
        { 5.1722460e-01, 2.0458686e-01, 8.3807950e-01 },
        { 5.1583164e-01, 2.1161965e-01, 8.4548968e-01 },
        { 5.1420278e-01, 2.1861411e-01, 8.5284106e-01 },
        { 5.1236488e-01, 2.2556386e-01, 8.6011117e-01 },
        { 5.1034481e-01, 2.3246254e-01, 8.6727754e-01 },
        { 5.0816944e-01, 2.3930377e-01, 8.7431771e-01 },
        { 5.0586565e-01, 2.4608116e-01, 8.8120921e-01 },
        { 5.0346029e-01, 2.5278835e-01, 8.8792958e-01 },
        { 5.0098024e-01, 2.5941895e-01, 8.9445636e-01 },
        { 4.9845236e-01, 2.6596659e-01, 9.0076707e-01 },
        { 4.9590353e-01, 2.7242489e-01, 9.0683926e-01 },
        { 4.9336061e-01, 2.7878748e-01, 9.1265045e-01 },
        { 4.9085048e-01, 2.8504797e-01, 9.1817819e-01 },
        { 4.8840000e-01, 2.9120000e-01, 9.2340000e-01 },
        { 4.8583410e-01, 2.9723401e-01, 9.2856861e-01 },
        { 4.8298305e-01, 3.0315203e-01, 9.3390988e-01 },
        { 4.7988169e-01, 3.0896510e-01, 9.3936110e-01 },
        { 4.7656489e-01, 3.1468423e-01, 9.4485956e-01 },
        { 4.7306749e-01, 3.2032045e-01, 9.5034252e-01 },
        { 4.6942436e-01, 3.2588478e-01, 9.5574728e-01 },
        { 4.6567035e-01, 3.3138824e-01, 9.6101110e-01 },
        { 4.6184032e-01, 3.3684186e-01, 9.6607128e-01 },
        { 4.5796911e-01, 3.4225665e-01, 9.7086509e-01 },
        { 4.5409159e-01, 3.4764364e-01, 9.7532981e-01 },
        { 4.5024261e-01, 3.5301385e-01, 9.7940273e-01 },
        { 4.4645703e-01, 3.5837830e-01, 9.8302112e-01 },
        { 4.4276970e-01, 3.6374802e-01, 9.8612226e-01 },
        { 4.3921547e-01, 3.6913403e-01, 9.8864344e-01 },
        { 4.3582921e-01, 3.7454735e-01, 9.9052194e-01 },
        { 4.3264577e-01, 3.7999900e-01, 9.9169503e-01 },
        { 4.2970000e-01, 3.8550000e-01, 9.9210000e-01 },
        { 4.2695563e-01, 3.9104078e-01, 9.9203343e-01 },
        { 4.2434186e-01, 3.9660128e-01, 9.9183773e-01 },
        { 4.2184177e-01, 4.0217685e-01, 9.9151893e-01 },
        { 4.1943843e-01, 4.0776280e-01, 9.9108304e-01 },
        { 4.1711493e-01, 4.1335446e-01, 9.9053607e-01 },
        { 4.1485434e-01, 4.1894717e-01, 9.8988406e-01 },
        { 4.1263974e-01, 4.2453625e-01, 9.8913302e-01 },
        { 4.1045420e-01, 4.3011704e-01, 9.8828896e-01 },
        { 4.0828080e-01, 4.3568486e-01, 9.8735791e-01 },
        { 4.0610263e-01, 4.4123503e-01, 9.8634588e-01 },
        { 4.0390274e-01, 4.4676290e-01, 9.8525890e-01 },
        { 4.0166424e-01, 4.5226379e-01, 9.8410298e-01 },
        { 3.9937018e-01, 4.5773302e-01, 9.8288414e-01 },
        { 3.9700365e-01, 4.6316593e-01, 9.8160841e-01 },
        { 3.9454773e-01, 4.6855784e-01, 9.8028179e-01 },
        { 3.9198548e-01, 4.7390409e-01, 9.7891032e-01 },
        { 3.8930000e-01, 4.7920000e-01, 9.7750000e-01 },
        { 3.8648582e-01, 4.8445911e-01, 9.7576022e-01 },
        { 3.8355650e-01, 4.8969731e-01, 9.7343316e-01 },
        { 3.8052366e-01, 4.9491348e-01, 9.7057403e-01 },
        { 3.7739892e-01, 5.0010652e-01, 9.6723805e-01 },
        { 3.7419390e-01, 5.0527528e-01, 9.6348043e-01 },
        { 3.7092021e-01, 5.1041866e-01, 9.5935639e-01 },
        { 3.6758947e-01, 5.1553554e-01, 9.5492115e-01 },
        { 3.6421330e-01, 5.2062478e-01, 9.5022992e-01 },
        { 3.6080330e-01, 5.2568528e-01, 9.4533792e-01 },
        { 3.5737111e-01, 5.3071592e-01, 9.4030036e-01 },
        { 3.5392834e-01, 5.3571557e-01, 9.3517246e-01 },
        { 3.5048660e-01, 5.4068311e-01, 9.3000944e-01 },
        { 3.4705751e-01, 5.4561742e-01, 9.2486651e-01 },
        { 3.4365269e-01, 5.5051739e-01, 9.1979888e-01 },
        { 3.4028375e-01, 5.5538189e-01, 9.1486178e-01 },
        { 3.3696232e-01, 5.6020980e-01, 9.1011041e-01 },
        { 3.3370000e-01, 5.6500000e-01, 9.0560000e-01 },
        { 3.3048249e-01, 5.6975377e-01, 9.0124733e-01 },
        { 3.2728549e-01, 5.7447380e-01, 8.9692542e-01 },
        { 3.2410562e-01, 5.7916107e-01, 8.9262386e-01 },
        { 3.2093952e-01, 5.8381658e-01, 8.8833222e-01 },
        { 3.1778382e-01, 5.8844133e-01, 8.8404006e-01 },
        { 3.1463515e-01, 5.9303631e-01, 8.7973698e-01 },
        { 3.1149014e-01, 5.9760250e-01, 8.7541253e-01 },
        { 3.0834542e-01, 6.0214091e-01, 8.7105631e-01 },
        { 3.0519762e-01, 6.0665252e-01, 8.6665787e-01 },
        { 3.0204336e-01, 6.1113833e-01, 8.6220681e-01 },
        { 2.9887930e-01, 6.1559933e-01, 8.5769269e-01 },
        { 2.9570204e-01, 6.2003652e-01, 8.5310509e-01 },
        { 2.9250822e-01, 6.2445089e-01, 8.4843358e-01 },
        { 2.8929448e-01, 6.2884343e-01, 8.4366774e-01 },
        { 2.8605744e-01, 6.3321513e-01, 8.3879715e-01 },
        { 2.8279374e-01, 6.3756699e-01, 8.3381138e-01 },
        { 2.7950000e-01, 6.4190000e-01, 8.2870000e-01 },
        { 2.7598461e-01, 6.4624437e-01, 8.2341928e-01 },
        { 2.7210766e-01, 6.5062184e-01, 8.1794531e-01 },
        { 2.6794330e-01, 6.5502072e-01, 8.1229744e-01 },
        { 2.6356571e-01, 6.5942929e-01, 8.0649502e-01 },
        { 2.5904907e-01, 6.6383587e-01, 8.0055739e-01 },
        { 2.5446753e-01, 6.6822875e-01, 7.9450388e-01 },
        { 2.4989526e-01, 6.7259624e-01, 7.8835385e-01 },
        { 2.4540644e-01, 6.7692662e-01, 7.8212663e-01 },
        { 2.4107524e-01, 6.8120820e-01, 7.7584158e-01 },
        { 2.3697582e-01, 6.8542928e-01, 7.6951802e-01 },
        { 2.3318235e-01, 6.8957816e-01, 7.6317531e-01 },
        { 2.2976900e-01, 6.9364314e-01, 7.5683279e-01 },
        { 2.2680995e-01, 6.9761252e-01, 7.5050980e-01 },
        { 2.2437935e-01, 7.0147460e-01, 7.4422568e-01 },
        { 2.2255138e-01, 7.0521767e-01, 7.3799978e-01 },
        { 2.2140021e-01, 7.0883004e-01, 7.3185144e-01 },
        { 2.2100000e-01, 7.1230000e-01, 7.2580000e-01 },
        { 2.2115886e-01, 7.1563185e-01, 7.1983518e-01 },
        { 2.2161806e-01, 7.1884582e-01, 7.1392667e-01 },
        { 2.2235149e-01, 7.2195409e-01, 7.0806381e-01 },
        { 2.2333306e-01, 7.2496887e-01, 7.0223592e-01 },
        { 2.2453667e-01, 7.2790233e-01, 6.9643233e-01 },
        { 2.2593622e-01, 7.3076667e-01, 6.9064237e-01 },
        { 2.2750563e-01, 7.3357410e-01, 6.8485537e-01 },
        { 2.2921878e-01, 7.3633679e-01, 6.7906065e-01 },
        { 2.3104959e-01, 7.3906695e-01, 6.7324754e-01 },
        { 2.3297195e-01, 7.4177676e-01, 6.6740537e-01 },
        { 2.3495978e-01, 7.4447842e-01, 6.6152346e-01 },
        { 2.3698697e-01, 7.4718412e-01, 6.5559116e-01 },
        { 2.3902743e-01, 7.4990606e-01, 6.4959777e-01 },
        { 2.4105506e-01, 7.5265642e-01, 6.4353263e-01 },
        { 2.4304376e-01, 7.5544740e-01, 6.3738507e-01 },
        { 2.4496744e-01, 7.5829120e-01, 6.3114442e-01 },
        { 2.4680000e-01, 7.6120000e-01, 6.2480000e-01 },
        { 2.4862041e-01, 7.6418134e-01, 6.1832794e-01 },
        { 2.5051993e-01, 7.6722669e-01, 6.1172167e-01 },
        { 2.5249090e-01, 7.7032416e-01, 6.0499644e-01 },
        { 2.5452568e-01, 7.7346185e-01, 5.9816752e-01 },
        { 2.5661659e-01, 7.7662787e-01, 5.9125017e-01 },
        { 2.5875600e-01, 7.7981033e-01, 5.8425966e-01 },
        { 2.6093623e-01, 7.8299732e-01, 5.7721124e-01 },
        { 2.6314964e-01, 7.8617696e-01, 5.7012017e-01 },
        { 2.6538857e-01, 7.8933734e-01, 5.6300173e-01 },
        { 2.6764536e-01, 7.9246658e-01, 5.5587117e-01 },
        { 2.6991236e-01, 7.9555278e-01, 5.4874375e-01 },
        { 2.7218191e-01, 7.9858404e-01, 5.4163474e-01 },
        { 2.7444636e-01, 8.0154847e-01, 5.3455940e-01 },
        { 2.7669805e-01, 8.0443418e-01, 5.2753300e-01 },
        { 2.7892932e-01, 8.0722927e-01, 5.2057079e-01 },
        { 2.8113253e-01, 8.0992184e-01, 5.1368803e-01 },
        { 2.8330000e-01, 8.1250000e-01, 5.0690000e-01 },
        { 2.8544103e-01, 8.1497036e-01, 5.0017313e-01 },
        { 2.8757151e-01, 8.1735384e-01, 4.9346428e-01 },
        { 2.8969370e-01, 8.1966003e-01, 4.8677434e-01 },
        { 2.9180987e-01, 8.2189852e-01, 4.8010421e-01 },
        { 2.9392226e-01, 8.2407890e-01, 4.7345478e-01 },
        { 2.9603315e-01, 8.2621074e-01, 4.6682693e-01 },
        { 2.9814479e-01, 8.2830364e-01, 4.6022155e-01 },
        { 3.0025945e-01, 8.3036719e-01, 4.5363955e-01 },
        { 3.0237938e-01, 8.3241097e-01, 4.4708180e-01 },
        { 3.0450684e-01, 8.3444457e-01, 4.4054920e-01 },
        { 3.0664411e-01, 8.3647758e-01, 4.3404264e-01 },
        { 3.0879343e-01, 8.3851958e-01, 4.2756302e-01 },
        { 3.1095706e-01, 8.4058015e-01, 4.2111121e-01 },
        { 3.1313728e-01, 8.4266890e-01, 4.1468811e-01 },
        { 3.1533633e-01, 8.4479540e-01, 4.0829462e-01 },
        { 3.1755649e-01, 8.4696923e-01, 4.0193162e-01 },
        { 3.1980000e-01, 8.4920000e-01, 3.9560000e-01 },
        { 3.2201574e-01, 8.5150895e-01, 3.8897128e-01 },
        { 3.2416288e-01, 8.5390101e-01, 3.8179781e-01 },
        { 3.2625916e-01, 8.5636123e-01, 3.7420177e-01 },
        { 3.2832233e-01, 8.5887469e-01, 3.6630531e-01 },
        { 3.3037011e-01, 8.6142645e-01, 3.5823062e-01 },
        { 3.3242024e-01, 8.6400158e-01, 3.5009984e-01 },
        { 3.3449047e-01, 8.6658516e-01, 3.4203516e-01 },
        { 3.3659853e-01, 8.6916224e-01, 3.3415873e-01 },
        { 3.3876215e-01, 8.7171789e-01, 3.2659273e-01 },
        { 3.4099907e-01, 8.7423719e-01, 3.1945932e-01 },
        { 3.4332704e-01, 8.7670521e-01, 3.1288066e-01 },
        { 3.4576379e-01, 8.7910700e-01, 3.0697893e-01 },
        { 3.4832705e-01, 8.8142763e-01, 3.0187630e-01 },
        { 3.5103456e-01, 8.8365218e-01, 2.9769492e-01 },
        { 3.5390407e-01, 8.8576572e-01, 2.9455696e-01 },
        { 3.5695330e-01, 8.8775330e-01, 2.9258460e-01 },
        { 3.6020000e-01, 8.8960000e-01, 2.9190000e-01 },
        { 3.6375294e-01, 8.9135657e-01, 2.9195123e-01 },
        { 3.6769295e-01, 8.9308212e-01, 2.9210069e-01 },
        { 3.7199585e-01, 8.9477423e-01, 2.9234201e-01 },
        { 3.7663747e-01, 8.9643047e-01, 2.9266881e-01 },
        { 3.8159362e-01, 8.9804845e-01, 2.9307474e-01 },
        { 3.8684013e-01, 8.9962574e-01, 2.9355342e-01 },
        { 3.9235281e-01, 9.0115993e-01, 2.9409850e-01 },
        { 3.9810749e-01, 9.0264860e-01, 2.9470359e-01 },
        { 4.0407999e-01, 9.0408935e-01, 2.9536235e-01 },
        { 4.1024613e-01, 9.0547974e-01, 2.9606839e-01 },
        { 4.1658173e-01, 9.0681738e-01, 2.9681536e-01 },
        { 4.2306262e-01, 9.0809984e-01, 2.9759688e-01 },
        { 4.2966460e-01, 9.0932471e-01, 2.9840659e-01 },
        { 4.3636351e-01, 9.1048958e-01, 2.9923813e-01 },
        { 4.4313517e-01, 9.1159203e-01, 3.0008512e-01 },
        { 4.4995539e-01, 9.1262964e-01, 3.0094120e-01 },
        { 4.5680000e-01, 9.1360000e-01, 3.0180000e-01 },
        { 4.6393324e-01, 9.1454775e-01, 3.0278211e-01 },
        { 4.7159035e-01, 9.1551270e-01, 3.0399270e-01 },
        { 4.7970368e-01, 9.1648522e-01, 3.0540225e-01 },
        { 4.8820556e-01, 9.1745568e-01, 3.0698124e-01 },
        { 4.9702832e-01, 9.1841444e-01, 3.0870017e-01 },
        { 5.0610430e-01, 9.1935186e-01, 3.1052952e-01 },
        { 5.1536583e-01, 9.2025832e-01, 3.1243979e-01 },
        { 5.2474524e-01, 9.2112417e-01, 3.1440145e-01 },
        { 5.3417488e-01, 9.2193978e-01, 3.1638501e-01 },
        { 5.4358707e-01, 9.2269553e-01, 3.1836094e-01 },
        { 5.5291414e-01, 9.2338176e-01, 3.2029974e-01 },
        { 5.6208844e-01, 9.2398886e-01, 3.2217188e-01 },
        { 5.7104230e-01, 9.2450718e-01, 3.2394787e-01 },
        { 5.7970805e-01, 9.2492709e-01, 3.2559819e-01 },
        { 5.8801803e-01, 9.2523895e-01, 3.2709333e-01 },
        { 5.9590457e-01, 9.2543313e-01, 3.2840377e-01 },
        { 6.0330000e-01, 9.2550000e-01, 3.2950000e-01 },
        { 6.1032426e-01, 9.2550000e-01, 3.3044806e-01 },
        { 6.1715271e-01, 9.2550000e-01, 3.3133737e-01 },
        { 6.2380082e-01, 9.2550000e-01, 3.3217347e-01 },
        { 6.3028405e-01, 9.2550000e-01, 3.3296191e-01 },
        { 6.3661787e-01, 9.2550000e-01, 3.3370824e-01 },
        { 6.4281775e-01, 9.2550000e-01, 3.3441801e-01 },
        { 6.4889916e-01, 9.2550000e-01, 3.3509678e-01 },
        { 6.5487755e-01, 9.2550000e-01, 3.3575008e-01 },
        { 6.6076841e-01, 9.2550000e-01, 3.3638348e-01 },
        { 6.6658720e-01, 9.2550000e-01, 3.3700252e-01 },
        { 6.7234938e-01, 9.2550000e-01, 3.3761276e-01 },
        { 6.7807043e-01, 9.2550000e-01, 3.3821973e-01 },
        { 6.8376581e-01, 9.2550000e-01, 3.3882900e-01 },
        { 6.8945099e-01, 9.2550000e-01, 3.3944611e-01 },
        { 6.9514144e-01, 9.2550000e-01, 3.4007662e-01 },
        { 7.0085262e-01, 9.2550000e-01, 3.4072606e-01 },
        { 7.0660000e-01, 9.2550000e-01, 3.4140000e-01 },
        { 7.1235518e-01, 9.2550000e-01, 3.4208737e-01 },
        { 7.1807926e-01, 9.2550000e-01, 3.4277339e-01 },
        { 7.2377191e-01, 9.2550000e-01, 3.4345807e-01 },
        { 7.2943285e-01, 9.2550000e-01, 3.4414140e-01 },
        { 7.3506176e-01, 9.2550000e-01, 3.4482337e-01 },
        { 7.4065834e-01, 9.2550000e-01, 3.4550399e-01 },
        { 7.4622229e-01, 9.2550000e-01, 3.4618325e-01 },
        { 7.5175330e-01, 9.2550000e-01, 3.4686114e-01 },
        { 7.5725107e-01, 9.2550000e-01, 3.4753766e-01 },
        { 7.6271529e-01, 9.2550000e-01, 3.4821281e-01 },
        { 7.6814566e-01, 9.2550000e-01, 3.4888658e-01 },
        { 7.7354188e-01, 9.2550000e-01, 3.4955896e-01 },
        { 7.7890364e-01, 9.2550000e-01, 3.5022996e-01 },
        { 7.8423063e-01, 9.2550000e-01, 3.5089957e-01 },
        { 7.8952256e-01, 9.2550000e-01, 3.5156778e-01 },
        { 7.9477912e-01, 9.2550000e-01, 3.5223459e-01 },
        { 8.0000000e-01, 9.2550000e-01, 3.5290000e-01 }
    },
    {
        { 4.7060000e-01, 0.0000000e+00, 5.2160000e-01 },
        { 4.7490659e-01, 2.5450855e-04, 5.3320035e-01 },
        { 4.7906383e-01, 9.9999531e-04, 5.4473627e-01 },
        { 4.8305730e-01, 2.2094019e-03, 5.5620710e-01 },
        { 4.8687258e-01, 3.8556700e-03, 5.6761220e-01 },
        { 4.9049528e-01, 5.9117412e-03, 5.7895090e-01 },
        { 4.9391097e-01, 8.3505573e-03, 5.9022257e-01 },
        { 4.9710525e-01, 1.1145060e-02, 6.0142655e-01 },
        { 5.0006371e-01, 1.4268190e-02, 6.1256218e-01 },
        { 5.0277193e-01, 1.7692890e-02, 6.2362882e-01 },
        { 5.0521551e-01, 2.1392102e-02, 6.3462581e-01 },
        { 5.0738003e-01, 2.5338767e-02, 6.4555251e-01 },
        { 5.0925109e-01, 2.9505826e-02, 6.5640826e-01 },
        { 5.1081427e-01, 3.3866222e-02, 6.6719241e-01 },
        { 5.1205516e-01, 3.8392895e-02, 6.7790431e-01 },
        { 5.1295935e-01, 4.3058788e-02, 6.8854331e-01 },
        { 5.1351244e-01, 4.7836843e-02, 6.9910876e-01 },
        { 5.1370000e-01, 5.2700000e-02, 7.0960000e-01 },
        { 5.1360308e-01, 5.8489255e-02, 7.2009765e-01 },
        { 5.1331967e-01, 6.5939772e-02, 7.3066384e-01 },
        { 5.1286082e-01, 7.4865742e-02, 7.4127023e-01 },
        { 5.1223757e-01, 8.5081359e-02, 7.5188845e-01 },
        { 5.1146097e-01, 9.6400816e-02, 7.6249015e-01 },
        { 5.1054206e-01, 1.0863831e-01, 7.7304697e-01 },
        { 5.0949188e-01, 1.2160802e-01, 7.8353055e-01 },
        { 5.0832148e-01, 1.3512415e-01, 7.9391254e-01 },
        { 5.0704190e-01, 1.4900090e-01, 8.0416459e-01 },
        { 5.0566418e-01, 1.6305244e-01, 8.1425832e-01 },
        { 5.0419937e-01, 1.7709299e-01, 8.2416540e-01 },
        { 5.0265851e-01, 1.9093673e-01, 8.3385746e-01 },
        { 5.0105265e-01, 2.0439784e-01, 8.4330614e-01 },
        { 4.9939282e-01, 2.1729054e-01, 8.5248309e-01 },
        { 4.9769007e-01, 2.2942900e-01, 8.6135996e-01 },
        { 4.9595545e-01, 2.4062742e-01, 8.6990838e-01 },
        { 4.9420000e-01, 2.5070000e-01, 8.7810000e-01 },
        { 4.9219823e-01, 2.5998247e-01, 8.8632782e-01 },
        { 4.8974721e-01, 2.6895369e-01, 8.9492874e-01 },
        { 4.8689184e-01, 2.7764253e-01, 9.0379024e-01 },
        { 4.8367701e-01, 2.8607784e-01, 9.1279980e-01 },
        { 4.8014762e-01, 2.9428850e-01, 9.2184488e-01 },
        { 4.7634856e-01, 3.0230336e-01, 9.3081299e-01 },
        { 4.7232472e-01, 3.1015130e-01, 9.3959158e-01 },
        { 4.6812099e-01, 3.1786118e-01, 9.4806815e-01 },
        { 4.6378228e-01, 3.2546186e-01, 9.5613016e-01 },
        { 4.5935346e-01, 3.3298221e-01, 9.6366511e-01 },
        { 4.5487944e-01, 3.4045109e-01, 9.7056046e-01 },
        { 4.5040510e-01, 3.4789737e-01, 9.7670371e-01 },
        { 4.4597535e-01, 3.5534991e-01, 9.8198232e-01 },
        { 4.4163507e-01, 3.6283758e-01, 9.8628377e-01 },
        { 4.3742915e-01, 3.7038924e-01, 9.8949555e-01 },
        { 4.3340250e-01, 3.7803376e-01, 9.9150513e-01 },
        { 4.2960000e-01, 3.8580000e-01, 9.9220000e-01 },
        { 4.2595604e-01, 3.9367858e-01, 9.9196784e-01 },
        { 4.2236892e-01, 4.0162883e-01, 9.9129248e-01 },
        { 4.1882939e-01, 4.0963269e-01, 9.9020556e-01 },
        { 4.1532823e-01, 4.1767209e-01, 9.8873873e-01 },
        { 4.1185618e-01, 4.2572897e-01, 9.8692365e-01 },
        { 4.0840402e-01, 4.3378527e-01, 9.8479196e-01 },
        { 4.0496250e-01, 4.4182292e-01, 9.8237532e-01 },
        { 4.0152238e-01, 4.4982386e-01, 9.7970539e-01 },
        { 3.9807443e-01, 4.5777004e-01, 9.7681380e-01 },
        { 3.9460940e-01, 4.6564337e-01, 9.7373222e-01 },
        { 3.9111806e-01, 4.7342581e-01, 9.7049230e-01 },
        { 3.8759116e-01, 4.8109928e-01, 9.6712568e-01 },
        { 3.8401947e-01, 4.8864573e-01, 9.6366402e-01 },
        { 3.8039375e-01, 4.9604709e-01, 9.6013897e-01 },
        { 3.7670476e-01, 5.0328529e-01, 9.5658218e-01 },
        { 3.7294325e-01, 5.1034229e-01, 9.5302531e-01 },
        { 3.6910000e-01, 5.1720000e-01, 9.4950000e-01 },
        { 3.6517339e-01, 5.2387836e-01, 9.4578589e-01 },
        { 3.6117203e-01, 5.3041662e-01, 9.4166198e-01 },
        { 3.5710196e-01, 5.3682568e-01, 9.3715897e-01 },
        { 3.5296924e-01, 5.4311647e-01, 9.3230755e-01 },
        { 3.4877994e-01, 5.4929990e-01, 9.2713841e-01 },
        { 3.4454012e-01, 5.5538688e-01, 9.2168225e-01 },
        { 3.4025582e-01, 5.6138833e-01, 9.1596977e-01 },
        { 3.3593312e-01, 5.6731517e-01, 9.1003166e-01 },
        { 3.3157806e-01, 5.7317830e-01, 9.0389861e-01 },
        { 3.2719672e-01, 5.7898865e-01, 8.9760133e-01 },
        { 3.2279513e-01, 5.8475713e-01, 8.9117050e-01 },
        { 3.1837938e-01, 5.9049466e-01, 8.8463681e-01 },
        { 3.1395550e-01, 5.9621214e-01, 8.7803098e-01 },
        { 3.0952957e-01, 6.0192051e-01, 8.7138369e-01 },
        { 3.0510763e-01, 6.0763066e-01, 8.6472563e-01 },
        { 3.0069576e-01, 6.1335352e-01, 8.5808750e-01 },
        { 2.9630000e-01, 6.1910000e-01, 8.5150000e-01 },
        { 2.9165313e-01, 6.2488239e-01, 8.4485697e-01 },
        { 2.8654758e-01, 6.3069741e-01, 8.3803932e-01 },
        { 2.8107890e-01, 6.3653267e-01, 8.3105835e-01 },
        { 2.7534265e-01, 6.4237574e-01, 8.2392535e-01 },
        { 2.6943439e-01, 6.4821421e-01, 8.1665161e-01 },
        { 2.6344967e-01, 6.5403569e-01, 8.0924842e-01 },
        { 2.5748404e-01, 6.5982774e-01, 8.0172709e-01 },
        { 2.5163306e-01, 6.6557797e-01, 7.9409890e-01 },
        { 2.4599229e-01, 6.7127396e-01, 7.8637514e-01 },
        { 2.4065727e-01, 6.7690331e-01, 7.7856712e-01 },
        { 2.3572357e-01, 6.8245359e-01, 7.7068611e-01 },
        { 2.3128674e-01, 6.8791240e-01, 7.6274342e-01 },
        { 2.2744232e-01, 6.9326733e-01, 7.5475035e-01 },
        { 2.2428589e-01, 6.9850596e-01, 7.4671817e-01 },
        { 2.2191299e-01, 7.0361589e-01, 7.3865819e-01 },
        { 2.2041917e-01, 7.0858471e-01, 7.3058171e-01 },
        { 2.1990000e-01, 7.1340000e-01, 7.2250000e-01 },
        { 2.2019710e-01, 7.1809266e-01, 7.1436170e-01 },
        { 2.2105253e-01, 7.2270338e-01, 7.0611410e-01 },
        { 2.2241249e-01, 7.2723448e-01, 6.9776648e-01 },
        { 2.2422318e-01, 7.3168826e-01, 6.8932813e-01 },
        { 2.2643080e-01, 7.3606702e-01, 6.8080835e-01 },
        { 2.2898156e-01, 7.4037307e-01, 6.7221641e-01 },
        { 2.3182164e-01, 7.4460872e-01, 6.6356162e-01 },
        { 2.3489727e-01, 7.4877626e-01, 6.5485326e-01 },
        { 2.3815463e-01, 7.5287801e-01, 6.4610061e-01 },
        { 2.4153992e-01, 7.5691627e-01, 6.3731297e-01 },
        { 2.4499935e-01, 7.6089335e-01, 6.2849963e-01 },
        { 2.4847911e-01, 7.6481154e-01, 6.1966988e-01 },
        { 2.5192541e-01, 7.6867317e-01, 6.1083300e-01 },
        { 2.5528445e-01, 7.7248052e-01, 6.0199828e-01 },
        { 2.5850243e-01, 7.7623590e-01, 5.9317501e-01 },
        { 2.6152554e-01, 7.7994163e-01, 5.8437249e-01 },
        { 2.6430000e-01, 7.8360000e-01, 5.7560000e-01 },
        { 2.6692287e-01, 7.8719494e-01, 5.6681235e-01 },
        { 2.6952974e-01, 7.9071324e-01, 5.5796478e-01 },
        { 2.7212455e-01, 7.9416155e-01, 5.4906720e-01 },
        { 2.7471126e-01, 7.9754647e-01, 5.4012951e-01 },
        { 2.7729382e-01, 8.0087466e-01, 5.3116164e-01 },
        { 2.7987619e-01, 8.0415273e-01, 5.2217348e-01 },
        { 2.8246232e-01, 8.0738732e-01, 5.1317496e-01 },
        { 2.8505616e-01, 8.1058505e-01, 5.0417598e-01 },
        { 2.8766167e-01, 8.1375256e-01, 4.9518645e-01 },
        { 2.9028279e-01, 8.1689647e-01, 4.8621629e-01 },
        { 2.9292348e-01, 8.2002341e-01, 4.7727541e-01 },
        { 2.9558770e-01, 8.2314001e-01, 4.6837372e-01 },
        { 2.9827939e-01, 8.2625291e-01, 4.5952113e-01 },
        { 3.0100252e-01, 8.2936873e-01, 4.5072755e-01 },
        { 3.0376103e-01, 8.3249409e-01, 4.4200290e-01 },
        { 3.0655887e-01, 8.3563564e-01, 4.3335707e-01 },
        { 3.0940000e-01, 8.3880000e-01, 4.2480000e-01 },
        { 3.1220664e-01, 8.4201987e-01, 4.1587594e-01 },
        { 3.1491640e-01, 8.4531210e-01, 4.0623112e-01 },
        { 3.1755626e-01, 8.4865951e-01, 3.9602839e-01 },
        { 3.2015324e-01, 8.5204493e-01, 3.8543057e-01 },
        { 3.2273435e-01, 8.5545118e-01, 3.7460052e-01 },
        { 3.2532660e-01, 8.5886109e-01, 3.6370108e-01 },
        { 3.2795698e-01, 8.6225748e-01, 3.5289509e-01 },
        { 3.3065251e-01, 8.6562319e-01, 3.4234538e-01 },
        { 3.3344020e-01, 8.6894104e-01, 3.3221481e-01 },
        { 3.3634704e-01, 8.7219385e-01, 3.2266622e-01 },
        { 3.3940005e-01, 8.7536446e-01, 3.1386243e-01 },
        { 3.4262624e-01, 8.7843568e-01, 3.0596631e-01 },
        { 3.4605260e-01, 8.8139035e-01, 2.9914068e-01 },
        { 3.4970615e-01, 8.8421129e-01, 2.9354840e-01 },
        { 3.5361390e-01, 8.8688133e-01, 2.8935229e-01 },
        { 3.5780285e-01, 8.8938329e-01, 2.8671521e-01 },
        { 3.6230000e-01, 8.9170000e-01, 2.8580000e-01 },
        { 3.6748365e-01, 8.9393187e-01, 2.8599256e-01 },
        { 3.7365113e-01, 8.9618011e-01, 2.8654625e-01 },
        { 3.8070798e-01, 8.9842873e-01, 2.8742504e-01 },
        { 3.8855977e-01, 9.0066175e-01, 2.8859294e-01 },
        { 3.9711204e-01, 9.0286318e-01, 2.9001394e-01 },
        { 4.0627037e-01, 9.0501706e-01, 2.9165202e-01 },
        { 4.1594029e-01, 9.0710740e-01, 2.9347117e-01 },
        { 4.2602738e-01, 9.0911821e-01, 2.9543539e-01 },
        { 4.3643718e-01, 9.1103351e-01, 2.9750868e-01 },
        { 4.4707525e-01, 9.1283733e-01, 2.9965501e-01 },
        { 4.5784714e-01, 9.1451368e-01, 3.0183838e-01 },
        { 4.6865842e-01, 9.1604658e-01, 3.0402279e-01 },
        { 4.7941464e-01, 9.1742006e-01, 3.0617222e-01 },
        { 4.9002135e-01, 9.1861811e-01, 3.0825066e-01 },
        { 5.0038411e-01, 9.1962478e-01, 3.1022212e-01 },
        { 5.1040847e-01, 9.2042407e-01, 3.1205056e-01 },
        { 5.2000000e-01, 9.2100000e-01, 3.1370000e-01 },
        { 5.2942585e-01, 9.2145141e-01, 3.1526035e-01 },
        { 5.3899658e-01, 9.2188679e-01, 3.1684122e-01 },
        { 5.4868286e-01, 9.2230469e-01, 3.1843612e-01 },
        { 5.5845538e-01, 9.2270364e-01, 3.2003855e-01 },
        { 5.6828478e-01, 9.2308219e-01, 3.2164201e-01 },
        { 5.7814176e-01, 9.2343885e-01, 3.2324001e-01 },
        { 5.8799698e-01, 9.2377218e-01, 3.2482605e-01 },
        { 5.9782111e-01, 9.2408071e-01, 3.2639364e-01 },
        { 6.0758482e-01, 9.2436298e-01, 3.2793629e-01 },
        { 6.1725879e-01, 9.2461752e-01, 3.2944750e-01 },
        { 6.2681368e-01, 9.2484287e-01, 3.3092077e-01 },
        { 6.3622016e-01, 9.2503756e-01, 3.3234961e-01 },
        { 6.4544891e-01, 9.2520014e-01, 3.3372752e-01 },
        { 6.5447060e-01, 9.2532913e-01, 3.3504801e-01 },
        { 6.6325590e-01, 9.2542308e-01, 3.3630458e-01 },
        { 6.7177547e-01, 9.2548053e-01, 3.3749074e-01 },
        { 6.8000000e-01, 9.2550000e-01, 3.3860000e-01 },
        { 6.8805337e-01, 9.2550000e-01, 3.3965609e-01 },
        { 6.9606766e-01, 9.2550000e-01, 3.4068688e-01 },
        { 7.0402581e-01, 9.2550000e-01, 3.4169205e-01 },
        { 7.1191080e-01, 9.2550000e-01, 3.4267129e-01 },
        { 7.1970556e-01, 9.2550000e-01, 3.4362430e-01 },
        { 7.2739304e-01, 9.2550000e-01, 3.4455075e-01 },
        { 7.3495621e-01, 9.2550000e-01, 3.4545034e-01 },
        { 7.4237802e-01, 9.2550000e-01, 3.4632275e-01 },
        { 7.4964141e-01, 9.2550000e-01, 3.4716768e-01 },
        { 7.5672934e-01, 9.2550000e-01, 3.4798480e-01 },
        { 7.6362476e-01, 9.2550000e-01, 3.4877381e-01 },
        { 7.7031063e-01, 9.2550000e-01, 3.4953439e-01 },
        { 7.7676990e-01, 9.2550000e-01, 3.5026624e-01 },
        { 7.8298552e-01, 9.2550000e-01, 3.5096903e-01 },
        { 7.8894044e-01, 9.2550000e-01, 3.5164247e-01 },
        { 7.9461761e-01, 9.2550000e-01, 3.5228623e-01 },
        { 8.0000000e-01, 9.2550000e-01, 3.5290000e-01 },
        { 8.0509091e-01, 9.2507424e-01, 3.5348354e-01 },
        { 8.0992380e-01, 9.2384233e-01, 3.5403863e-01 },
        { 8.1452682e-01, 9.2187232e-01, 3.5456798e-01 },
        { 8.1892812e-01, 9.1923225e-01, 3.5507432e-01 },
        { 8.2315587e-01, 9.1599019e-01, 3.5556037e-01 },
        { 8.2723822e-01, 9.1221418e-01, 3.5602884e-01 },
        { 8.3120332e-01, 9.0797227e-01, 3.5648245e-01 },
        { 8.3507933e-01, 9.0333252e-01, 3.5692393e-01 },
        { 8.3889441e-01, 8.9836297e-01, 3.5735600e-01 },
        { 8.4267670e-01, 8.9313167e-01, 3.5778137e-01 },
        { 8.4645437e-01, 8.8770668e-01, 3.5820277e-01 },
        { 8.5025558e-01, 8.8215604e-01, 3.5862291e-01 },
        { 8.5410847e-01, 8.7654782e-01, 3.5904452e-01 },
        { 8.5804120e-01, 8.7095005e-01, 3.5947031e-01 },
        { 8.6208193e-01, 8.6543079e-01, 3.5990301e-01 },
        { 8.6625881e-01, 8.6005809e-01, 3.6034533e-01 },
        { 8.7060000e-01, 8.5490000e-01, 3.6080000e-01 },
        { 8.7520770e-01, 8.4978432e-01, 3.6128681e-01 },
        { 8.8012629e-01, 8.4449564e-01, 3.6181746e-01 },
        { 8.8529724e-01, 8.3903718e-01, 3.6238246e-01 },
        { 8.9066199e-01, 8.3341218e-01, 3.6297236e-01 },
        { 8.9616199e-01, 8.2762385e-01, 3.6357769e-01 },
        { 9.0173870e-01, 8.2167544e-01, 3.6418898e-01 },
        { 9.0733357e-01, 8.1557016e-01, 3.6479678e-01 },
        { 9.1288804e-01, 8.0931124e-01, 3.6539161e-01 },
        { 9.1834356e-01, 8.0290192e-01, 3.6596402e-01 },
        { 9.2364160e-01, 7.9634541e-01, 3.6650453e-01 },
        { 9.2872360e-01, 7.8964496e-01, 3.6700368e-01 },
        { 9.3353101e-01, 7.8280378e-01, 3.6745201e-01 },
        { 9.3800528e-01, 7.7582510e-01, 3.6784004e-01 },
        { 9.4208786e-01, 7.6871216e-01, 3.6815833e-01 },
        { 9.4572021e-01, 7.6146817e-01, 3.6839739e-01 },
        { 9.4884377e-01, 7.5409638e-01, 3.6854777e-01 },
        { 9.5140000e-01, 7.4660000e-01, 3.6860000e-01 },
        { 9.5364611e-01, 7.3892555e-01, 3.6855317e-01 },
        { 9.5586868e-01, 7.3102117e-01, 3.6841425e-01 },
        { 9.5805316e-01, 7.2289249e-01, 3.6818563e-01 },
        { 9.6018502e-01, 7.1454515e-01, 3.6786969e-01 },
        { 9.6224974e-01, 7.0598480e-01, 3.6746882e-01 },
        { 9.6423279e-01, 6.9721709e-01, 3.6698539e-01 },
        { 9.6611963e-01, 6.8824765e-01, 3.6642178e-01 },
        { 9.6789574e-01, 6.7908214e-01, 3.6578038e-01 },
        { 9.6954658e-01, 6.6972618e-01, 3.6506357e-01 },
        { 9.7105762e-01, 6.6018543e-01, 3.6427372e-01 },
        { 9.7241433e-01, 6.5046553e-01, 3.6341323e-01 },
        { 9.7360219e-01, 6.4057213e-01, 3.6248447e-01 },
        { 9.7460665e-01, 6.3051086e-01, 3.6148982e-01 },
        { 9.7541319e-01, 6.2028737e-01, 3.6043167e-01 },
        { 9.7600729e-01, 6.0990730e-01, 3.5931240e-01 },
        { 9.7637440e-01, 5.9937629e-01, 3.5813438e-01 },
        { 9.7650000e-01, 5.8870000e-01, 3.5690000e-01 }
    },
    {
        { 1.0000000e+00, 5.3950000e-01, 1.0000000e+00 },
        { 1.0000000e+00, 5.3126068e-01, 1.0000000e+00 },
        { 1.0000000e+00, 5.2299691e-01, 1.0000000e+00 },
        { 1.0000000e+00, 5.1470879e-01, 1.0000000e+00 },
        { 1.0000000e+00, 5.0639643e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.9806172e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.8970413e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.8131941e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.7290332e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.6445797e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.5598560e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.4747773e-01, 1.0000000e+00 },
        { 1.0000000e+00, 4.3892580e-01, 1.0000000e+00 },
        { 9.9849343e-01, 4.3032928e-01, 1.0000000e+00 },
        { 9.9346826e-01, 4.2169342e-01, 1.0000000e+00 },
        { 9.8582020e-01, 4.1301023e-01, 1.0000000e+00 },
        { 9.7649002e-01, 4.0427150e-01, 1.0000000e+00 },
        { 9.6231887e-01, 3.9547721e-01, 1.0000000e+00 },
        { 9.3843265e-01, 3.8663339e-01, 1.0000000e+00 },
        { 9.0986193e-01, 3.7772833e-01, 1.0000000e+00 },
        { 8.8189052e-01, 3.6874988e-01, 1.0000000e+00 },
        { 8.5734643e-01, 3.5969582e-01, 1.0000000e+00 },
        { 8.3300966e-01, 3.5057551e-01, 1.0000000e+00 },
        { 8.0868136e-01, 3.4137428e-01, 1.0000000e+00 },
        { 7.8435248e-01, 3.3207646e-01, 1.0000000e+00 },
        { 7.6002021e-01, 3.2267448e-01, 1.0000000e+00 },
        { 7.3569268e-01, 3.1317791e-01, 1.0000000e+00 },
        { 7.1136186e-01, 3.0357798e-01, 1.0000000e+00 },
        { 6.8701869e-01, 2.9386468e-01, 1.0000000e+00 },
        { 6.6266855e-01, 2.8403786e-01, 1.0000000e+00 },
        { 6.3833257e-01, 2.7411009e-01, 1.0000000e+00 },
        { 6.1397011e-01, 2.6405398e-01, 1.0000000e+00 },
        { 5.8953628e-01, 2.5383931e-01, 1.0000000e+00 },
        { 5.6501862e-01, 2.4345494e-01, 1.0000000e+00 },
        { 5.4047810e-01, 2.3293294e-01, 1.0000000e+00 },
        { 5.1584018e-01, 2.2222595e-01, 1.0000000e+00 },
        { 4.9101700e-01, 2.1127882e-01, 1.0000000e+00 },
        { 4.6599323e-01, 2.0007192e-01, 1.0000000e+00 },
        { 4.4093621e-01, 1.8867946e-01, 1.0000000e+00 },
        { 4.1562033e-01, 1.7698878e-01, 1.0000000e+00 },
        { 3.8977353e-01, 1.6486480e-01, 1.0000000e+00 },
        { 3.6335549e-01, 1.5227964e-01, 1.0000000e+00 },
        { 3.3702787e-01, 1.3952150e-01, 1.0000000e+00 },
        { 3.0988421e-01, 1.2614040e-01, 1.0000000e+00 },
        { 2.8079406e-01, 1.1158191e-01, 1.0000000e+00 },
        { 2.4812041e-01, 9.4140350e-02, 1.0000000e+00 },
        { 2.0943679e-01, 6.9026036e-02, 1.0000000e+00 },
        { 1.6839276e-01, 4.3158013e-02, 1.0000000e+00 },
        { 1.2950927e-01, 2.5273234e-02, 1.0000000e+00 },
        { 9.5686694e-02, 2.3546970e-02, 9.9962176e-01 },
        { 5.9580566e-02, 3.6669498e-02, 9.9654215e-01 },
        { 2.6386687e-02, 5.7114779e-02, 9.9215920e-01 },
        { 4.3865053e-03, 7.7476791e-02, 9.8866252e-01 },
        { 2.3708758e-03, 9.0993866e-02, 9.8820137e-01 },
        { 2.6578639e-02, 9.9907409e-02, 9.9116431e-01 },
        { 6.5405733e-02, 1.0768246e-01, 9.9553485e-01 },
        { 1.0445442e-01, 1.1670819e-01, 9.9912338e-01 },
        { 1.3066448e-01, 1.2963914e-01, 1.0000000e+00 },
        { 1.5102794e-01, 1.5036996e-01, 1.0000000e+00 },
        { 1.6900383e-01, 1.7627491e-01, 1.0000000e+00 },
        { 1.8370171e-01, 2.0318066e-01, 1.0000000e+00 },
        { 1.9439855e-01, 2.2715572e-01, 1.0000000e+00 },
        { 2.0319684e-01, 2.4938135e-01, 1.0000000e+00 },
        { 2.1060104e-01, 2.7112135e-01, 1.0000000e+00 },
        { 2.1610163e-01, 2.9249254e-01, 1.0000000e+00 },
        { 2.1926344e-01, 3.1360685e-01, 1.0000000e+00 },
        { 2.2144552e-01, 3.3443841e-01, 1.0000000e+00 },
        { 2.2318696e-01, 3.5496583e-01, 1.0000000e+00 },
        { 2.2429628e-01, 3.7520845e-01, 1.0000000e+00 },
        { 2.2457610e-01, 3.9518475e-01, 1.0000000e+00 },
        { 2.2375811e-01, 4.1488493e-01, 1.0000000e+00 },
        { 2.2206052e-01, 4.3431960e-01, 1.0000000e+00 },
        { 2.1987199e-01, 4.5351569e-01, 1.0000000e+00 },
        { 2.1756867e-01, 4.7249945e-01, 1.0000000e+00 },
        { 2.1474905e-01, 4.9125723e-01, 1.0000000e+00 },
        { 2.1125055e-01, 5.0979425e-01, 1.0000000e+00 },
        { 2.0734849e-01, 5.2814047e-01, 1.0000000e+00 },
        { 2.0331553e-01, 5.4632553e-01, 1.0000000e+00 },
        { 1.9904874e-01, 5.6433972e-01, 1.0000000e+00 },
        { 1.9442413e-01, 5.8218037e-01, 1.0000000e+00 },
        { 1.8953022e-01, 5.9987067e-01, 1.0000000e+00 },
        { 1.8445537e-01, 6.1743373e-01, 1.0000000e+00 },
        { 1.7918183e-01, 6.3486004e-01, 1.0000000e+00 },
        { 1.7364807e-01, 6.5214668e-01, 1.0000000e+00 },
        { 1.6785412e-01, 6.6931841e-01, 1.0000000e+00 },
        { 1.6180000e-01, 6.8640000e-01, 1.0000000e+00 },
        { 1.5549800e-01, 7.0338331e-01, 1.0000000e+00 },
        { 1.4890923e-01, 7.2026103e-01, 1.0000000e+00 },
        { 1.4195693e-01, 7.3705911e-01, 1.0000000e+00 },
        { 1.3456432e-01, 7.5380354e-01, 1.0000000e+00 },
        { 1.2682025e-01, 7.7049709e-01, 1.0000000e+00 },
        { 1.1867722e-01, 7.8712673e-01, 1.0000000e+00 },
        { 1.0979153e-01, 8.0370091e-01, 1.0000000e+00 },
        { 9.9818987e-02, 8.2022812e-01, 1.0000000e+00 },
        { 8.7568023e-02, 8.3666906e-01, 1.0000000e+00 },
        { 7.2864866e-02, 8.5301829e-01, 1.0000000e+00 },
        { 5.7641240e-02, 8.6937378e-01, 1.0000000e+00 },
        { 4.3841720e-02, 8.8583420e-01, 1.0000000e+00 },
        { 3.0878015e-02, 9.0392758e-01, 9.9975435e-01 },
        { 1.7151172e-02, 9.2392851e-01, 9.9888493e-01 },
        { 5.8327139e-03, 9.4208113e-01, 9.9745104e-01 },
        { 1.6894745e-04, 9.5457662e-01, 9.9551900e-01 },
        { 0.0000000e+00, 9.6108626e-01, 9.9013857e-01 },
        { 0.0000000e+00, 9.6651257e-01, 9.7699408e-01 },
        { 0.0000000e+00, 9.7045065e-01, 9.5897305e-01 },
        { 0.0000000e+00, 9.7232245e-01, 9.3914820e-01 },
        { 0.0000000e+00, 9.7000878e-01, 9.1767991e-01 },
        { 0.0000000e+00, 9.6133609e-01, 8.9079911e-01 },
        { 0.0000000e+00, 9.4944811e-01, 8.6128515e-01 },
        { 0.0000000e+00, 9.3767601e-01, 8.3220173e-01 },
        { 0.0000000e+00, 9.2799649e-01, 8.0534942e-01 },
        { 0.0000000e+00, 9.1837871e-01, 7.7884051e-01 },
        { 0.0000000e+00, 9.0866708e-01, 7.5258584e-01 },
        { 0.0000000e+00, 8.9887766e-01, 7.2665768e-01 },
        { 0.0000000e+00, 8.8901541e-01, 7.0106625e-01 },
        { 0.0000000e+00, 8.7906204e-01, 6.7571025e-01 },
        { 0.0000000e+00, 8.6903421e-01, 6.5070585e-01 },
        { 0.0000000e+00, 8.5895144e-01, 6.2618583e-01 },
        { 0.0000000e+00, 8.4882308e-01, 6.0219709e-01 },
        { 0.0000000e+00, 8.3862937e-01, 5.7856274e-01 },
        { 0.0000000e+00, 8.2838516e-01, 5.5542806e-01 },
        { 0.0000000e+00, 8.1810912e-01, 5.3297197e-01 },
        { 0.0000000e+00, 8.0781515e-01, 5.1126235e-01 },
        { 0.0000000e+00, 7.9749240e-01, 4.9001342e-01 },
        { 0.0000000e+00, 7.8713901e-01, 4.6944028e-01 },
        { 0.0000000e+00, 7.7675525e-01, 4.4982364e-01 },
        { 0.0000000e+00, 7.6633906e-01, 4.3130329e-01 },
        { 0.0000000e+00, 7.5588327e-01, 4.1342051e-01 },
        { 0.0000000e+00, 7.4539918e-01, 3.9641503e-01 },
        { 0.0000000e+00, 7.3490051e-01, 3.8064597e-01 },
        { 0.0000000e+00, 7.2439985e-01, 3.6631949e-01 },
        { 0.0000000e+00, 7.1389510e-01, 3.5279221e-01 },
        { 0.0000000e+00, 7.0337699e-01, 3.4026060e-01 },
        { 0.0000000e+00, 6.9283656e-01, 3.2910727e-01 },
        { 0.0000000e+00, 6.8227000e-01, 3.1960747e-01 },
        { 0.0000000e+00, 6.7170092e-01, 3.1115069e-01 },
        { 0.0000000e+00, 6.6110450e-01, 3.0358848e-01 },
        { 0.0000000e+00, 6.5044539e-01, 2.9694370e-01 },
        { 0.0000000e+00, 6.3970123e-01, 2.9127695e-01 },
        { 0.0000000e+00, 6.2894625e-01, 2.8674656e-01 },
        { 0.0000000e+00, 6.1811366e-01, 2.8274569e-01 },
        { 0.0000000e+00, 6.0709784e-01, 2.7851800e-01 },
        { 0.0000000e+00, 5.9579925e-01, 2.7335099e-01 },
        { 0.0000000e+00, 5.8422909e-01, 2.6729836e-01 },
        { 0.0000000e+00, 5.7242367e-01, 2.6052772e-01 },
        { 0.0000000e+00, 5.6040435e-01, 2.5309099e-01 },
        { 0.0000000e+00, 5.4818985e-01, 2.4502264e-01 },
        { 0.0000000e+00, 5.3575418e-01, 2.3606957e-01 },
        { 0.0000000e+00, 5.2309640e-01, 2.2616201e-01 },
        { 0.0000000e+00, 5.1023152e-01, 2.1533816e-01 },
        { 0.0000000e+00, 4.9698942e-01, 2.0339702e-01 },
        { 0.0000000e+00, 4.8023725e-01, 1.8622389e-01 },
        { 0.0000000e+00, 4.6270234e-01, 1.6705498e-01 },
        { 0.0000000e+00, 4.4950174e-01, 1.5218074e-01 },
        { 0.0000000e+00, 4.4559841e-01, 1.4770542e-01 },
        { 0.0000000e+00, 4.4894869e-01, 1.5124700e-01 },
        { 0.0000000e+00, 4.5583646e-01, 1.5835237e-01 },
        { 0.0000000e+00, 4.6458682e-01, 1.6702715e-01 },
        { 0.0000000e+00, 4.7358037e-01, 1.7533354e-01 },
        { 0.0000000e+00, 4.8452518e-01, 1.8474093e-01 },
        { 0.0000000e+00, 4.9773559e-01, 1.9533613e-01 },
        { 0.0000000e+00, 5.1152698e-01, 2.0517414e-01 },
        { 0.0000000e+00, 5.2422588e-01, 2.1233214e-01 },
        { 0.0000000e+00, 5.3591169e-01, 2.1794430e-01 },
        { 0.0000000e+00, 5.4729923e-01, 2.2299139e-01 },
        { 0.0000000e+00, 5.5831675e-01, 2.2659993e-01 },
        { 0.0000000e+00, 5.6889288e-01, 2.2789863e-01 },
        { 0.0000000e+00, 5.7911452e-01, 2.2727351e-01 },
        { 0.0000000e+00, 5.8901405e-01, 2.2564758e-01 },
        { 0.0000000e+00, 5.9847977e-01, 2.2322252e-01 },
        { 0.0000000e+00, 6.0740000e-01, 2.2020000e-01 },
        { 0.0000000e+00, 6.1583267e-01, 2.1471383e-01 },
        { 0.0000000e+00, 6.2387479e-01, 2.0510311e-01 },
        { 0.0000000e+00, 6.3147329e-01, 1.9187744e-01 },
        { 0.0000000e+00, 6.3857511e-01, 1.7554645e-01 },
        { 2.5553924e-03, 6.4478758e-01, 1.4030177e-01 },
        { 1.0080803e-02, 6.5026428e-01, 8.2641036e-02 },
        { 2.1546911e-02, 6.5591907e-01, 2.7595524e-02 },
        { 3.5917615e-02, 6.6266700e-01, 2.4657768e-04 },
        { 6.6240946e-02, 6.7162591e-01, 0.0000000e+00 },
        { 1.2168967e-01, 6.8269160e-01, 0.0000000e+00 },
        { 1.8493483e-01, 6.9461874e-01, 0.0000000e+00 },
        { 2.3846654e-01, 7.0615648e-01, 0.0000000e+00 },
        { 2.7531230e-01, 7.1680747e-01, 0.0000000e+00 },
        { 3.0713412e-01, 7.2731672e-01, 0.0000000e+00 },
        { 3.3633506e-01, 7.3770090e-01, 0.0000000e+00 },
        { 3.6509245e-01, 7.4796005e-01, 0.0000000e+00 },
        { 3.9432877e-01, 7.5810108e-01, 0.0000000e+00 },
        { 4.2272056e-01, 7.6813012e-01, 0.0000000e+00 },
        { 4.5056256e-01, 7.7803357e-01, 0.0000000e+00 },
        { 4.7820792e-01, 7.8779740e-01, 0.0000000e+00 },
        { 5.0580848e-01, 7.9741357e-01, 0.0000000e+00 },
        { 5.3312596e-01, 8.0689383e-01, 0.0000000e+00 },
        { 5.6026641e-01, 8.1624712e-01, 0.0000000e+00 },
        { 5.8735339e-01, 8.2548201e-01, 0.0000000e+00 },
        { 6.1444524e-01, 8.3460911e-01, 0.0000000e+00 },
        { 6.4145149e-01, 8.4363152e-01, 0.0000000e+00 },
        { 6.6841671e-01, 8.5253579e-01, 0.0000000e+00 },
        { 6.9539499e-01, 8.6130782e-01, 0.0000000e+00 },
        { 7.2242040e-01, 8.6993680e-01, 0.0000000e+00 },
        { 7.4945903e-01, 8.7843207e-01, 0.0000000e+00 },
        { 7.7651132e-01, 8.8680721e-01, 0.0000000e+00 },
        { 8.0358175e-01, 8.9507551e-01, 0.0000000e+00 },
        { 8.3271386e-01, 9.0387086e-01, 0.0000000e+00 },
        { 8.6766991e-01, 9.1434006e-01, 0.0000000e+00 },
        { 9.0013006e-01, 9.2393977e-01, 0.0000000e+00 },
        { 9.2077677e-01, 9.2982241e-01, 0.0000000e+00 },
        { 9.2456924e-01, 9.2781057e-01, 0.0000000e+00 },
        { 9.2564105e-01, 9.1285551e-01, 0.0000000e+00 },
        { 9.2635741e-01, 8.9046807e-01, 0.0000000e+00 },
        { 9.2697083e-01, 8.6719474e-01, 0.0000000e+00 },
        { 9.2769157e-01, 8.4821687e-01, 0.0000000e+00 },
        { 9.2842486e-01, 8.3008167e-01, 0.0000000e+00 },
        { 9.2909844e-01, 8.1188739e-01, 0.0000000e+00 },
        { 9.2966360e-01, 7.9361226e-01, 0.0000000e+00 },
        { 9.3009633e-01, 7.7523909e-01, 0.0000000e+00 },
        { 9.3050412e-01, 7.5678148e-01, 0.0000000e+00 },
        { 9.3084924e-01, 7.3823664e-01, 0.0000000e+00 },
        { 9.3106311e-01, 7.1959687e-01, 0.0000000e+00 },
        { 9.3108805e-01, 7.0085813e-01, 0.0000000e+00 },
        { 9.3097635e-01, 6.8203628e-01, 0.0000000e+00 },
        { 9.3076913e-01, 6.6311377e-01, 0.0000000e+00 },
        { 9.3049333e-01, 6.4406561e-01, 0.0000000e+00 },
        { 9.3014986e-01, 6.2487209e-01, 0.0000000e+00 },
        { 9.2952249e-01, 6.0555926e-01, 0.0000000e+00 },
        { 9.2862972e-01, 5.8610923e-01, 0.0000000e+00 },
        { 9.2756287e-01, 5.6648973e-01, 0.0000000e+00 },
        { 9.2639385e-01, 5.4667400e-01, 0.0000000e+00 },
        { 9.2495091e-01, 5.2669651e-01, 0.0000000e+00 },
        { 9.2321277e-01, 5.0653324e-01, 0.0000000e+00 },
        { 9.2123278e-01, 4.8613718e-01, 0.0000000e+00 },
        { 9.1905177e-01, 4.6546939e-01, 0.0000000e+00 },
        { 9.1650878e-01, 4.4460368e-01, 0.0000000e+00 },
        { 9.1358608e-01, 4.2348241e-01, 0.0000000e+00 },
        { 9.1034507e-01, 4.0199188e-01, 0.0000000e+00 },
        { 9.0683876e-01, 3.8003436e-01, 0.0000000e+00 },
        { 9.0292399e-01, 3.5782239e-01, 0.0000000e+00 },
        { 8.9856441e-01, 3.3519814e-01, 0.0000000e+00 },
        { 8.9380805e-01, 3.1181741e-01, 0.0000000e+00 },
        { 8.8869627e-01, 2.8738118e-01, 0.0000000e+00 },
        { 8.8305194e-01, 2.6285785e-01, 0.0000000e+00 },
        { 8.7686088e-01, 2.3752669e-01, 0.0000000e+00 },
        { 8.7023664e-01, 2.0970498e-01, 0.0000000e+00 },
        { 8.6328379e-01, 1.7743686e-01, 2.3965082e-05 },
        { 8.5564798e-01, 1.2724033e-01, 1.4656006e-03 },
        { 8.4743561e-01, 6.7202891e-02, 4.9128609e-03 },
        { 8.3911937e-01, 1.7980839e-02, 1.0052615e-02 },
        { 8.3117216e-01, 0.0000000e+00, 1.6721722e-02 },
        { 8.2389423e-01, 0.0000000e+00, 4.0759297e-02 },
        { 8.1690117e-01, 0.0000000e+00, 8.1419458e-02 },
        { 8.0964806e-01, 0.0000000e+00, 1.2221268e-01 },
        { 8.0159045e-01, 0.0000000e+00, 1.4668505e-01 },
        { 7.9251961e-01, 0.0000000e+00, 1.5830472e-01 },
        { 7.8260832e-01, 0.0000000e+00, 1.6792460e-01 },
        { 7.7194048e-01, 0.0000000e+00, 1.7447844e-01 },
        { 7.6060000e-01, 0.0000000e+00, 1.7690000e-01 }
    },
    {
        { 0.0000000e+00, 0.0000000e+00, 0.0000000e+00 },
        { 0.0000000e+00, 0.0000000e+00, 3.5287604e-02 },
        { 0.0000000e+00, 0.0000000e+00, 7.0536129e-02 },
        { 0.0000000e+00, 0.0000000e+00, 1.0570649e-01 },
        { 0.0000000e+00, 0.0000000e+00, 1.4075962e-01 },
        { 0.0000000e+00, 0.0000000e+00, 1.7565642e-01 },
        { 0.0000000e+00, 0.0000000e+00, 2.1035783e-01 },
        { 0.0000000e+00, 0.0000000e+00, 2.4482475e-01 },
        { 0.0000000e+00, 0.0000000e+00, 2.7901812e-01 },
        { 0.0000000e+00, 0.0000000e+00, 3.1289884e-01 },
        { 0.0000000e+00, 0.0000000e+00, 3.4642784e-01 },
        { 0.0000000e+00, 0.0000000e+00, 3.7956605e-01 },
        { 0.0000000e+00, 0.0000000e+00, 4.1227437e-01 },
        { 0.0000000e+00, 0.0000000e+00, 4.4451374e-01 },
        { 0.0000000e+00, 0.0000000e+00, 4.7624506e-01 },
        { 0.0000000e+00, 0.0000000e+00, 5.0742927e-01 },
        { 0.0000000e+00, 0.0000000e+00, 5.3802727e-01 },
        { 0.0000000e+00, 0.0000000e+00, 5.6800000e-01 },
        { 0.0000000e+00, 0.0000000e+00, 5.9730837e-01 },
        { 0.0000000e+00, 0.0000000e+00, 6.2591329e-01 },
        { 0.0000000e+00, 0.0000000e+00, 6.5377570e-01 },
        { 0.0000000e+00, 0.0000000e+00, 6.8085650e-01 },
        { 0.0000000e+00, 0.0000000e+00, 7.0711663e-01 },
        { 0.0000000e+00, 0.0000000e+00, 7.3251700e-01 },
        { 0.0000000e+00, 0.0000000e+00, 7.5701852e-01 },
        { 0.0000000e+00, 0.0000000e+00, 7.8058213e-01 },
        { 0.0000000e+00, 0.0000000e+00, 8.0316874e-01 },
        { 0.0000000e+00, 0.0000000e+00, 8.2473926e-01 },
        { 0.0000000e+00, 0.0000000e+00, 8.4525463e-01 },
        { 0.0000000e+00, 0.0000000e+00, 8.6467576e-01 },
        { 0.0000000e+00, 0.0000000e+00, 8.8296357e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.0007897e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.1598290e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.3063627e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.4400000e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.5603501e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.6670222e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.7596255e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.8377692e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.9010625e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.9491146e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.9815347e-01 },
        { 0.0000000e+00, 0.0000000e+00, 9.9979320e-01 },
        { 0.0000000e+00, 4.1196825e-04, 1.0000000e+00 },
        { 0.0000000e+00, 3.6490942e-03, 1.0000000e+00 },
        { 0.0000000e+00, 9.9735396e-03, 1.0000000e+00 },
        { 0.0000000e+00, 1.9228984e-02, 1.0000000e+00 },
        { 0.0000000e+00, 3.1259108e-02, 1.0000000e+00 },
        { 0.0000000e+00, 4.5907592e-02, 1.0000000e+00 },
        { 0.0000000e+00, 6.3018115e-02, 1.0000000e+00 },
        { 0.0000000e+00, 8.2434358e-02, 1.0000000e+00 },
        { 0.0000000e+00, 1.0400000e-01, 1.0000000e+00 },
        { 0.0000000e+00, 1.2755872e-01, 1.0000000e+00 },
        { 0.0000000e+00, 1.5295420e-01, 1.0000000e+00 },
        { 0.0000000e+00, 1.8003012e-01, 1.0000000e+00 },
        { 0.0000000e+00, 2.0863016e-01, 1.0000000e+00 },
        { 0.0000000e+00, 2.3859801e-01, 1.0000000e+00 },
        { 0.0000000e+00, 2.6977733e-01, 1.0000000e+00 },
        { 0.0000000e+00, 3.0201181e-01, 1.0000000e+00 },
        { 0.0000000e+00, 3.3514513e-01, 1.0000000e+00 },
        { 0.0000000e+00, 3.6902096e-01, 1.0000000e+00 },
        { 0.0000000e+00, 4.0348300e-01, 1.0000000e+00 },
        { 0.0000000e+00, 4.3837492e-01, 1.0000000e+00 },
        { 0.0000000e+00, 4.7354040e-01, 1.0000000e+00 },
        { 0.0000000e+00, 5.0882312e-01, 1.0000000e+00 },
        { 0.0000000e+00, 5.4406676e-01, 1.0000000e+00 },
        { 0.0000000e+00, 5.7911500e-01, 1.0000000e+00 },
        { 0.0000000e+00, 6.1381152e-01, 1.0000000e+00 },
        { 0.0000000e+00, 6.4800000e-01, 1.0000000e+00 },
        { 0.0000000e+00, 6.8152412e-01, 1.0000000e+00 },
        { 0.0000000e+00, 7.1422756e-01, 1.0000000e+00 },
        { 0.0000000e+00, 7.4595400e-01, 1.0000000e+00 },
        { 0.0000000e+00, 7.7654712e-01, 1.0000000e+00 },
        { 0.0000000e+00, 8.0585060e-01, 1.0000000e+00 },
        { 0.0000000e+00, 8.3370812e-01, 1.0000000e+00 },
        { 0.0000000e+00, 8.5996336e-01, 1.0000000e+00 },
        { 0.0000000e+00, 8.8446000e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.0704173e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.2755221e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.4583513e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.6173417e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.7509302e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.8575534e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.9356483e-01, 1.0000000e+00 },
        { 0.0000000e+00, 9.9836515e-01, 1.0000000e+00 },
        { 0.0000000e+00, 1.0000000e+00, 1.0000000e+00 },
        { 1.6348463e-03, 1.0000000e+00, 1.0000000e+00 },
        { 6.4351720e-03, 1.0000000e+00, 1.0000000e+00 },
        { 1.4244657e-02, 1.0000000e+00, 1.0000000e+00 },
        { 2.4906981e-02, 1.0000000e+00, 1.0000000e+00 },
        { 3.8265825e-02, 1.0000000e+00, 1.0000000e+00 },
        { 5.4164869e-02, 1.0000000e+00, 1.0000000e+00 },
        { 7.2447792e-02, 1.0000000e+00, 1.0000000e+00 },
        { 9.2958274e-02, 1.0000000e+00, 1.0000000e+00 },
        { 1.1554000e-01, 1.0000000e+00, 1.0000000e+00 },
        { 1.4003664e-01, 1.0000000e+00, 1.0000000e+00 },
        { 1.6629188e-01, 1.0000000e+00, 1.0000000e+00 },
        { 1.9414940e-01, 1.0000000e+00, 1.0000000e+00 },
        { 2.2345288e-01, 1.0000000e+00, 1.0000000e+00 },
        { 2.5404600e-01, 1.0000000e+00, 1.0000000e+00 },
        { 2.8577244e-01, 1.0000000e+00, 1.0000000e+00 },
        { 3.1847588e-01, 1.0000000e+00, 1.0000000e+00 },
        { 3.5200000e-01, 1.0000000e+00, 1.0000000e+00 },
        { 3.8618848e-01, 1.0000000e+00, 1.0000000e+00 },
        { 4.2088500e-01, 1.0000000e+00, 1.0000000e+00 },
        { 4.5593324e-01, 1.0000000e+00, 1.0000000e+00 },
        { 4.9117688e-01, 1.0000000e+00, 1.0000000e+00 },
        { 5.2645960e-01, 1.0000000e+00, 1.0000000e+00 },
        { 5.6162508e-01, 1.0000000e+00, 1.0000000e+00 },
        { 5.9651700e-01, 1.0000000e+00, 1.0000000e+00 },
        { 6.3097904e-01, 1.0000000e+00, 1.0000000e+00 },
        { 6.6485487e-01, 1.0000000e+00, 1.0000000e+00 },
        { 6.9798819e-01, 1.0000000e+00, 1.0000000e+00 },
        { 7.3022267e-01, 1.0000000e+00, 1.0000000e+00 },
        { 7.6140199e-01, 1.0000000e+00, 1.0000000e+00 },
        { 7.9136984e-01, 1.0000000e+00, 1.0000000e+00 },
        { 8.1996988e-01, 1.0000000e+00, 1.0000000e+00 },
        { 8.4704580e-01, 1.0000000e+00, 1.0000000e+00 },
        { 8.7244128e-01, 1.0000000e+00, 1.0000000e+00 },
        { 8.9600000e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.1756564e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.3698188e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.5409241e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.6874089e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.8077102e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.9002646e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.9635091e-01, 1.0000000e+00, 1.0000000e+00 },
        { 9.9958803e-01, 1.0000000e+00, 1.0000000e+00 },
        { 1.0000000e+00, 1.0000000e+00, 9.9958803e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.9635091e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.9002646e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.8077102e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.6874089e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.5409241e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.3698188e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.1756564e-01 },
        { 1.0000000e+00, 1.0000000e+00, 8.9600000e-01 },
        { 1.0000000e+00, 1.0000000e+00, 8.7244128e-01 },
        { 1.0000000e+00, 1.0000000e+00, 8.4704580e-01 },
        { 1.0000000e+00, 1.0000000e+00, 8.1996988e-01 },
        { 1.0000000e+00, 1.0000000e+00, 7.9136984e-01 },
        { 1.0000000e+00, 1.0000000e+00, 7.6140199e-01 },
        { 1.0000000e+00, 1.0000000e+00, 7.3022267e-01 },
        { 1.0000000e+00, 1.0000000e+00, 6.9798819e-01 },
        { 1.0000000e+00, 1.0000000e+00, 6.6485487e-01 },
        { 1.0000000e+00, 1.0000000e+00, 6.3097904e-01 },
        { 1.0000000e+00, 1.0000000e+00, 5.9651700e-01 },
        { 1.0000000e+00, 1.0000000e+00, 5.6162508e-01 },
        { 1.0000000e+00, 1.0000000e+00, 5.2645960e-01 },
        { 1.0000000e+00, 1.0000000e+00, 4.9117688e-01 },
        { 1.0000000e+00, 1.0000000e+00, 4.5593324e-01 },
        { 1.0000000e+00, 1.0000000e+00, 4.2088500e-01 },
        { 1.0000000e+00, 1.0000000e+00, 3.8618848e-01 },
        { 1.0000000e+00, 1.0000000e+00, 3.5200000e-01 },
        { 1.0000000e+00, 1.0000000e+00, 3.1847588e-01 },
        { 1.0000000e+00, 1.0000000e+00, 2.8577244e-01 },
        { 1.0000000e+00, 1.0000000e+00, 2.5404600e-01 },
        { 1.0000000e+00, 1.0000000e+00, 2.2345288e-01 },
        { 1.0000000e+00, 1.0000000e+00, 1.9414940e-01 },
        { 1.0000000e+00, 1.0000000e+00, 1.6629188e-01 },
        { 1.0000000e+00, 1.0000000e+00, 1.4003664e-01 },
        { 1.0000000e+00, 1.0000000e+00, 1.1554000e-01 },
        { 1.0000000e+00, 1.0000000e+00, 9.2958274e-02 },
        { 1.0000000e+00, 1.0000000e+00, 7.2447792e-02 },
        { 1.0000000e+00, 1.0000000e+00, 5.4164869e-02 },
        { 1.0000000e+00, 1.0000000e+00, 3.8265825e-02 },
        { 1.0000000e+00, 1.0000000e+00, 2.4906981e-02 },
        { 1.0000000e+00, 1.0000000e+00, 1.4244657e-02 },
        { 1.0000000e+00, 1.0000000e+00, 6.4351720e-03 },
        { 1.0000000e+00, 1.0000000e+00, 1.6348463e-03 },
        { 1.0000000e+00, 1.0000000e+00, 0.0000000e+00 },
        { 1.0000000e+00, 9.9836515e-01, 0.0000000e+00 },
        { 1.0000000e+00, 9.9356483e-01, 0.0000000e+00 },
        { 1.0000000e+00, 9.8575534e-01, 0.0000000e+00 },
        { 1.0000000e+00, 9.7509302e-01, 0.0000000e+00 },
        { 1.0000000e+00, 9.6173417e-01, 0.0000000e+00 },
        { 1.0000000e+00, 9.4583513e-01, 0.0000000e+00 },
        { 1.0000000e+00, 9.2755221e-01, 0.0000000e+00 },
        { 1.0000000e+00, 9.0704173e-01, 0.0000000e+00 },
        { 1.0000000e+00, 8.8446000e-01, 0.0000000e+00 },
        { 1.0000000e+00, 8.5996336e-01, 0.0000000e+00 },
        { 1.0000000e+00, 8.3370812e-01, 0.0000000e+00 },
        { 1.0000000e+00, 8.0585060e-01, 0.0000000e+00 },
        { 1.0000000e+00, 7.7654712e-01, 0.0000000e+00 },
        { 1.0000000e+00, 7.4595400e-01, 0.0000000e+00 },
        { 1.0000000e+00, 7.1422756e-01, 0.0000000e+00 },
        { 1.0000000e+00, 6.8152412e-01, 0.0000000e+00 },
        { 1.0000000e+00, 6.4800000e-01, 0.0000000e+00 },
        { 1.0000000e+00, 6.1381152e-01, 0.0000000e+00 },
        { 1.0000000e+00, 5.7911500e-01, 0.0000000e+00 },
        { 1.0000000e+00, 5.4406676e-01, 0.0000000e+00 },
        { 1.0000000e+00, 5.0882312e-01, 0.0000000e+00 },
        { 1.0000000e+00, 4.7354040e-01, 0.0000000e+00 },
        { 1.0000000e+00, 4.3837492e-01, 0.0000000e+00 },
        { 1.0000000e+00, 4.0348300e-01, 0.0000000e+00 },
        { 1.0000000e+00, 3.6902096e-01, 0.0000000e+00 },
        { 1.0000000e+00, 3.3514513e-01, 0.0000000e+00 },
        { 1.0000000e+00, 3.0201181e-01, 0.0000000e+00 },
        { 1.0000000e+00, 2.6977733e-01, 0.0000000e+00 },
        { 1.0000000e+00, 2.3859801e-01, 0.0000000e+00 },
        { 1.0000000e+00, 2.0863016e-01, 0.0000000e+00 },
        { 1.0000000e+00, 1.8003012e-01, 0.0000000e+00 },
        { 1.0000000e+00, 1.5295420e-01, 0.0000000e+00 },
        { 1.0000000e+00, 1.2755872e-01, 0.0000000e+00 },
        { 1.0000000e+00, 1.0400000e-01, 0.0000000e+00 },
        { 1.0000000e+00, 8.2434358e-02, 0.0000000e+00 },
        { 1.0000000e+00, 6.3018115e-02, 0.0000000e+00 },
        { 1.0000000e+00, 4.5907592e-02, 0.0000000e+00 },
        { 1.0000000e+00, 3.1259108e-02, 0.0000000e+00 },
        { 1.0000000e+00, 1.9228984e-02, 0.0000000e+00 },
        { 1.0000000e+00, 9.9735396e-03, 0.0000000e+00 },
        { 1.0000000e+00, 3.6490942e-03, 0.0000000e+00 },
        { 1.0000000e+00, 4.1196825e-04, 0.0000000e+00 },
        { 9.9979320e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.9815347e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.9491146e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.9010625e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.8377692e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.7596255e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.6670222e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.5603501e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.4400000e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.3063627e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.1598290e-01, 0.0000000e+00, 0.0000000e+00 },
        { 9.0007897e-01, 0.0000000e+00, 0.0000000e+00 },
        { 8.8296357e-01, 0.0000000e+00, 0.0000000e+00 },
        { 8.6467576e-01, 0.0000000e+00, 0.0000000e+00 },
        { 8.4525463e-01, 0.0000000e+00, 0.0000000e+00 },
        { 8.2473926e-01, 0.0000000e+00, 0.0000000e+00 },
        { 8.0316874e-01, 0.0000000e+00, 0.0000000e+00 },
        { 7.8058213e-01, 0.0000000e+00, 0.0000000e+00 },
        { 7.5701852e-01, 0.0000000e+00, 0.0000000e+00 },
        { 7.3251700e-01, 0.0000000e+00, 0.0000000e+00 },
        { 7.0711663e-01, 0.0000000e+00, 0.0000000e+00 },
        { 6.8085650e-01, 0.0000000e+00, 0.0000000e+00 },
        { 6.5377570e-01, 0.0000000e+00, 0.0000000e+00 },
        { 6.2591329e-01, 0.0000000e+00, 0.0000000e+00 },
        { 5.9730837e-01, 0.0000000e+00, 0.0000000e+00 },
        { 5.6800000e-01, 0.0000000e+00, 0.0000000e+00 },
        { 5.3802727e-01, 0.0000000e+00, 0.0000000e+00 },
        { 5.0742927e-01, 0.0000000e+00, 0.0000000e+00 },
        { 4.7624506e-01, 0.0000000e+00, 0.0000000e+00 },
        { 4.4451374e-01, 0.0000000e+00, 0.0000000e+00 },
        { 4.1227437e-01, 0.0000000e+00, 0.0000000e+00 },
        { 3.7956605e-01, 0.0000000e+00, 0.0000000e+00 },
        { 3.4642784e-01, 0.0000000e+00, 0.0000000e+00 },
        { 3.1289884e-01, 0.0000000e+00, 0.0000000e+00 },
        { 2.7901812e-01, 0.0000000e+00, 0.0000000e+00 },
        { 2.4482475e-01, 0.0000000e+00, 0.0000000e+00 },
        { 2.1035783e-01, 0.0000000e+00, 0.0000000e+00 },
        { 1.7565642e-01, 0.0000000e+00, 0.0000000e+00 },
        { 1.4075962e-01, 0.0000000e+00, 0.0000000e+00 },
        { 1.0570649e-01, 0.0000000e+00, 0.0000000e+00 },
        { 7.0536129e-02, 0.0000000e+00, 0.0000000e+00 },
        { 3.5287604e-02, 0.0000000e+00, 0.0000000e+00 },
        { 1.1102230e-16, 0.0000000e+00, 0.0000000e+00 }
    }
};
