/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include <QDateTime>
#include <QDebug>
#include <QImage>
#include "RemoteFileSystemModel.h"
#include "CubeError.h"
#include "CubeClientConnection.h"
#include "CubeNetworkRequest.h"
#include "CubeSocket.h"
#include "Globals.h"

using namespace cubegui;
using namespace cube;
using namespace std;

#include <QCoreApplication>
#include <QThread>



RemoteFileSystemModel::RemoteFileSystemModel( const QString& url ) : directory( url )
{
}

RemoteFileSystemModel::~RemoteFileSystemModel()
{
}

/**
 * @brief RemoteFileSystemModel::setDirectory
 * @param path
 * request the contents of the directory from the server
 * emits directorySelected() when ready (via updateDirectoryContents())
 */
void
RemoteFileSystemModel::setDirectory( QString path )
{
    if ( path.isEmpty() )
    {
        path = "."; // use server working directory as default
    }

    // use Directory class to read directory contents in a separate thread to avoid blocking the GUI
    QThread* thread = new QThread();
    directory.moveToThread( thread );
    // start lambda without context object (this as 3rd argument), to force it to run
    // in the new thread which is started later
    connect( thread, &QThread::started, [ this, path ](){
        directory.update( path );
    } );
    connect( thread, &QThread::finished, this, &RemoteFileSystemModel::updateDirectoryContents );

    thread->start();
}

/**
 * @brief RemoteFileSystemModel::updateDirectoryContents
 * called after the update thread of the directory class has finished
 */
void
RemoteFileSystemModel::updateDirectoryContents()
{
    const std::vector<FileInfo>& newFiles     = directory.getFiles();
    QString                      errorMessage = directory.getErrorMessage();

    if ( !errorMessage.isEmpty() )
    {
        files.clear();
        beginResetModel();
        endResetModel();
        Globals::setStatusMessage( errorMessage, Error );
        return;
    }

    std::vector<FileInfo> oldFiles = files;
    if ( newFiles.size() > 0 )
    {
        files = newFiles;
        QString directory = files.at( 0 ).name().c_str();
        files.erase( files.begin() );
        emit directorySelected( directory );
        beginResetModel();
        endResetModel();
    }
    else
    {
        files = oldFiles;
    }
}

bool
RemoteFileSystemModel::containsFile( const QString& absoluteFile ) const
{
    if ( absoluteFile.isEmpty() )
    {
        return false;
    }
    for ( const FileInfo& info : files )
    {
        if ( !info.isDirectory() && ( info.name().c_str() == absoluteFile ) )
        {
            return true;
        }
    }
    return false;
}

void
RemoteFileSystemModel::itemActivated( QModelIndex index )
{
    FileInfo* file = getFile( index );

    if ( !file->isDirectory() )
    {
        emit fileSelected(  file->name().c_str() );
    }
    else
    {
        setDirectory( file->name().c_str() );
    }
}

void
RemoteFileSystemModel::itemClicked( QModelIndex index )
{
    FileInfo* file = getFile( index );

    if ( !file->isDirectory() )
    {
        emit fileClicked(  file->name().c_str() );
    }
}

FileInfo*
RemoteFileSystemModel::getFile( const QModelIndex& idx ) const
{
    if ( !idx.isValid() ) // invalid index is used for tree root
    {
        return NULL;
    }
    else
    {
        return static_cast<FileInfo*>( idx.internalPointer() );
    }
}

QModelIndex
RemoteFileSystemModel::getModelIndex( const QString& path )
{
    for ( unsigned i = 0; i < files.size(); i++ )
    {
        if ( files[ i ].name().c_str() == path )
        {
            return index( i, 0, QModelIndex() );
        }
    }
    return QModelIndex();
}

QModelIndex
RemoteFileSystemModel::index( int row, int column, const QModelIndex& parent ) const
{
    if ( !hasIndex( row, column, parent ) )
    {
        return QModelIndex();
    }

    return createIndex( row, column, ( void* )&files.at( row ) );
}

QModelIndex
RemoteFileSystemModel::parent( const QModelIndex& ) const
{
    return QModelIndex();
}

int
RemoteFileSystemModel::rowCount( const QModelIndex& index ) const
{
    if ( index.isValid() )
    {
        return 0;                  // don't show contents of subdirectories
    }
    return files.size();
}

int
RemoteFileSystemModel::columnCount( const QModelIndex& ) const
{
    return COLUMNS;
}

QString
fileSize( uint64_t size )
{
    double      num = size;
    QStringList list;
    list << QObject::tr( "KB" ) << QObject::tr( "MB" ) << QObject::tr( "GB" ) << QObject::tr( "TB" );

    QStringListIterator i( list );
    QString             unit( QObject::tr( "bytes" ) );

    while ( num >= 1024.0 && i.hasNext() )
    {
        unit = i.next();
        num /= 1024.0;
    }
    return QString().setNum( num, 'f', 2 ) + " " + unit;
}

QVariant
RemoteFileSystemModel::data( const QModelIndex& index, int role ) const
{
    if ( !index.isValid() || index.model() != this )
    {
        return QVariant();
    }
    FileInfo* file = getFile( index );
    if ( !file )
    {
        return QVariant();
    }
    switch ( role )
    {
        case Qt::EditRole:
        case Qt::DisplayRole:
        {
            switch ( index.column() )
            {
                case NAME:
                    return QFileInfo( file->name().c_str() ).fileName();
                case SIZE:
                    return file->isDirectory() ? "" : fileSize( file->size() );
                case DATE:
                    QDateTime date;
#if QT_VERSION >= 0x050800
                    date = QDateTime::fromSecsSinceEpoch( file->time() );
#else
                    date = QDateTime::fromTime_t( file->time() );
#endif
                    return spacing + date.toString( "dd.MM.yyyy  HH:mm" );
            }
            break;
        }
        case Qt::DecorationRole:
            if ( index.column() == 0 )
            {
                QIcon icon;

                if ( file->isDirectory() )
                {
                    icon = iconProvider.icon( QFileIconProvider::Folder );
                }
                else
                {
                    icon = iconProvider.icon( QFileIconProvider::File );
                }
                return icon;
            }
            break;
        case Qt::TextAlignmentRole:
            if ( index.column() == 1 )
            {
                return ( int )( Qt::AlignVCenter | Qt::AlignRight ); // align file size right and vertically centered
            }
            break;
        case LSIZE:
            return QVariant( static_cast<qlonglong> ( file->size() ) );
        case LDATE:
            return QVariant( static_cast<qlonglong> ( file->time() ) );
        case IS_DIR:
            return QVariant( file->isDirectory() );
        default:
            return QVariant();
    }


    return QVariant();
}

Qt::ItemFlags
RemoteFileSystemModel::flags( const QModelIndex& ) const
{
    return Qt::ItemIsEnabled | Qt::ItemIsSelectable;
}

bool
RemoteFileSystemModel::hasChildren( const QModelIndex& parent ) const
{
    return QAbstractItemModel::hasChildren( parent );
}

QVariant
RemoteFileSystemModel::headerData( int section, Qt::Orientation orientation, int role ) const
{
    switch ( role )
    {
        case Qt::DecorationRole:
            break;
        case Qt::TextAlignmentRole:
            return Qt::AlignLeft;
    }
    if ( role != Qt::DisplayRole )
    {
        return QAbstractItemModel::headerData( section, orientation, role );
    }

    QString returnValue;
    switch ( section )
    {
        case NAME:
            returnValue = tr( "Name" );
            break;
        case SIZE:
            returnValue = tr( "Size" );
            break;
        case DATE:
            returnValue = tr( "Date modified" );
            break;
    }
    return spacing + returnValue;
}

// --- FileSortFilterProxyModel ---

bool
FileSortFilterProxyModel::lessThan( const QModelIndex& left, const QModelIndex& right ) const
{
    if ( left.column() == NAME )
    {
        QString first     = sourceModel()->data( left ).toString();
        QString second    = sourceModel()->data( right ).toString();
        bool    firstDir  = sourceModel()->data( left, IS_DIR ).toBool();
        bool    secondDir = sourceModel()->data( right, IS_DIR ).toBool();
        if ( firstDir && !secondDir )
        {
            return true;
        }
        else if ( !firstDir && secondDir )
        {
            return false;
        }
        else
        {
            return first < second;
        }
    }
    else if ( left.column() == SIZE )
    {
        return sourceModel()->data( left, LSIZE ).toLongLong() < sourceModel()->data( right, LSIZE ).toLongLong();
    }
    else if ( left.column() == DATE )
    {
        return sourceModel()->data( left, LDATE ).toLongLong() < sourceModel()->data( right, LDATE ).toLongLong();
    }
    return true;
}


// ==========================================================================================
// Directory class retreives directory contents from a cube server
// ==========================================================================================

/**
 *  creates a client connection to a cube server, which is closed if this object is deleted
 */
Directory::Directory( const QString& url ) : serverUrl( url )
{
    clientConnection = nullptr;
}

Directory::~Directory()
{
    clientConnection = nullptr;
}

void
Directory::update( const QString& path )
{
    try
    {
        if ( !clientConnection )
        {
            clientConnection = cube::ClientConnection::create( Socket::create(), serverUrl.toStdString() );
        }

        NetworkRequest::Ptr request = FileSystemRequest::create( path.toStdString() );
        FileSystemRequest*  fr      = dynamic_cast<FileSystemRequest*>( request.get() );

        request->sendRequest( *clientConnection, NULL );
        request->receiveResponse( *clientConnection, NULL );

        std::vector<FileInfo> allFiles = fr->getFiles(); // all files

        files.clear();
        for ( FileInfo& file : allFiles )
        {
            QString filename = QFileInfo( file.name().c_str() ).fileName();
            if ( !filename.isEmpty() &&
                 ( ( filename.at( 0 ) != '.' ) || ( filename == ".." ) ) )
            {
                files.push_back( file );
            }
        }
    }
    catch ( const cube::NetworkError& e )
    {
        networkError = e.what();  // handle in main thread
    }

    QThread::currentThread()->quit();                             // -> QThread::finished signal is triggered
    this->moveToThread( QCoreApplication::instance()->thread() ); // move back to main thread
}

QString
Directory::getErrorMessage() const
{
    return networkError;
}

const std::vector<cube::FileInfo>&
Directory::getFiles() const
{
    return files;
}
