/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include "cubelib-version.h"

#include <QDebug>
#include <QApplication>
#include <QPushButton>
#include <QStyle>
#include <QLabel>
#include "InfoWidget.h"
#include "Globals.h"
#include "TabManager.h"
#include "HtmlWidget.h"
#include "TreeView.h"
#include "TreeItemMarker.h"
#include "ValueView.h"

using namespace cubegui;

InfoWidget* InfoWidget::single = 0;

InfoWidget*
InfoWidget::getInstance()
{
    return single ? single : single = new InfoWidget();
}

void
InfoWidget::deleteInstance()
{
    if ( single )
    {
        if ( single->isVisible() )
        {
            Globals::getTabManager()->getTab( SYSTEM )->removePluginTab( single );
        }
        delete single;
        single = nullptr;
    }
}

InfoWidget::~InfoWidget()
{
    single = 0;
}

InfoWidget::InfoWidget() : embeddedStart( "info:" )
{
    content = Info;
    doc     = nullptr;
}

void
InfoWidget::showSystemInfo()
{
    QList<int> sizeList;
    sizeList << width() / 3 << width() / 3 << width() / 3;
    if ( horizontal )
    {
        horizontal->setSizes( sizeList );
    }
}

void
InfoWidget::showTreeInfo( TreeItem* metricItem, TreeItem* callItem, TreeItem* systemItem )
{
    if ( !isVisible() )
    {
        return;
    }

    if ( content != TreeInfo )
    {
        deleteContent( this );
        metricInfo = new QTextBrowser();
        callInfo   = new QTextBrowser();
        sysInfo    = new QTextBrowser();
        globalInfo = new QTextBrowser();
        if ( Globals::optionIsSet( ExpertMode ) && Globals::optionIsSet( VerboseMode ) )
        {
            debugInfo = new QTextBrowser();
        }
        doc = new QTabWidget();
        doc->addTab( HtmlWidget::createHtmlWidget(), tr( "Metric Documentation" ) );
        doc->addTab( HtmlWidget::createHtmlWidget(), tr( "Call path/Region Documentation" ) );
        doc->addTab( globalInfo, tr( "Global Information" ) );
        if ( Globals::optionIsSet( ExpertMode ) && Globals::optionIsSet( VerboseMode ) )
        {
            doc->addTab( debugInfo, tr( "Debug Information" ) );
        }
        horizontal = new QSplitter( Qt::Horizontal );
        vertical   = new QSplitter( Qt::Vertical );
        horizontal->addWidget( metricInfo );
        horizontal->addWidget( callInfo );
        horizontal->addWidget( sysInfo );
        vertical->addWidget( horizontal );
        vertical->addWidget( doc );

        QList<int> sizeList;
        sizeList << width() / 2 << width() / 2 << 0;
        horizontal->setSizes( sizeList );

        sizeList.clear();
        sizeList << height() / 2 << height() / 2;
        vertical->setSizes( sizeList );

        setLayout( new QVBoxLayout );
        layout()->setContentsMargins( 0, 0, 0, 0 );
        layout()->addWidget( vertical );
    }

    QString metric    = getInfoString( metricItem );
    QString call      = getInfoString( callItem );
    QString system    = getInfoString( systemItem );
    QString metricUrl = getDocumentation( metricItem );
    QString callUrl   = getDocumentation( callItem );

    metricInfo->setText( metric );
    callInfo->setText( call );
    sysInfo->setText( system );

    globalInfo->setText( globalInfoText );
    if ( Globals::optionIsSet( ExpertMode ) && Globals::optionIsSet( VerboseMode ) )
    {
        debugInfo->setText( debugInfoText );
    }

    if ( doc->height() > 0 )
    {
        HtmlWidget* html = static_cast<HtmlWidget*> ( doc->widget( 0 ) );
        if ( metricUrl.startsWith( embeddedStart ) ) // embedded documentation
        {
            QString htmlStr = metricUrl.remove( 0, embeddedStart.size() );
            html->showHtml( htmlStr, QUrl( metricItem->getUrl() ).fragment() );
        }
        else
        {
            html->showUrl( metricUrl );
        }

        html = static_cast<HtmlWidget*> ( doc->widget( 1 ) );
        if ( callUrl.startsWith( embeddedStart ) ) // embedded documentation
        {
            QString htmlStr = callUrl.remove( 0, embeddedStart.size() );
            html->showHtml( htmlStr, QUrl( callItem->getUrl() ).fragment() );
        }
        else
        {
            html->showUrl( callUrl );
        }
    }
    else
    {
        HtmlWidget* html = static_cast<HtmlWidget*> ( doc->widget( 0 ) );
        html->showUrl( "" );
        html = static_cast<HtmlWidget*> ( doc->widget( 1 ) );
        html->showUrl( "" );
    }
    toFront();

    content = TreeInfo;
}

void
InfoWidget::changeTreeInfoOrientation()
{
    if ( horizontal->orientation() == Qt::Horizontal )
    {
        horizontal->setOrientation( Qt::Vertical );
    }
    else
    {
        horizontal->setOrientation( Qt::Horizontal );
    }
}

void
InfoWidget::deleteContent( QWidget* widget )
{
    horizontal = nullptr;
    vertical   = nullptr;
    metricInfo = nullptr;
    callInfo   = nullptr;
    sysInfo    = nullptr;
    globalInfo = nullptr;
    debugInfo  = nullptr;
    doc        = nullptr;
    treeInfo   = nullptr;
    qDeleteAll( this->findChildren<QWidget*>( QString(), Qt::FindDirectChildrenOnly ) );
    delete layout();
}

void
InfoWidget::showPluginInfo( const QString& text, QPushButton* button )
{
    if ( !isVisible() )
    {
        return;
    }
    deleteContent( this );
    QVBoxLayout* verticalLayout = new QVBoxLayout( this );
    verticalLayout->setSpacing( 6 );
    verticalLayout->setContentsMargins( 11, 11, 11, 11 );
    setLayout( verticalLayout );

    QTextBrowser* textBrowser = new QTextBrowser( this );
    textBrowser->setText( text );
    layout()->addWidget( textBrowser );
    layout()->addWidget( button );

    layout()->invalidate();
    content = PluginInfo;
}

/** activate the metric or call tree documentation tab */
void
InfoWidget::activateTab( DisplayType type )
{
    if ( doc && isVisible() )
    {
        doc->setCurrentIndex( type == METRIC ? 0 : 1 );
    }
}

void
InfoWidget::activate()
{
    setGlobalInfo();
    if ( !this->isVisible() )
    {
        Globals::getTabManager()->getTab( SYSTEM )->addTabInterface( this, true, OTHER_PLUGIN_TAB );
    }
}

void
InfoWidget::toFront()
{
    Globals::getTabManager()->getTab( SYSTEM )->toFront( this );
}

QWidget*
InfoWidget::widget()
{
    return this;
}

QString
InfoWidget::label() const
{
    return tr( "Info" );
}

QIcon
InfoWidget::icon() const
{
    return QApplication::style()->standardIcon( QStyle::SP_MessageBoxInformation );
}

void
InfoWidget::setActive( bool active )
{
    if ( active )
    {
        emit tabActivated(); // send info if info widget tab has been selected
    }
}

QString
InfoWidget::formatAsTable( const QList<QPair<QString, QString> >& list )
{
    QWidget widget;
    QColor  backColor = QApplication::palette().color( QPalette::AlternateBase );
    QString colorStr  = backColor.name( QColor::HexRgb );

    QString table     = "<table>";
    QString colorLine = "<tr style=\"background-color:" + colorStr + ";\">";
    int     i         = 0;
    for ( const QPair<QString, QString>& pair : list )
    {
        QString tr = ( ++i % 2 == 0 ) ? "<tr>" : colorLine;
        table += tr + "<td><b>" + pair.first + "</b></td>";
        table += "<td>" + pair.second + "</td></tr>";
    }
    table += "</table>";
    return table;
}

void
InfoWidget::setGlobalInfo()
{
    cube::CubeProxy* cube = Globals::getTabManager()->getCube();

    QList<QPair<QString, QString> > list; // QPair<description, value>
    list.append( QPair<QString, QString>( tr( "Path" ), QString::fromStdString( cube->get_cubename() ) ) );
    const std::vector< std::string >& _mirrors    = cube->getMirrors();
    QString                           mirrorLabel = tr( "Mirrors" ) + ":";
    if ( _mirrors.size() == 0 )
    {
        list.append( QPair<QString, QString>( mirrorLabel, "" ) );
    }
    for ( std::vector< std::string >::const_iterator iter = _mirrors.begin(); iter != _mirrors.end(); ++iter )
    {
        list.append( QPair<QString, QString>( mirrorLabel, QString::fromStdString( *iter ) ) );
        mirrorLabel = "";
    }
    list.append( QPair<QString, QString>( tr( "Global attributes" ) + ":", "" ) );
    const std::map< std::string, std::string >& _attrs = cube->getAttributes();
    for ( std::map< std::string, std::string >::const_iterator iter = _attrs.begin(); iter != _attrs.end(); ++iter )
    {
        list.append( QPair<QString, QString>( QString::fromStdString( iter->first ), QString::fromStdString( iter->second ) ) );
    }

    globalInfoText  = "<html><style>table, th, td { padding: 1px 10px 1px 0px } </style> <body>";
    globalInfoText += formatAsTable( list );
    globalInfoText += "</body></html>";
}

void
InfoWidget::setDebugInfo()
{
    cube::CubeProxy* cube = Globals::getTabManager()->getCube();
    QString          _debugInfo;
    if ( Globals::optionIsSet( ExpertMode ) && Globals::optionIsSet( VerboseMode ) )
    {
        if ( cube->hasCubePlMemoryManager() )
        {
            _debugInfo = QString::fromStdString( cube->getCubePlMemoryManager().dump_memory() );
        }
        else
        {
            _debugInfo = tr( "CubePL memory dump is not available for Network Cube Proxy." );
        }
    }
    debugInfoText = _debugInfo;
}

QString
InfoWidget::getInfoString( TreeItem* item ) const
{
    QList<QPair<QString, QString> > list; // QPair<description, value>

    TabManager* manager = Globals::getTabManager();
    TreeView*   view    = manager->getView( item->getTree() );
    if ( item->getValueObject() ) // add extended info from ValueView if available
    {
        cube::DataType dataType = item->getValueObject()->myDataType();
        QString        extended = Globals::getValueView( dataType )->getExtendedInfo( item );
        if ( !extended.isEmpty() )
        {
            list.append( QPair<QString, QString>( tr( "Value details" ), extended ) );
        }
    }

    if ( item->markerList.size() > 0 )
    {
        QString label = tr( "Defined marker" );
        foreach( const TreeItemMarker * marker, item->markerList )
        {
            list.append( QPair<QString, QString>( label, marker->getLabel( item ) ) );
            label = "";
        }
    }

    // add path info
    QString   path;
    TreeItem* currentItem = item;
    while ( currentItem->getParent() )
    {
        QString sep   = currentItem->getDepth() - 1 > 0 ? "+ " : "";
        QString space = "";
        for ( int i = 0; i < currentItem->getDepth() - 1; i++ )
        {
            space += "&nbsp; ";
        }
        QString line = space + sep + currentItem->getLabel();
        path        = ( currentItem == item ) ? line : line + "<br>" + path;
        currentItem = currentItem->getParent();
    }

    QString table_css   = "table, th, td { padding: 0px 10px 0px 0px }";
    QString description = "<html><style>" + table_css + "</style><body>";
    description += view->getContextDescription( item );
    description += formatAsTable( list );
    description += "<p><b>" + tr( "Path" ) + "</b><br>" + path + "<br>";
    description += "</body></html>";

    return description;
}

QString
InfoWidget::getDocumentation( TreeItem* item )
{
    QString name = item->getUrl().remove( "@mirror@" );
    if ( name.isEmpty() )
    {
        return "";
    }

    // remove anchor from URL
    int idx = name.indexOf( "#" );
    if ( idx > 0 )
    {
        name.truncate( idx );
    }

    QString doc = "";
    auto    it  = documentCache.find( name );
    if ( it == documentCache.end() )
    {
#ifndef WEB_SOCKETS // todo embedded docs
        // search for embedded documentation
        if ( Globals::cubelibVersion() > CUBELIB_VERSION_CHECK( 4, 6, 0 ) )
        {
            // getMiscData is available for cubelib > 4.6.0
            std::vector<char> data = item->getTree()->getCube()->getMiscData( name.toStdString() );
            if ( data.size() > 0 ) // embedded documentation was found
            {
                doc = embeddedStart;
                for ( char c : data )
                {
                    doc += c;
                }
                documentCache[ name ] = doc;
            }
        }
#endif
        if ( doc.isEmpty() ) // embedded doc is not available -> read from network
        {
            doc = item->getUrl();
        }
    }
    else // cached embedded documentation found
    {
        doc = it->second;
    }
    return doc;
}
