/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include <QNetworkReply>
#include "HtmlWidget.h"
#include "Environment.h"
#include "PluginManager.h"
#include "CubeProxy.h"

#include "HtmlWidgetTextView.h"
#ifdef WITH_WEB_ENGINE
#include "HtmlWidgetWebEngine.h"
#endif

using namespace std;
using namespace cubegui;

HtmlWidget::HtmlWidget()
{
    lastUrl = "";
    connect( &check, SIGNAL( mirrorFound( QUrl ) ), this, SLOT( loadURL( QUrl ) ) );
}


HtmlWidget*
HtmlWidget::createHtmlWidget()
{
#ifdef WITH_WEB_ENGINE
    if ( Globals::optionIsSet( WebEngine ) )
    {
        return new HtmlWidgetWebEngine;
    }
    else
    {
        return new HtmlWidgetTextView;
    }
#else
    return new HtmlWidgetTextView;
#endif
}

static QString
baseUrl( const QString& url )
{
    QString baseUrl = url;
    int     idx     = url.lastIndexOf( '#' );
    if ( idx > 0 )
    {
        baseUrl = url.left( idx );
    }
    return baseUrl;
}

void
HtmlWidget::loadURL( const QUrl& url )
{
    if ( url.isValid() )
    {
        QString urlString = url.toString();
        foreach( QString mirror, mirrorList )
        {
            if ( urlString.contains( mirror ) )
            {
                QString mirrorUrl = baseUrl( urlString );
                mirrorUrl.replace( mirror, "@mirror@" );
                lastMirrors[ mirrorUrl ] = mirror;
                break;
            }
        }
        showUrl( url );
        emit urlLoaded( url );
    }
    else
    {
        emit loadingFailed( url );
        lastUrl.clear(); // if failed, don't cache contents
    }
}

/** Loads the given url. If the string contains the tag "@mirror@", the tag is replaced with each defined mirror until
 * a vaild URL is found
 */
void
HtmlWidget::showUrl( const QString& url )
{
    if ( lastUrl == url )
    {
        emit urlLoaded( url );
        return;
    }
    lastUrl = url;

    if ( url.contains( "@mirror" ) )
    {
        QString mUrl   = url;
        QString mirror = lastMirrors.value( baseUrl( mUrl ) );
        if ( !mirror.isEmpty() )
        {
            mUrl.replace( QString( "@mirror@" ), mirror );
            Globals::debug( "HtmlWidget" ) << "used previous mirror: " << mUrl << Qt::endl;
            showUrl( QUrl( mUrl ) );
            emit urlLoaded( QUrl( mUrl ) );
            return;
        }

        // Check whether HTTP access to online documentation should be disabled
        bool no_http = env_str2bool( getenv( "CUBE_DISABLE_HTTP_DOCS" ) );

        cube::CubeProxy*                    cube    = PluginManager::getInstance()->getCube();
        const std::vector<std::string>&     mirrors = cube->getMirrors();
        vector<std::string>::const_iterator it      = mirrors.begin();

        if ( mirrors.size() == 0 )
        {
            Globals::setStatusMessage( "No documentation mirrors found", MessageType::Error );
            return;
        }

        QList<QUrl> urlList;
        mirrorList.clear();
        while ( it != mirrors.end() )
        {
            QString mirror = QString::fromStdString( *it );
            if ( !mirror.endsWith( "/" ) ) // ensure that separator exists (QUrl)
            {
                mirror += "/";
            }
#ifdef Q_OS_WIN
            if ( !mirror.startsWith( "http://" ) && !mirror.startsWith( "https://" ) ) // ensure valid syntax for local files
            {
                mirror.remove( "file://" );
                mirror = QUrl::fromLocalFile( mirror ).toString();
            }
#endif
            QString tmpUrl = url;
            tmpUrl.replace( QString( "@mirror@" ), mirror );
            if ( no_http && ( tmpUrl.startsWith( "http://" ) || tmpUrl.startsWith( "https://" ) ) )
            {
                ++it;
                Globals::debug( "HtmlWidget" ) << "ignoring mirror because CUBE_DISABLE_HTTP_DOCS is set: " << mirror << Qt::endl;
                continue;
            }
            if ( tmpUrl.trimmed().length() > 0 )
            {
                QUrl    url( tmpUrl );
                QString mirror = QString::fromStdString( *it );
                if ( url.isValid() )
                {
                    urlList.append( url );
                    mirrorList.append( mirror );
                    Globals::debug( "HtmlWidget" ) << "got Url from mirror list: " << mirror << Qt::endl;
                }
                else
                {
                    Globals::debug( "HtmlWidget" ) << "ignoring invalid mirror: " << mirror << Qt::endl;
                }
            }
            ++it;
        }
        if ( urlList.size() == 0 )
        {
            QString status = tr( "No valid documentation mirrors found" );
            if ( no_http && mirrors.size() > 0 )
            {
                status += tr( " and remote mirrors are disabled because CUBE_DISABLE_HTTP_DOCS is set" );
            }
            Globals::setStatusMessage( status, MessageType::Error );
        }
        else
        {
            check.checkMirrors( urlList ); // -> loadURL
        }
    }
    else
    {
        QList<QUrl> urlList;
        urlList.append( QUrl( url ) );
        check.checkMirrors( urlList ); // -> loadURL
    }
}

// =================================================================================================

MirrorCheck::MirrorCheck()
{
    connect( &manager, SIGNAL( finished( QNetworkReply* ) ), this, SLOT( onFinished( QNetworkReply* ) ) );
}

void
MirrorCheck::checkMirrors( const QList<QUrl>& urlList )
{
    mirrorList = urlList;
    foreach( QUrl url, urlList )
    {
        // url differs from previous url only in anchor
        if ( lastValidUrl.isValid() && url.toString( QUrl::RemoveFragment ) == lastValidUrl.toString( QUrl::RemoveFragment ) )
        {
            Globals::debug( "HtmlWidget" ) << "reusing mirror with same base URL: " << url.toString() << Qt::endl;
            emit mirrorFound( url );
            return;
        }
    }
    checkMirrors();
}

void
MirrorCheck::checkMirrors()
{
    if ( mirrorList.size() == 0 )
    {
        return;
    }
    url = mirrorList.takeFirst();

    Globals::debug( "HtmlWidget" ) << "check url: " << url.toString() << Qt::endl;

    // Load data depending on the specified protocol
    QString protocol = url.scheme();
    if ( "http" == protocol || "https" == protocol )
    {
        manager.head( QNetworkRequest( url ) );
    }
    else
    {
        QFile file( url.toLocalFile() );
        if ( QFileInfo( file ).isReadable() )
        {
            emit mirrorFound( url );
        }
        else
        {
            onFinished( nullptr );
        }
    }
}

void
MirrorCheck::onFinished( QNetworkReply* reply )
{
    if ( reply && !reply->error() && reply->attribute( QNetworkRequest::HttpStatusCodeAttribute ).toInt() < 300 )
    {
        Globals::debug( "HtmlWidget" ) << url.toString() << " -> ok" << Qt::endl;
        lastValidUrl = url;
        lastValidUrl.setFragment( "" );
        emit mirrorFound( url );
    }
    else
    {
        if ( reply )
        {
            Globals::debug( "HtmlWidget" ) << "error loading " << url.toString() << " : " << reply->errorString() << Qt::endl;
        }
        else
        {
            Globals::debug( "HtmlWidget" ) << "local file cannot be opened: " << url.toString() << Qt::endl;
        }
        if ( !mirrorList.empty() )
        {
            checkMirrors();
        }
        else
        {
            QString msg     = tr( "Last mirror could not be opened" ) + " : " + url.toString();
            bool    verbose = Globals::optionIsSet( VerboseMode );
            msg += verbose ? "" : ". " + tr( "Start cube with \"-verbose\" for further details" );
            Globals::setStatusMessage( msg, MessageType::Information );

            msg = tr( "Documentation could not be loaded. Click here for further information" );
            Globals::setStatusMessage( msg, MessageType::Error );

            emit mirrorFound( QUrl() );
        }
    }
}
