/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2022                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"

#include "BoxPlot.h"
#include "StatisticalInformation.h"
#include "Globals.h"

#include <fstream>
#include <iostream>
#include <limits>
#include <math.h>
#include <sstream>

#include <QDialog>
#include <QFontMetrics>
#include <QLabel>
#include <QMouseEvent>
#include <QPen>
#include <QPushButton>
#include <QString>
#include <QVBoxLayout>

using namespace std;
using namespace cubegui;

void
BoxPlot::drawBoxPlots( QPainter& painter )
{
    if ( statList.size() == 0 )
    {
        return;
    }

    horizontalPadding = chartWidth / 20;                                          // padding on the left and right border of the drawing area
    boxAreaWidth      = ( chartWidth - 2 * horizontalPadding ) / statList.size(); // width of the area for one boxplot
    boxWidth          = boxWidthPercent * boxAreaWidth / 100;                     // width of one boxplot in each area

    int idx = 0;
    for ( const StatisticPair& stat : statList )
    {
        drawBoxPlot( painter, stat.getCurrent(), getBoxplotCenter( idx++ ), boxWidth );
    }
}

/**
 * @brief BoxPlotNew::drawBoxPlot
 * @param painter
 * @param posX center of the boxplot
 * @param boxWidth width of the boxplot
 */
void
BoxPlot::drawBoxPlot( QPainter& painter, const StatisticalInformation& stat,
                      int posX, int boxWidth )
{
    QColor penColor    = palette().color( QPalette::WindowText );
    QPen   standardPen = QPen( penColor );
    QPen   thickPen( standardPen );
    thickPen.setWidth( 2 );

    // Draw the horizontal lines for minimum and maximum, check range for zooming
    painter.setPen( thickPen );
    if ( stat.getMinimum() >= yMin )
    {
        int posY = getY( stat.getMinimum() );
        painter.drawLine( posX - boxWidth / 4, posY,
                          posX + boxWidth / 4, posY );
    }
    if ( stat.getMaximum() <= yMax )
    {
        int posY = getY( stat.getMaximum() );
        posY = std::max( posY, startY - chartHeight + 1 );     // getY+1 to draw thick line inside the chart area
        painter.drawLine( posX - boxWidth / 4, posY,
                          posX + boxWidth / 4, posY );
    }

    if ( stat.getCount() >= 5 ) // if there are enough values to have Q1 and Q3
    {
        // Draw the connection lines between min/max and Q1/Q3
        QPen pen( standardPen );
        pen.setStyle( Qt::DashLine );
        painter.setPen( pen );

        if ( getY( stat.getMaximum() ) != getY( stat.getQ3() ) )
        {
            painter.drawLine( posX, getY( stat.getMaximum() ), posX, getY( stat.getQ3() ) );
        }
        if ( getY( stat.getMinimum() ) != getY( stat.getQ1() ) )
        {
            painter.drawLine( posX, getY( stat.getMinimum() ), posX, getY( stat.getQ1() ) );
        }

        // Draw the box
        QColor backCol = palette().color( QPalette::Window );
        painter.setPen( standardPen );
        if ( getY( stat.getQ3() ) != getY( stat.getQ1() ) )
        {
            int   boxHeight = getY( stat.getQ1() ) - getY( stat.getQ3() );
            QRect myBox( posX - boxWidth / 2, getY( stat.getQ3() ), boxWidth, boxHeight );
            painter.fillRect( myBox, QBrush( backCol ) );
            painter.drawRect( myBox );
        }
    }

    // draw the median line, if the median position is not Q1 or Q3
    int median = getY( stat.getMedian() );
    if ( ( median != getY( stat.getQ1() ) ) && ( median != getY( stat.getQ3() ) ) )
    {
        thickPen.setWidth( 3 );
        painter.setPen( thickPen );
        painter.drawLine( posX - boxWidth / 2, median,
                          posX + boxWidth / 2, median );
    }

    int mean = getY( stat.getMean() );
    // draw the mean line
    QPen pen( standardPen );
    pen.setStyle( Qt::DashLine );
    painter.setPen( pen );

    painter.drawLine( posX - boxWidth / 2, mean,
                      posX + boxWidth / 2, mean );

    // values are invalid. Paint error message
    if ( !stat.isValid() )
    {
        QPen redPen;
        redPen.setColor( Qt::red );
        painter.setPen( redPen );
        QRect rect = painter.boundingRect( 0, 0, 0, 0, Qt::AlignLeft, "W" ); // get a hight of the line
        for ( int i = 0; i < stat.getDataStatusDescription().size(); i++ )
        {
            painter.drawText( startX, startY + ( rect.height() + 3 ) * ( i + 1 ),
                              stat.getDataStatusDescription().at( i )  );
        }
    }
    painter.setPen( standardPen );
}


std::vector<AxisLabel>
BoxPlot::generateRightAxisValues()
{
    if ( statList.size() == 0 || !statList.front().getCurrent().isValid() )
    {
        return std::vector<AxisLabel>();
    }
    return generateStatisticAxis( statList.front().getCurrent(), statList.front().getAbsolute() );
}

/**
 * @brief BoxPlot::getLowerBorderHeight return the height of the legend at the bottom of the chart
 */
int
BoxPlot::calculateLowerBorderHeight( int chartWidth )
{
    if ( statList.size() <= 1 )
    {
        return Chart::calculateLowerBorderHeight( chartWidth ); // no legend if only one StatisticalInformation exist
    }
    // calculate length of longest label
    string longestName = "";
    for ( StatisticPair info : statList )
    {
        if ( info.getCurrent().getName().length() > longestName.length() )
        {
            longestName = info.getCurrent().getName();
        }
    }
    QFontMetrics fm( font() );
    boxLabelHeight = fm.height() * 3 / 2;
    boxLabelWidth  = fm.boundingRect( QString( "00:" ) + longestName.c_str() ).width();

    boxLabelColumns = chartWidth / boxLabelWidth;
    if ( boxLabelColumns == 0 )
    {
        return Chart::calculateLowerBorderHeight( chartWidth ); // not enough space to draw a legend
    }

    int numberOfRows = ( statList.size() + boxLabelColumns - 1 ) / boxLabelColumns;
    return ( numberOfRows + 1 ) * boxLabelHeight; // one additional row for the ticks
}

QString
BoxPlot::getAreaDescription() const
{
    double                        y1   = getValue( selectedArea.y() + selectedArea.height() );
    double                        y2   = getValue( selectedArea.y() );
    const StatisticalInformation& stat = statList.front().getCurrent();
    return QString::number( stat.countRange( y1, y2 ) ) + " elements / " +  QString::number( stat.getCount() );
}

void
BoxPlot::drawLegend( QPainter& painter )
{
    if ( statList.size() == 1 || boxLabelColumns == 0 )
    {
        return; // only one chart or not enough space to draw a legend
    }

    int idx  = 0;
    int posY = startY + boxLabelHeight;
    for ( StatisticPair info : statList )
    {
        // draw ticks with index as label
        int posX = getBoxplotCenter( idx );
        painter.drawLine( posX, startY, posX, startY + tickWidth );
        QRect textBox( posX - boxLabelWidth / 2, startY, boxLabelWidth, boxLabelHeight );
        painter.drawText( textBox, Qt::AlignHCenter | Qt::AlignBottom, QString::number( idx + 1 ) );

        // draw description text
        int   width = chartWidth / boxLabelColumns;
        QRect labelBox( startX + ( idx % boxLabelColumns ) * width, posY, boxLabelWidth, boxLabelHeight );
        painter.drawText( labelBox, Qt::AlignLeft | Qt::AlignVCenter, QString::number( idx + 1 ) + ": " + info.getCurrent().getName().c_str() );

        if ( idx % boxLabelColumns == boxLabelColumns - 1 )
        {
            posY += boxLabelHeight;
        }
        idx++;
    }
}

void
BoxPlot::addStatistics( const StatisticalInformation& statistics )
{
    statList.push_back( StatisticPair( statistics ) );
    resize( 150 + 80 * statList.size(), 450 );
    if ( this->parentWidget() )
    {
        parentWidget()->adjustSize();
    }
}

void
BoxPlot::setStatistics( const StatisticPair& stat )
{
    statList.clear();
    statList.push_back( stat );
    setYRange( stat.getCurrent().getMinimum(), stat.getCurrent().getMaximum() );
}

void
BoxPlot::drawChart( QPainter& painter )
{
    drawBoxPlots( painter );
    drawLegend( painter );
}

/** returns the x position of the center of the boxplot with the given index (starting with 0) */
int
BoxPlot::getBoxplotCenter( int index ) const
{
    return startX + horizontalPadding + boxAreaWidth / 2 + index * boxAreaWidth;
}

const StatisticPair*
BoxPlot::getStatistics( int posX ) const
{
    int index = ( posX - startX - horizontalPadding ) / boxAreaWidth;

    if ( ( index >= 0 ) && ( ( unsigned )index < statList.size() ) )
    {
        int center = getBoxplotCenter( index );
        if ( std::abs( center - posX ) < boxWidth / 2 )
        {
            return &statList[ index ];
        }
    }
    return nullptr;
}

void
BoxPlot::mousePressEvent( QMouseEvent* event )
{
    Chart::mousePressEvent( event );
    if ( statList.size() == 0 )
    {
        return;
    }
    else if ( event->button() == Qt::RightButton )
    {
        const StatisticPair* selected = getStatistics( event->pos().x() );
        if ( selected )
        {
            QPoint mousePos = this->mapToGlobal( event->pos() );
            tooltip = BoxPlot::showStatisticToolTip( this, mousePos, selected->toHtml() );
        }
    }
    mousePressPos = event->pos();
}

void
BoxPlot::mouseReleaseEvent( QMouseEvent* event )
{
    Chart::mouseReleaseEvent( event );

    if ( statList.size() == 0 )
    {
        return;
    }
    if ( event->button() == Qt::LeftButton )
    {
        // left mouse click inside box -> show statistic window
        const int minimumRange = 5;  // minimum number of pixels for range selection
        if ( std::abs( event->pos().y() - mousePressPos.y() ) < minimumRange )
        {
            const StatisticPair* selected = getStatistics( event->pos().x() );
            if ( selected )
            {
                Chart::showStatisticWindow( this, tr( "Statistics info" ),
                                            selected->toHtml(), infoDialog );
            }
        }
    }
    if ( tooltip )
    {
        tooltip->close();
        delete tooltip;
        tooltip = nullptr;
    }
}
