/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2021-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"

#include "CallTreeLabelDialog.h"
#include "CubeCnode.h"
#include "CubeRegion.h"
#include "CubeVertex.h"
#include "DefaultCallTree.h"
#include "Globals.h"
#include "TabManager.h"
#include "TreeConfig.h"
#include "Tree.h"
#include "TreeItem.h"
#include "TreeModelProxy.h"
#include "TreeView.h"

#ifdef CXXABI_H_AVAILABLE
#include <cxxabi.h>
#else
#ifdef DEMANGLE_H_AVAILABLE
#define HAVE_DECL_BASENAME 1
#include <demangle.h>
#endif
#endif

#include "Compatibility.h"
#ifdef HAS_QREGULAR_EXPRESSION
#include <QRegularExpression>
#else
#include <QRegExp>
#endif

using namespace cubegui;

TreeConfig* TreeConfig::single = 0;

TreeConfig::TreeConfig()
{
}


TreeConfig*
TreeConfig::getInstance()
{
    return single ? single : single = new TreeConfig();
}

QMenu*
TreeConfig::getMenu()
{
    QMenu* treeMenu = new QMenu( tr( "Trees" ) );
    treeMenu->setStatusTip( tr( "Ready" ) );
    treeMenu->setEnabled( false );

    QAction* markerAction = PluginManager::getInstance()->getMarkerConfigAction();
    treeMenu->addAction( markerAction );

    treeMenu->addSeparator(); // -- call tree or call flat specific
    demangleAction = new QAction( tr( "Demangle Function Names" ), this );
    demangleAction->setCheckable( true );
    treeMenu->addAction( demangleAction );
    connect( demangleAction, &QAction::toggled, this, &TreeConfig::enableDemangling );
#if defined( CXXABI_H_AVAILABLE ) || defined( DEMANGLE_H_AVAILABLE )
    demangleAction->setEnabled( true );
#endif

    QAction* labelAction = new QAction( tr( "Shorten Function Names..." ), this );
    treeMenu->addAction( labelAction );
    connect( labelAction, &QAction::triggered, this, &TreeConfig::configureCallTreeLabel );

    treeMenu->addSeparator(); // system tree specific
    systemLabelAction = new QAction( tr( "Append rank to system tree items" ), this );
    systemLabelAction->setCheckable( true );
    treeMenu->addAction( systemLabelAction );
    connect( systemLabelAction, &QAction::toggled, this, &TreeConfig::enableSystemLabelRank );
    systemHideAction = new QAction( tr( "Hide unvisited locations" ), this );
    systemHideAction->setStatusTip( tr( "Hides locations in the system tree that do not contain a region. These locations can be generated by OTF2 or CUDA and don't provide the user with any information." ) );
    systemHideAction->setWhatsThis( systemHideAction->statusTip() );
    systemHideAction->setCheckable( true );
    treeMenu->addAction( systemHideAction );
    connect( systemHideAction, &QAction::toggled, this, &TreeConfig::hideUnvisitedLocations );

    return treeMenu;
}

void
TreeConfig::enableDemangling( bool enable )
{
    callConfig.demangleFunctions = enable;
    updateCallTreeLabel();
}

void
TreeConfig::enableSystemLabelRank( bool enable )
{
    appendSystemRank = enable;
    Tree* tree = Globals::getTabManager()->getActiveTree( SYSTEM );
    if ( !tree )
    {
        return;
    }

    tree->invalidateItemDisplayNames();
    tree = Globals::getTabManager()->getActiveTree( SYSTEM );
    tree->invalidateItemDisplayNames();
    Globals::getTabManager()->updateTreeItemProperties();
}

bool
TreeConfig::isHidingOfUnvisitedEnabled() const
{
    return unvisitedLocationsHidden;
}

void
TreeConfig::hideUnvisitedLocations( bool hidden )
{
    unvisitedLocationsHidden = hidden;
    Tree* tree =  Globals::getTabManager()->getActiveTree( SYSTEM );
    if ( tree )
    {
        tree->setPredefinedHiddenItems();
        TreeView*       view  = static_cast<TreeView*> ( Globals::getTabManager()->getActiveView( SYSTEM ) );
        TreeModelProxy* model = dynamic_cast<TreeModelProxy*> ( view->model() );
        if ( model )
        {
            model->invalidate();
        }
    }
}

void
TreeConfig::updateCallDisplayConfig( const CallDisplayConfig& newConfig )
{
    callConfig = newConfig;
    updateCallTreeLabel();
}

void
TreeConfig::configureCallTreeLabel()
{
    CallTreeLabelDialog* dialog = new CallTreeLabelDialog( callConfig, Globals::getMainWindow() );
    connect( dialog, &CallTreeLabelDialog::configChanged, this, &TreeConfig::updateCallDisplayConfig );
    dialog->setVisible( true );
}

// invalidates the displayName of all call tree and call flat tree items
void
TreeConfig::updateCallTreeLabel()
{
    for ( Tree* tree : Globals::getTabManager()->getTrees() )
    {
        tree->invalidateItemDisplayNames();
    }
    Globals::getTabManager()->updateTreeItemProperties();
}

void
TreeConfig::saveGlobalSettings( QSettings& settings )
{
    settings.setValue( "calltree/hideArguments", callConfig.hideArguments );
    settings.setValue( "calltree/hideReturnValue", callConfig.hideReturnValue );
    settings.setValue( "calltree/hideClassHierarchy", callConfig.hideClassHierarchy );
    settings.setValue( "calltree/hideModules", callConfig.hideModules );
    settings.setValue( "calltree/enableDemangling", callConfig.demangleFunctions );
    settings.setValue( "systemtree/appendRank", appendSystemRank );
    settings.setValue( "systemtree/hideUnvisitedLocations", unvisitedLocationsHidden );
}

void
TreeConfig::loadGlobalSettings( QSettings& settings )
{
    callConfig.hideArguments      = settings.value( "calltree/hideArguments", false ).toBool();
    callConfig.hideReturnValue    = settings.value( "calltree/hideReturnValue", false ).toBool();
    callConfig.hideClassHierarchy = settings.value( "calltree/hideClassHierarchy", 0 ).toInt();
    callConfig.hideModules        = settings.value( "calltree/hideModules", false ).toBool();
    callConfig.demangleFunctions  = settings.value( "calltree/enableDemangling", true ).toBool();
    appendSystemRank              = settings.value( "systemtree/appendRank", false ).toBool();
    unvisitedLocationsHidden      = settings.value( "systemtree/hideUnvisitedLocations", false ).toBool();

    demangleAction->setChecked( callConfig.demangleFunctions );
    systemLabelAction->setChecked( appendSystemRank );
    systemHideAction->setChecked( unvisitedLocationsHidden );
}

/**
 * @brief TreeItem::createDisplayName
 * @param any tree item
 * @return the name of the tree item adapted to the user settings, e.g. demangling, ...
 */
QString
TreeConfig::createDisplayName( TreeItem* item )
{
    if ( item->getDisplayType() == CALL )
    {
        return createCallDisplayName( item );
    }
    else if ( item->getDisplayType() == SYSTEM )
    {
        return createSystemDisplayName( item );
    }
    else
    {
        return item->getOriginalName();
    }
}

QString
TreeConfig::createSystemDisplayName( TreeItem* item )
{
    if ( !appendSystemRank || item->getType() != LOCATIONITEM )
    {
        return item->getOriginalName();
    }
    TreeItem* parent = item->getParent();
    if ( parent && parent->getType() == LOCATIONGROUPITEM )
    {
        const QString  mpi  = "MPI Rank ";
        const QString& name = parent->getOriginalName();
        if ( name.startsWith( mpi ) )
        {
            QString number = name.right( name.size() - mpi.size() );
            return QString().append( item->getOriginalName() ).append( ":" ).append( number );
        }
    }
    else
    {
        return ""; // tree is not yet initialised
    }
    return item->getOriginalName();
}

/*
 * remove the namespace/class part of the name
 * removes all "::name" from orig but keeps the last keepLevel
 */
static QString
removeNamespace( const QString& orig, int keepLevel )
{
    QString shortName = orig;

    int lastColon = shortName.lastIndexOf( "::", -1 );
    while ( keepLevel >= 0 && lastColon > 0 )
    {
        int index      = shortName.length();
        int colon      = lastColon;
        int colonCount = 0;
        while ( colon > 0 )
        {
            QChar c;
            index = colon - 1;
            do // search left for next char that isn't part of an identifier
            {
                c = shortName.at( index );
                index--;
            }
            while ( index > 0 && ( c.isLetterOrNumber() || c == '_' ) );

            if ( index == 0 )
            {
                colon = -1;
            }
            else if ( c == ':' && shortName.at( index ) == ':' ) // found next :: on the left
            {
                colon = index;
            }
            else
            {
                colon  = -1;
                index += 2;
            }
            if ( ++colonCount <= keepLevel ) // show the first level classes/namespaces
            {
                lastColon = colon;
            }
        }
        if ( lastColon > 0 )
        {
            shortName.remove( index, lastColon - index + 2 );
            lastColon = shortName.lastIndexOf( "::", index - shortName.length() );
        }
    }
    return shortName;
}


/** @brief Removes every text between begin and end character, which may be nested.
 *  @returns the string without the text between begin and end character.
 */
static QString
removeString( const QString& str, char beginChar, char endChar )
{
    QString shortName = str;
    int     startIdx  = shortName.indexOf( beginChar );
    while ( startIdx >= 0 )
    {
        int count  = 1;
        int endIdx = startIdx;
        while ( count > 0 && ++endIdx < shortName.length() )
        {
            QChar c = shortName.at( endIdx );
            if ( c == beginChar )
            {
                count++;
            }
            else if ( c == endChar )
            {
                count--;
            }
        }
        if ( count == 0 )
        {
            shortName.remove( startIdx, endIdx - startIdx + 1 );
            startIdx = shortName.indexOf( beginChar );
        }
        else // brackets are not matching -> abort
        {
            startIdx = -1;
        }
    }
    return shortName;
}

/**
 * @brief TreeItem::createCallDisplayName
 * changes the original call tree/flat tree label acoording to the settings in CallDisplayConfig
 * - demangled c++ function names
 * - C++ function names without argument list, return value, templates
 * - C++ function names with limited class/namespace hierarchy
 * - fortran subroutines without module names
 * @returns the name that will be displayed in the tree view
 */
QString
TreeConfig::createCallDisplayName( TreeItem* item )
{
    QString displayName = item->getOriginalName();

    CallDisplayConfig& config     = callConfig;
    cube::Vertex*      cubeObject = item->getCubeObject();

    if ( config.demangleFunctions )
    {
        std::string dName, mName;
        if ( item->getType() == CALLITEM )
        {
            cube::Cnode* cnode = item->getCnode();
            if ( cnode && cnode->get_callee() )
            {
                dName = cnode->get_callee()->get_name();
                mName = cnode->get_callee()->get_mangled_name();
            }
        }
        else if ( item->getType() == REGIONITEM )
        {
            cube::Region* region = static_cast<cube::Region* > ( cubeObject );
            if ( region )
            {
                dName = region->get_name();
                mName = region->get_mangled_name();
            }
        }
        bool isMangled = ( dName.rfind( "_Z", 0 ) == 0 ) || ( dName.rfind( "__Z", 0 ) );
        if ( isMangled && dName == mName )
        {
#ifdef CXXABI_H_AVAILABLE
            int   status;
            char* demangled = abi::__cxa_demangle( dName.c_str(), 0, 0, &status );
            if ( status == 0 )
            {
                displayName = demangled;
                free( demangled );
            }
#else
#ifdef DEMANGLE_H_AVAILABLE
            char* demangled = cplus_demangle( dName.c_str(), DMGL_PARAMS );
            if ( demangled )
            {
                displayName = demangled;
                free( demangled );
            }
#endif
#endif
        }
    }

    bool shorten = true;

    if ( config.hideClassHierarchy == -1 && !config.hideArguments && !config.hideReturnValue && !config.hideModules )
    {
        shorten = false;
    }

    if ( cubeObject ) // cubeObject is null for iterations
    {                 // check, if region should be ignored because it isn't user code
        cube::Cnode*  cnode  = item->getCnode();
        cube::Region* region = 0;
        if ( cnode )
        {
            region = dynamic_cast<cube::Region*>( cnode->get_callee() );
        }
        if ( region && region->get_paradigm() == "unknown" ) // old format, paradigm isn't set
        {
            if ( displayName.contains( "!$omp" ) )
            {
                shorten = false;
            }
        }
        else if ( region && region->get_paradigm() != "compiler" && region->get_paradigm() != "cuda" )
        {
            shorten = false; // only shorten regions with compiler and cuda tag, ignore e.g. !$omp
        }
    }

    if ( shorten )
    {
        QString shortName = displayName;
        shortName.remove( QRegularExpression( "\\[with(.*)\\]" ) ); // e.g. "FT getValue(const DictionaryDatum&, Name) [with FT = lockPTRDatum...]"

        //========= for C/C++ subroutines =========
        int argIdx = -1; // starting position of the argument list, -1 => not found

        QString op    = "operator(";
        int     start = displayName.indexOf( op );
        start = ( start > 0 ) ? start + op.length() : 0; // "operator(" is not start of a function -> search next '('

        int idx = shortName.indexOf( '(', start );
        if ( ( idx > 0 ) && ( idx > displayName.indexOf( "<" ) ) && ( idx < displayName.indexOf( ">" ) ) )
        {
            idx = shortName.indexOf( '(', displayName.indexOf( ">" ) ); // if '(' is inside a template -> ignore
        }
        if ( ( idx > 0 ) && ( shortName.indexOf( '(', idx + 1 ) == -1 ) )
        {
            // simplified: if only one parenthesis exist -> start of argument list found
            argIdx = idx;
        }
        if ( argIdx != -1 )             // argument list found
        {
            if ( config.hideArguments ) // don't show argument list, ignore everything right of the '('
            {
                shortName = displayName.left( argIdx );
            }

            if ( config.hideReturnValue )
            {
                QString tmp = removeString( shortName, '(', ')' );
                tmp = removeString( tmp, '<', '>' );
                QStringList elements = tmp.split( " ", SkipEmptyParts );
                if ( elements.size() > 1 ) // assume that first part is the return value
                {
                    shortName.remove( elements.first() );
                }
            }
        }

        if ( config.hideClassHierarchy > -1 )
        {
            shortName = removeString( shortName, '<', '>' ); // remove template
            shortName = removeNamespace( shortName, config.hideClassHierarchy );
        }

        //========= shorten Fortran subroutines if not already shortened as C++ function =========
        if ( config.hideModules && ( shortName.length() == displayName.length() ) )
        {
            idx = displayName.indexOf( '@' ); // e.g. !$omp parallel @jacobi.F90:58
            if ( idx < 0 )
            {
                // Intel ifort: remove "<modulename>." prefix
                // Ensure that the prefix is not a number because kernel parameters may contain double numbers
                QRegularExpression      regex( "([a-zA-Z]+[a-zA-Z0-9_]*\\.)[^0-9]+" );
                QRegularExpressionMatch match = regex.match( displayName );
                if ( match.hasMatch() )                     // remove module name from "<modulename>.name" (Intel ifort)
                {
                    QString captured = match.captured( 1 ); // Get the part matched by the first capture group
                    shortName.remove( captured );           // Remove the captured part from the string
                }
                else
                {
                    // IBM XLF: "__<modulename>_NMOD_" prefix
                    idx = displayName.indexOf( "_NMOD_" );
                    if ( idx > 0 )
                    {
                        shortName.remove( 0, idx + 6 );
                    }
                }
            }
        }
        displayName = shortName;
    } // shorten

    if ( item->getDisplayType() == CALL )
    {
        DefaultCallTree* defaultCallTree = dynamic_cast<DefaultCallTree*> ( item->getTree() );
        if ( defaultCallTree && item == defaultCallTree->getLoopRootItem() ) // non-aggreated loop root
        {
            displayName.append( tr( " [loop]" ) );
        }
    }

    return displayName;
}
