/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#include "config.h"

#include "Tree.h"
#include "TreeModel.h"
#include "TreeView.h"

#include "Future.h"
#include "CubeProxy.h"
#include "CubeMetric.h"
#include "CubeCnode.h"
#include "CubeRegion.h"
#include "CubeMachine.h"
#include "CubeNode.h"
#include "CubeProcess.h"
#include "CubeThread.h"
#include "CubeVertex.h"
#include "CubeError.h"
#include "CubeGeneralEvaluation.h"
#include "CubeIdIndexMap.h"
#include "Globals.h"
#include "MetricTree.h"
#include "TreeStatistics.h"
#include "TreeItem.h"

using namespace std;
using namespace cube;
using namespace cubegui;

Tree::Tree( CubeProxy* cubeProxy ) : cube( cubeProxy )
{
    top           = nullptr;
    lastSelection = nullptr;
    maxValue      = 0.;
    maxDepth      = -1;
    integerType   = false;
    active        = true;
    valueModus    = ABSOLUTE_VALUES;
    treeModel     = new TreeModel( this );
    activeFilter  = FILTER_NONE;
    isValid       = true;
    toCalculate   = !Globals::optionIsSet( ManualCalculation );

    calculateInvisible       = CalculateInvisible::None;
    selectionSummary         = nullptr;
    selectionRootSummary     = nullptr;
    invalidateItemsPreCalc   = true;
    userDefinedMinMaxValues  = false;
    secondRunRequired        = false;
    valueModusReferenceValue = 0.;

    statistics = new TreeStatistics( this );
}

Tree::~Tree()
{
    for ( auto& elem : treeItemHash )
    {
        TreeItem* item = elem.second;
        delete item;
    }
    delete selectionSummary;
    delete selectionRootSummary;
    delete treeModel;
    delete top;
}

void
Tree::initialize()
{
    if ( top ) // reinit tree
    {
        delete top;
        top = nullptr;
    }

    top = createTree();
    top->setExpandedStatus( true );

    // create artificial tree item to add up selected items
    selectionSummary               = new TreeItem( this, "SumSel", top->getType(), nullptr );
    selectionSummary->rootItem     = top;
    selectionRootSummary           = new TreeItem( this, "SumRoot", top->getType(), nullptr );
    selectionRootSummary->rootItem = top;

    updateItemList();
}

QString
Tree::getLabel() const
{
    return label;
}

// update ordered list of all tree items of the tree and set visible root item for each element
void
Tree::updateItemList()
{
    treeItems.clear();
    foreach( TreeItem * rootItem, top->getChildren() )
    {
        QStack<TreeItem*> stack;
        stack.append( rootItem );
        while ( !stack.isEmpty() )
        {
            TreeItem* item = stack.pop();
            item->rootItem = rootItem;
            treeItems.append( item );

            const QList<TreeItem*>& children = item->getChildren();
            for ( auto i = children.size() - 1; i >= 0; i-- )
            {
                TreeItem* child = children[ i ];
                stack.push( child );
            }
        }
    }
}

/**
 * @brief Tree::updateSelectionSummary updates selectionSummary and SelectionRootSummary
 * If multiple items are selected, the aggregated value of the selections is calculated in calulateValues(). If only
 * one item is selected, the value can be copied from the selected item
 */
void
Tree::updateSelectionSummary()
{
    if ( selectionList.size() == 1 )
    {
        // selectionSummary is not calculated and value can be copied from the selected item
        if ( lastSelection && lastSelection->getValueObject() )
        {
            selectionSummary->setValue( lastSelection->getValueObject()->copy() );
            selectionSummary->maxValue = lastSelection->maxValue;
        }
        // selectionRootSummary is not calculated and value can be copied from the root of the selected item.
        // The inclusive values of the root items are always calculated
        if ( lastSelection && lastSelection->rootItem && lastSelection->rootItem->getTotalValueObject() )
        {
            selectionRootSummary->setValue( lastSelection->rootItem->getTotalValueObject()->copy() );
        }
    }
}

/**
 * @brief Tree::calculateSelectedValues
 * reasons to re-calculate selected values:
 * - multiple selections
 * - hidden children in selected items, if filtering has been enabled in this step
 * @param leftTrees
 * @param rightTrees */
void
Tree::calculateSelectedValues( const QList<Tree*>& leftTrees,
                               const QList<Tree*>& rightTrees )
{
    this->leftTrees  = leftTrees;
    this->rightTrees = rightTrees;

    selectionSummary->invalidate();
    selectionRootSummary->invalidate();

    // restart filtering, if uncalculated items exist (e.g. selected item is expanded)
    QList<TreeItem*> items = getItemsToCalculate();
    secondRunRequired = ( activeFilter != Tree::FILTER_NONE ) && ( items.size() > 0 );
    calculateValuesIntern( items );
}

void
Tree::calculateValues( const QList<Tree*>&     leftTrees,
                       const QList<Tree*>&     rightTrees,
                       const QList<TreeItem*>& itemsToCalculate
                       )
{
    secondRunRequired = activeFilter != Tree::FILTER_NONE;
    this->leftTrees   = leftTrees;
    this->rightTrees  = rightTrees;

    // any tree on the left is invalid -> current tree is invalid too -> abort
    isValid = true;
    for ( Tree* tree : leftTrees )
    {
        if ( !tree->isValidSelection() )
        {
            isValid = false;
            invalidateItemsNow();
            emit calculationFinished();
            return;
        }
    }

    // all values have to be invalidated and recalculated because the selection of the left tree has been changed
    if ( invalidateItemsPreCalc )
    {
        invalidateItemsNow();
        invalidateItemsPreCalc = false;
    }

    // set integerType of the tree to true, if all selected metrics have integer type
    integerType = false;
    foreach( Tree * tree, leftTrees )
    {
        if ( tree->getType() == METRIC )
        {
            integerType = true;
            const QList<TreeItem*>& selected = tree->getSelectionList();
            foreach( TreeItem * item, selected )
            {
                if ( !static_cast<cube::Metric*> ( item->getCubeObject() )->isIntegerType() )
                {
                    integerType = false;
                    break;
                }
            }
        }
    }

    QList<TreeItem*> items = itemsToCalculate.size() == 0 ? getItemsToCalculate() : itemsToCalculate;

    if ( ( getFilter() == Tree::FILTER_DYNAMIC ) && ( items.size() > 0 ) )
    {
        // unhide items before dynamic filtering is started or if it is diabled
        // if no items to calculate are found, the items to hide haven't changed
        unhideItems();
    }
    calculateValuesIntern( items );
}

QList<Task*>
Tree::generateTasks( const QList<TreeItem*>& itemsToCalculate )
{
    QList<Task*> workerData;
    for ( TreeItem* item : itemsToCalculate )
    {
        if ( !item->isCalculated() )
        {
            workerData += setBasicValues( leftTrees, rightTrees, { item } );
        }
    }
    return workerData;
}

void
Tree::calculateInvisibleValues( CalculateInvisible type )
{
    calculateInvisible = type;
    this->calculateValuesIntern();
}

/**
 * @brief getAdditionalTasks checks for all visible tree items, if an additonal
 * exclusive or inclusive value (see CalculateInvisible) should be calculated.
 * If so, the corresponding task is created and added the the returned list.
 * These values are not displayed, but required for sorting by exclusive/inclusive value.
 * @return list of tasks to be calculated
 */
QList<Task*>
Tree::getAdditionalTasks()
{
    QList<Task*> tasks;

    if ( this->isManualCalculation() || calculateInvisible == CalculateInvisible::None )
    {
        return tasks;
    }

    QList<TreeItem*> toCalculate;
    // get visible tree items
    QStack<TreeItem*> stack;
    stack.push( this->getRootItem() );
    QList<TreeItem*> visibleItems;
    while ( !stack.isEmpty() )
    {
        TreeItem* parent = stack.pop();
        if ( parent != this->getRootItem() )
        {
            visibleItems.append( parent );
        }
        if ( parent->isExpanded() )
        {
            for ( TreeItem* child : parent->getChildren() )
            {
                stack.append( child );
            }
        }
    }
    if ( calculateInvisible == CalculateInvisible::Exclusive )
    {
        // calculate exclusive values of collapsed tree items (inclusive value is the visible one)
        for ( TreeItem* item : visibleItems )
        {
            if ( !item->isLeaf() && !item->isExpanded() && !item->isCalculated( ItemState::Exclusive ) )
            {
                toCalculate.append( item );
            }
        }
    }
    else if ( calculateInvisible == CalculateInvisible::Inclusive )
    {
        // calculate inclusive values of expanded tree items (exclusive value is the visible one)
        for ( TreeItem* item : visibleItems )
        {
            if ( item->isExpanded() && !item->isCalculated( ItemState::Inclusive ) )
            {
                toCalculate.append( item );
            }
        }
    }

    QList<bool> currentState;
    bool        expanded = ( calculateInvisible == CalculateInvisible::Exclusive );
    // temporary set desired item state to create corresponding task
    std::for_each( toCalculate.begin(), toCalculate.end(), [ &currentState, expanded ]( TreeItem* i ){
        currentState.push_back( i->isExpanded() );
        i->expanded = expanded;
    } );
    tasks += generateTasks( toCalculate );
    std::for_each( toCalculate.begin(), toCalculate.end(), [ &currentState ]( TreeItem* i ){
        i->expanded = currentState.takeFirst();
    } );

    return tasks;
}

void
Tree::calculateValuesIntern( const QList<TreeItem*>& itemsToCalculate )
{
    calculationIsFinished = false;
    if ( getSelectionList().isEmpty() )
    {
        return;
    }
    selectionSummary->rootItem = getSelectionList().first()->rootItem; // todo: "own root percent" for multiple selections

    Globals::setStatusMessage( tr( "Calculating " ) + this->getLabel() + tr( " values ..." ) );

    QList<Task*> workerData;

    // Ensure that the maximum value of the tree can be calculated. This is required to determine the icon color and
    // to show the maximum value in the value view. The systemtree is always calculated completely.
    // - for non-derived metrics, the maximum value is the top level inclusive value
    // - for derived metrics, the relative maximum is calculated instead
    if ( getType() != SYSTEM )
    {
        QList<TreeItem*> inclusiveItemsToCalculate;

        // find metricTree on the left of the current tree
        auto it = std::find_if( leftTrees.begin(), leftTrees.end(), [ ]( Tree* tree ) {
            return tree->getTreeType() == METRICTREE;
        } );
        Tree* metricTree = ( it == leftTrees.end() ) ?  nullptr : *it;

        // special case derived metrics: to calculate absolute maximum, all values would have to be calculated which would be too slow
        // -> calculate relative maximum: find maximum of all visible items including their inclusive values.
        bool isDerivedMetric = metricTree && metricTree->getLastSelection()->isDerivedMetric();
        // The inclusive values of all expanded items are also required for filtering
        bool dynamicFiltering = this->getFilter() == Tree::FILTER_DYNAMIC;

        if ( dynamicFiltering || isDerivedMetric ) // calculate inclusive values of all expanded items
        {
            QList<TreeItem*> list;
            list.append( top );
            while ( !list.isEmpty() )
            {
                TreeItem* item = list.takeLast();
                foreach( TreeItem * child, item->getChildren() )
                {
                    if ( child->isExpanded() )
                    {
                        if ( child->getTotalValueObject() == nullptr )
                        {
                            inclusiveItemsToCalculate.append( child );
                        }
                        list.append( child );
                    }
                }
            }
        }
        else // default case: maximum value = top level inclusive value
        {
            for ( TreeItem* item : top->getChildren() )
            {
                if ( item->isExpanded() )
                {
                    inclusiveItemsToCalculate.append( item );
                }
            }
        }

        if ( !this->isManualCalculation() )
        {
            // handle items whose inclusive value should be calculated to determine the maximum or relative maximum
            std::for_each( inclusiveItemsToCalculate.begin(), inclusiveItemsToCalculate.end(), [ ]( TreeItem* i ){
                i->expanded = false;
            } );
            workerData += generateTasks( inclusiveItemsToCalculate );
            std::for_each( inclusiveItemsToCalculate.begin(), inclusiveItemsToCalculate.end(), [ ]( TreeItem* i ){
                i->expanded = true;
            } );
        }
    } // getType() != SYSTEM

    workerData += getAdditionalTasks(); // calculate values required for sorting

    QList<TreeItem*> items = itemsToCalculate.size() == 0 ? getItemsToCalculate() : itemsToCalculate;
    workerData += setBasicValues( leftTrees, rightTrees, items );
    if ( workerData.size() > 0 )
    {
        connect( future, SIGNAL( calculationFinished() ), this, SLOT( calculationFinishedSlot() ) );
        future->addCalculations( workerData );
        future->startCalculation( true );

#ifdef CUBE_CONCURRENT_LIB
        emit calculationStarted( future );
#endif
    }
    else // no items to calculate
    {
        secondRunRequired = false;
        calculationFinishedSlot(); // update selection
    }
}

void
Tree::invalidateItemsNow()
{
    selectionSummary->invalidate();
    selectionRootSummary->invalidate();
    foreach( TreeItem * item, this->getItems() )
    {
        item->invalidate();
        if ( activeFilter == FILTER_DYNAMIC )
        {
            setHidden( item, false ); // hidden items aren't recalculated -> ensure recalculation of all items for dynamic filtering
        }
    }
}

/**
 * clear data after work started in computeAllValues() is done
 * update the view
 */
void
Tree::calculationFinishedSlot()
{
    calculationIsFinished = true;
    disconnect( future, SIGNAL( calculationFinished() ), this, SLOT( calculationFinishedSlot() ) );

    updateItems();             // update tree item values

    if ( ( activeFilter == Tree::FILTER_NONE ) || ( !secondRunRequired )  )
    {
        emit calculationFinished();  // finished -> update all view elements
    }
    else // 2nd run is required to calculate parents with hidden children
    {
        if ( activeFilter == Tree::FILTER_DYNAMIC )
        {
            computeMaxValues();
            hideMinorValues( filterThreshold ); // values and max have been calculated -> find values, that should be hidden
        }
        secondRunRequired = false;
        calculateValuesIntern();
    }
}

TreeModel*
Tree::getModel() const
{
    return treeModel;
}

void
Tree::setTrees( QList<Tree*> trees )
{
    treeList = trees;
}

bool
Tree::hasIntegerType() const
{
    if ( getValueModus() != ABSOLUTE_VALUES )
    {
        return false;
    }
    else
    {
        return integerType;
    }
}

const QList<TreeItem*>&
Tree::getItems() const
{
    return treeItems;
}

bool
Tree::itemIsValid( cube::Vertex* )
{
    return true;
}

TreeItem*
Tree::getRootItem() const
{
    return top;
}

DisplayType
Tree::getType() const
{
    return displayType;
}

TreeType
Tree::getTreeType() const
{
    return treeType;
}

double
Tree::getValueModusReferenceValue() const
{
    return valueModusReferenceValue;
}

void
Tree::valueModusChanged( ValueModus modus )
{
    this->valueModus = modus;
    this->invalidateItemLabel();
    updateItems();
}


Tree*
Tree::getActiveTree( DisplayType type ) const
{
    for ( Tree* tree: treeList )
    {
        if ( tree->getType() == type && tree->isActive() )
        {
            return tree;
        }
    }
    return nullptr;
}

void
Tree::setActive()
{
    for ( Tree* tree: treeList )
    {
        if ( tree->getType() == this->displayType )
        {
            tree->active = ( tree == this );
        }
    }
}

bool
Tree::isActive() const
{
    return active;
}

void
Tree::expandItem( TreeItem* item, bool expand )
{
    emit itemExpanded( item, expand );
}

void
Tree::selectItem( TreeItem* item, bool addToSelection )
{
    emit itemSelected( item, addToSelection );
}

void
Tree::deselectItem( TreeItem* item )
{
    emit itemDeselected( item );
}

TreeItem*
Tree::getLastSelection() const
{
    return lastSelection;
}

void
Tree::setLastSelection( TreeItem* value )
{
    lastSelection = value;
}

const QList<TreeItem*>&
Tree::getSelectionList() const
{
    return selectionList;
}

/** deselects all selected items and selects all items in the given list */
void
Tree::setSelectionList( const QList<TreeItem*>& list )
{
    foreach( TreeItem * item, selectionList )
    {
        item->setSelectionStatus( false );
    }

    selectionList = list;
    foreach( TreeItem * item, selectionList )
    {
        item->setSelectionStatus( true );
    }
}

cube::CubeProxy*
Tree::getCube() const
{
    return cube;
}

/*
 * updates the tree item values and their label after calculation has been done
 */
void
Tree::updateItems( bool calculationFinished )
{
    bool maximumIsCalculated = false;
    if ( calculationFinished )
    {
        valueModusReferenceValue = computeReferenceValue( valueModus );
        computeMaxValues();
        maximumIsCalculated = true;
    }

    /* update the color and the label of the visible (expanded) items */
    QList<TreeItem*> list;
    list.append( top->getChildren() );

    while ( !list.isEmpty() )
    {
        TreeItem* item = list.takeFirst();
        item->updateItem( maximumIsCalculated );
        // Only update expanded items, only system tree has to update all its values as they are used in the topologies.
        // All system tree values are calculated at once, so no further call to this method is required if items are expanded.
        if ( item->isExpanded() || displayType == SYSTEM )
        {
            list.append( item->getChildren() );
        }
    }

    updateSelectionSummary(); // update selection and selected roots
    statistics->update();     // update statistics for value view
}

void
Tree::invalidateItems()
{
    invalidateItemsPreCalc = true;
    statistics->invalidate();
}

void
Tree::invalidateItemLabel()
{
    foreach( TreeItem * item, this->getItems() )
    {
        item->invalidateLabel();
    }
}

void
Tree::invalidateItemDisplayNames()
{
    foreach( TreeItem * item, this->getItems() )
    {
        item->displayName = "";
    }
}

void
Tree::removeItem( TreeItem* item )
{
    if ( item->getParent() )
    {
        item->getParent()->children.removeOne( item ); // remove item from parent
    }

    // remove item and all its children from treeItem and selection list
    QStack<TreeItem*> stack;
    stack.push( item );
    while ( !stack.isEmpty() )
    {
        TreeItem* item = stack.pop();

        selectionList.removeOne( item );
        treeItems.removeOne( item );

        foreach( TreeItem * child, item->getChildren() )
        {
            stack.push( child );
        }
    }
}

TreeItem*
Tree::getSelectionSummary() const
{
    return selectionSummary;
}

TreeItem*
Tree::getSelectionRootSummary() const
{
    return selectionRootSummary;
}

void
Tree::addItem( TreeItem* item, TreeItem* parent )
{
    assert( parent );
    parent->addChild( item );

    // add item and all its children to treeItem list and hash
    QStack<TreeItem*> stack;
    stack.push( item );
    while ( !stack.isEmpty() )
    {
        TreeItem* item = stack.pop();
        treeItemHash[ item->getCubeObject() ] = item;
        treeItems.append( item );
        foreach( TreeItem * child, item->getChildren() )
        {
            stack.push( child );
        }
    }
}

double
Tree::getMaxValue( const TreeItem* ) const
{
    return maxValue;
}

/* Returns the absolute or relative maximum of the tree.
 * For non-derived metrics, the inclusive top level items are the maximum values.
   For derived metrics, this is not the absolute maximum. The value of a child may be bigger than its parent.
 */
void
Tree::computeMaxValues()
{
    QList<TreeItem*> list;
    list.append( top );
    maxValue = top->getChildren().first()->totalValue;
    while ( !list.isEmpty() )
    {
        TreeItem* item = list.takeLast();
        foreach( TreeItem * child, item->getChildren() )
        {
            maxValue = std::max( maxValue, child->totalValue );
            if ( child->isExpanded() )
            {
                maxValue = std::max( maxValue, child->ownValue );
            }
            if ( child->isExpanded() )
            {
                list.append( child );
            }
        }
    }

    double roundThreshold = Globals::getRoundThreshold( FORMAT_TREES );
    if ( fabs( maxValue ) <= fabs( roundThreshold ) )
    {
        maxValue = 0.0;
    }
}

// this method computes the reference value (100%) for percentage-based value modi
//
double
Tree::computeReferenceValue( ValueModus valueModus )
{
    Tree* tree;
    if ( valueModus == METRICSELECTED_VALUES  ||
         valueModus == METRICROOT_VALUES ||
         valueModus == EXTERNAL_VALUES )
    {
        tree = getActiveTree( METRIC );
    }
    else if ( valueModus == CALLSELECTED_VALUES  ||
              valueModus == CALLROOT_VALUES )
    {
        tree = getActiveTree( CALL );
    }
    else // ( valueModus == SYSTEMSELECTED  || valueModus == SYSTEMROOT )
    {
        tree = getActiveTree( SYSTEM );
    }

    // since we divide the absolute value by the reference value,
    // the reference value 0.0 would result in the value "undefined"
    double referenceValue = 0.0;

    // get the proper value of the required selected, root, or external item
    if ( valueModus == METRICSELECTED_VALUES ||
         valueModus == CALLSELECTED_VALUES ||
         valueModus == SYSTEMSELECTED_VALUES )
    {
        referenceValue = tree->getSelectionSummary()->totalValue;
    }
    else if ( valueModus == METRICROOT_VALUES ||
              valueModus == CALLROOT_VALUES   ||
              valueModus == SYSTEMROOT_VALUES )
    {
        referenceValue = tree->getSelectionRootSummary()->totalValue;
    }
    else if ( valueModus == EXTERNAL_VALUES )
    {
        // get root item of the recently selected metric tree item as reference item for trees but metric tree
        TreeItem*     item       = tree->getLastSelection();
        cube::Metric* metric     = static_cast<cube::Metric* > ( item->getTopLevelItem()->getCubeObject() );
        QString       metricName = QString::fromStdString( metric->get_uniq_name() );
        MetricTree*   metricTree = static_cast<MetricTree*> ( tree );
        referenceValue = metricTree->getExternalReferenceValue( metricName );
    }

    // round values very close to 0.0 down to 0.0
    double roundThreshhold = Globals::getRoundThreshold( FORMAT_TREES );
    if ( referenceValue <= roundThreshhold && referenceValue >= -roundThreshhold )
    {
        referenceValue = 0.0;
    }
    return referenceValue;
}
// end of computeReferenceValues()

/**
 * @brief Tree::hideItem marks the item as hidden/unhidden and inserts it into hiddenItemMap
 */
void
Tree::setHidden( TreeItem* item, bool hidden )
{
    TreeItem* parent = item->getParent();
    if ( hidden != item->hidden ) // hidden status has changed -> invalidate parent item
    {
        parent->invalidate( false, true );
    }
    item->hidden = hidden;

    if ( hidden )
    {
        auto it = hiddenItemMap.find( parent );
        if ( it == hiddenItemMap.end() )
        {
            QList<TreeItem*> hiddenList;
            hiddenList.append( item );
            hiddenItemMap[ parent ] = hiddenList;
        }
        else
        {
            QList<TreeItem*>& hiddenList = it->second;
            hiddenList.append( item );
        }
    }
    else
    {
        hiddenItemMap[ parent ].removeOne( item );
    }
}

/**
 * marks all items with lower percentage than the threshold as hidden
 */
void
Tree::hideMinorValues( double threshold )
{
    hiddenItemMap.clear();
    QList<TreeItem*> parents; // parents with hidden children

    TreeItem* metric    = getActiveTree( METRIC )->getLastSelection();
    bool      isDerived = metric->isDerivedMetric();

    // hide elements with lower percentage that threshold
    QList<TreeItem*> items = top->getChildren();
    while ( !items.isEmpty() )
    {
        TreeItem* item = items.takeLast();
        if ( item->getCallType() == CallItemType::LoopItem )
        {
            continue; // hiding of loop items is not supported
        }

        if ( !isnan( item->getTotalValue() ) )
        {
            double maxValue = getMaxValue( item );
            double value    = item->getTotalValue();
            if ( isDerived && !isnan( item->getOwnValue() ) ) // derived metrics: total value may be lower than own value -> use maximum of both
            {
                value = std::max( item->getTotalValue(), item->getOwnValue() );
            }

            bool hidden = value * 100 / maxValue < threshold;
            item->setHidden( hidden );

            // handle parents of the examined item
            if ( item->getParent() != top )
            {
                if ( hidden )
                {
                    // parent of hidden item has to be invalidated to force recalculation with additional hidden child values
                    parents.append( item->getParent() );
                }
                else if ( isDerived ) // ensure that all parents of visible derived items remain visible
                {
                    TreeItem* parent = item->getParent();
                    while ( parent != top )
                    {
                        if ( parent->isHidden() )
                        {
                            parent->setHidden( false );
                            parents.removeOne( parent->getParent() );
                        }
                        parent = parent->getParent();
                    }
                }
            }
        }
        else // inclusive value is not yet available -> don't hide
        {
            item->setHidden( false );
        }

        // examine children of all expanded items
        if ( item->expanded )
        {
            // used expanded instead of isExpanded() because isExpanded() excludes hiddden items, but hidden derived items
            // may have large unhidden children
            if ( !item->isHidden() || isDerived ) // derived children may be larger than its hidden parents
            {
                items.append( item->getChildren() );
            }
        }
    }
    hidePresetItems();

    fixHiddenSelections();
}

/**
 * Used after hiding to ensure that only visible items remain selected. Deselects all hidden elements and select visible parents instead.
 *
 * Selected and deselected items are only marked, no signal should be emitted before filtering is done in TreeModelProxy.
 * This prevents a new filtering call to be started (because of the selection signal) before the current one has been finished.
 */
void
Tree::fixHiddenSelections()
{
    QList<TreeItem*> previousSelections = selectionList;
    for ( TreeItem* selection : previousSelections )
    {
        /* check, if selected item or one of its parents is hidden */
        TreeItem* hiddenItem = selection;
        while ( !hiddenItem->isHidden() && hiddenItem != top )
        {
            hiddenItem = hiddenItem->getParent();
        }

        if ( hiddenItem != top )
        {
            // unselect hidden item
            selection->setSelectionStatus( false ); // mark hidden item deselected but don't emit signal
            selectionList.removeOne( selection );

            TreeItem* visibleParent = hiddenItem->getParent();
            if ( visibleParent != top )                    // hidden item has visible parent -> select parent instead
            {
                visibleParent->setSelectionStatus( true ); // mark parent item selected but don't emit signal
                if ( !selectionList.contains( visibleParent ) )
                {
                    selectionList.append( visibleParent );
                }
            }
        }
    }

    /* all selected top level trees got hidden -> select any visible top level element instead */
    if ( selectionList.size() == 0 )
    {
        for ( TreeItem* item : top->getChildren() )
        {
            if ( !item->isHidden() )
            {
                item->setSelectionStatus( true );
                selectionList.append( item );
                break;
            }
        }
    }

    // all top elements are hidden -> unhide all and use previous selection
    if ( selectionList.size() == 0 )
    {
        TreeItem* item = top->getChildren().first();
        item->setHidden( false );
        for ( TreeItem* item : previousSelections )
        {
            item->setSelectionStatus( true );
            selectionList.append( item );
        }
    }
}

void
Tree::unhideItems()
{
    for ( TreeItem* item : qAsConst( treeItems ) )
    {
        item->hidden = false; // directly set to hidden to prevent invalidation of parent items of unchanged children
    }
    for ( const auto& it : hiddenItemMap )
    {
        TreeItem* parent = it.first;
        parent->invalidate(); // parent value is invalid because of removed hidden children
    }

    hiddenItemMap.clear();

    hidePresetItems();
}

void
Tree::hidePresetItems()
{
    if ( getType() == SYSTEM ) // initially hidden items should stay hidden
    {
        const QList<TreeItem*>& unvisited = getPredefinedHiddenItems();
        for ( TreeItem* item : unvisited )
        {
            setHidden( item, true );
        }
    }
}

bool
Tree::hasUserDefinedMinMaxValues() const
{
    return userDefinedMinMaxValues;
}

double
Tree::getUserDefinedMinValue() const
{
    return userMinValue;
}

double
Tree::getUserDefinedMaxValue() const
{
    return userMaxValue;
}

void
Tree::unsetUserDefinedMinMaxValues()
{
    userDefinedMinMaxValues = false;
}

void
Tree::setUserDefinedMinMaxValues( double minValue, double maxValue )
{
    this->userDefinedMinMaxValues = true;
    this->userMinValue            = minValue;
    this->userMaxValue            = maxValue;
}

void
Tree::setFilter( FilterType filter, double threshold )
{
    activeFilter    = filter;
    filterThreshold = threshold;
    if ( filter == FILTER_NONE )
    {
        unhideItems();
    }
    else if ( threshold >= 0 )
    {
        filterThreshold = threshold;
        hideMinorValues( threshold );
    }
}

int
Tree::getDepth() const
{
    return maxDepth;
}

QList<TreeItem*>
Tree::getHiddenChildren( TreeItem* parent ) const
{
    if ( activeFilter == FILTER_NONE )
    {
        return QList<TreeItem*>();
    }
    auto it = hiddenItemMap.find( parent );
    if ( it == hiddenItemMap.end() )
    {
        return QList<TreeItem*>();
    }
    else
    {
        return it->second;
    }
}

QList<TreeItem*>
Tree::getItemsToCalculate()
{
    QList<TreeItem*> list = top->getChildren();
    QList<TreeItem*> itemsToCalculate;
    while ( !list.isEmpty() )
    {
        TreeItem* item = list.takeFirst();
        if ( item->isExpanded() )
        {
            foreach( TreeItem * child, item->getChildren() )
            list.append( child );
        }
        if ( item->isCalculationEnabled() && !item->isCalculated() )
        {
            itemsToCalculate.append( item );
        }
    }

    // qDebug() << "Tree::getItemsToCalculate() " << this->getLabel() << itemsToCalculate.size();
    return itemsToCalculate;
}

// static method for use in QtConcurrent
void
Tree::calculate( Task* data )
{
    data->calculate();
    delete data;
}

/**
 * @brief Tree::getStatisticValues returns the required statistical values for the value widgets.
 */
TreeStatistics*
Tree::getStatistics() const
{
    return statistics;
}

void
Tree::setFuture( Future* future )
{
    this->future = future;
}


#include "Globals.h"
#include "TabManager.h"
/** used to calculate the data of root_cnode in a thread */
void
TreeTask::calculate()
{
    Tree*            tree = static_cast<Tree*> ( item->getTree() );
    cube::CubeProxy* cube = tree->getCube();


    // temporary hack to support artificial kernel nodes, which only exist in GUI -> cubeProxy change is planned
    if ( cnode_selection.empty() )
    {
        TreeItem* callItem = Globals::getTabManager()->getActiveTree( CALL )->getSelectionList().first();
        if ( callItem->getCallType() == CallItemType::ArtificialKernelRoot )
        {
            item->invalidate();
            item->setCalculationStatus( TreeItem::CalculationStatus::CALCULATED );
            item->singleValue = true;
            return;
        }
    }

    Value* result = cube->calculateValue( metric_selection,
                                          cnode_selection,
                                          sysres_selection );

    if ( expanded ) // set exclusive/inclusive value depending of the state of the item when the task was created
    {
        item->setExclusiveValue( result );
    }
    else
    {
        item->setInclusiveValue( result );
    }
}
