/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2021                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"

#include "MetricTreeView.h"
#include "CallTree.h"
#include "Globals.h"
#include "TabManager.h"
using namespace cubegui;

MetricTreeView::MetricTreeView( TreeModelInterface* model,
                                const QString&      tabLabel ) : TreeView( model, tabLabel )
{
    setIcon( QIcon( ":images/tree-icon.png" ) );
    this->setUniformRowHeights( false ); // items of metric tree may have different size, if special value view is used
}

/**
 * @brief MetricTreeView::selectionIsValid checks if newItem can be added to the current selection
 * @param oldItems previously selected tree items
 * @param newItem item that should become selected
 * @return true, if the item can be added to the current selection
 */
bool
MetricTreeView::selectionIsValid( QList<TreeItem*>& oldItems, TreeItem* newItem )
{
    if ( oldItems.size() > 0 )
    {
        if ( newItem->isDerivedMetric() || oldItems.at( 0 )->isDerivedMetric() )
        {
            QString message( tr( "Multiple selection of derived metrics is not possible." ) );
            Globals::setStatusMessage( message, Error );
            return false;
        }
    }

    bool    sameUnit = true;
    QString first_uom, second_uom;
    for ( int i = 0; i < oldItems.size() && sameUnit; i++ )
    {
        const TreeItem* oldItem = oldItems.at( i )->getTopLevelItem();
        first_uom  = QString::fromStdString( ( ( cube::Metric* )( oldItem->getCubeObject() ) )->get_uom() );
        second_uom = QString::fromStdString( ( ( cube::Metric* )( newItem->getCubeObject() ) )->get_uom() );
        if ( first_uom != second_uom )
        {
            sameUnit = false;
            break;
        }
    }
    if ( !sameUnit )
    {
        QString message( tr( "Multiple metric selection is possible only if the unit of measurement is compatible. " ) );
        message += QString( tr( "In this case \"%1\" is incompatible with \"%2\"" ) ).arg( first_uom, second_uom );
        Globals::setStatusMessage( message, Error );
        return false;
    }

    bool sameRoot = true;
    for ( int i = 0; i < oldItems.size() && sameRoot; i++ )
    {
        const TreeItem* first  = oldItems.at( i )->getTopLevelItem();
        const TreeItem* second = newItem->getTopLevelItem();
        if ( first != second )
        {
            sameRoot = false;
            break;
        }
    }

    if ( !sameRoot )
    {
        Globals::setStatusMessage( tr( "Be careful. Metrics with different roots might be incompatible for operation \"+\"." ), Warning );
    }

    return true;
}

void
MetricTreeView::selectionChanged( const QItemSelection& selected,
                                  const QItemSelection& deselected )
{
    Globals::getTabManager()->updateValueViews(); // selection of new metric may change the icon size of the tree items in all trees
    TreeView::selectionChanged( selected, deselected );
}

/**
 * creates context menu items for call trees
 */
void
MetricTreeView::fillContextMenu()
{
    addExpertContextMenuOptions();

    contextMenu->addAction( contextMenuHash.at( TreeItemInfo ) );
    contextMenu->addAction( contextMenuHash.at( Documentation ) );

    contextMenu->addSeparator();
    contextMenu->addAction( contextMenuHash.at( ExpandMenu ) );

    contextMenu->addAction( contextMenuHash.at( FindItems ) );
    contextMenu->addAction( contextMenuHash.at( ClearFound ) );
    contextMenu->addAction( contextMenuHash.at( SortingMenu ) );
    contextMenu->addSeparator();
    contextMenu->addAction( contextMenuHash.at( CopyClipboard ) );
    contextMenu->addSeparator();
}

QString
MetricTreeView::getContextDescription( TreeItem* item ) const
{
    cube::Metric* _met = static_cast<cube::Metric* >( item->getCubeObject() );

    // store the description text in the string "descr";
    // we get the description from the cube object "cubeObject" of the clicked item,
    // where we have to distinguish on its type
    std::string title;

    std::string uniq_name;
    std::string disp_name;
    std::string dtype;
    std::string uom;
    std::string val;
    std::string url;
    std::string kind;
    std::string descr;
    std::string cubepl_expression;
    std::string cubepl_init_expression;
    std::string cubepl_plus_expression;
    std::string cubepl_minus_expression;
    std::string cubepl_aggr_expression;
    title = tr( "No information" ).toUtf8().data();

    uniq_name               = _met->get_uniq_name();
    disp_name               = _met->get_disp_name();
    dtype                   = _met->get_dtype();
    uom                     = _met->get_uom();
    val                     = _met->get_val();
    url                     = _met->get_url();
    descr                   = _met->get_descr();
    kind                    = _met->get_metric_kind();
    cubepl_expression       = _met->get_expression();
    cubepl_init_expression  = _met->get_init_expression();
    cubepl_plus_expression  = _met->get_aggr_plus_expression();
    cubepl_minus_expression = _met->get_aggr_minus_expression();
    cubepl_aggr_expression  = _met->get_aggr_aggr_expression();


    QList<QPair<QString, QString> > list; // QPair<description, value>
    list.append( QPair<QString, QString>( tr( "Metric" ), _met->get_descr().c_str() ) );
    list.append( QPair<QString, QString>( tr( "Display name" ), disp_name.c_str() ) );
    list.append( QPair<QString, QString>( tr( "Unique name" ), uniq_name.c_str() ) );
    list.append( QPair<QString, QString>( tr( "Data type" ), dtype.c_str() ) );
    list.append( QPair<QString, QString>( tr( "Unit" ), uom.c_str() ) );
    list.append( QPair<QString, QString>( tr( "Value" ), val.c_str() ) );
    list.append( QPair<QString, QString>( tr( "URL" ), url.c_str() ) );
    list.append( QPair<QString, QString>( tr( "Kind of values" ), kind.c_str() ) );

    if ( !( cubepl_expression.compare( "" ) == 0 ) )
    {
        list.append( QPair<QString, QString>( tr( "CubePL expression" ), cubepl_expression.c_str() ) );
    }
    if ( !( cubepl_init_expression.compare( "" ) == 0 ) )
    {
        list.append( QPair<QString, QString>( tr( "CubePL Init expression" ), cubepl_init_expression.c_str() ) );
    }
    if ( !( cubepl_plus_expression.compare( "" ) == 0 ) )
    {
        list.append( QPair<QString, QString>( tr( "CubePL Plus expression" ), cubepl_plus_expression.c_str() ) );
    }
    if ( !( cubepl_minus_expression.compare( "" ) == 0 ) )
    {
        list.append( QPair<QString, QString>( tr( "CubePL Minus expression" ), cubepl_minus_expression.c_str() ) );
    }
    if ( !( cubepl_aggr_expression.compare( "" ) == 0 ) )
    {
        list.append( QPair<QString, QString>( tr( "CubePL Aggr expression" ), cubepl_aggr_expression.c_str() ) );
    }

    QString label = tr( "Properties" );
    list.append( QPair<QString, QString>( label, ( _met->isConvertible() ) ? tr( "Convertible to data" ) : tr( "Non convertible to data" ) ) );
    list.append( QPair<QString, QString>( "", ( _met->isCacheable() ) ? tr( "Cacheable" ) : tr( "Non cacheable" ) ) );
    list.append( QPair<QString, QString>( "", ( _met->isGhost() ) ? tr( "Ghost metric" ) : tr( "Normal metric" ) ) );
    list.append( QPair<QString, QString>( "", ( _met->isRowWise() ) ? tr( "Rowwise metric" ) : tr( "Elementswise metric" ) ) );

    QString info = InfoWidget::formatAsTable( list );
    info += formatAttributes( _met->get_attrs() );

    return info;
}
