/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBEGUI_GLOBALS_H
#define CUBEGUI_GLOBALS_H

#include <unordered_map>
#include <QColor>
#include <QMainWindow>
#include <QTextStream>
#include <QVariant>
#include "CubeTypes.h"
#include "CubeValues.h"
#include "Constants.h"

namespace cubepluginapi
{
class ValueView;
}

namespace cubegui
{
enum MessageType { AppendInformation, Verbose, DelayedInformation, Information, Warning, Error, Critical };
enum UserOption { DisablePlugins, ExpertMode, VerboseMode, ExperimentalMode, Single, ManualCalculation, DocPath, WebEngine };

class ColorMap;
class PrecisionWidget;
class MainWidget;
class TabManager;
class SettingsHandler;
class InternalSettingsHandler;
class ValueViewConfig;
class TreeItem;
class Settings;

class Globals
{
    friend class MainWidget;
    friend class PluginManager;
    friend class ValueViewConfig;
public:
    static Globals*
    getInstance();

    /** returns the corresponding color of the currently selected colormap */
    static QColor
    getColor( double value,
              double minValue,
              double maxValue,
              bool   whiteForZero = true );

    static QPair<QString, QString>
    formatNumberAndUnit( double                   value,
                         const QString&           unit,
                         bool                     integerType = false,
                         PrecisionFormat          format = FORMAT_TREES,
                         const cubegui::TreeItem* item = 0 );

    static QString
    formatNumber( double                   value,
                  bool                     integerType = false,
                  PrecisionFormat          format = FORMAT_TREES,
                  const cubegui::TreeItem* item = 0 );

    static QString
    formatNumber( double          value,
                  double          referenceValue,
                  bool            scientificSmall = false,
                  bool            integerType = false,
                  PrecisionFormat format = FORMAT_TREES );

    static double
    getPrecision( PrecisionFormat format );

    static double
    getRoundNumber( PrecisionFormat format );

    static double
    getRoundThreshold( PrecisionFormat format );

    /** prints the given text into the status line below the main widget */
    static void
    setStatusMessage( const QString& msg,
                      MessageType    type = Information,
                      bool           isLogged = true );

    static QMainWindow*
    getMainWindow();

    static TabManager*
    getTabManager();

    static void
    setVerbose( bool value );

    static QTextStream&
    debug( const QString& sender );

    static cubepluginapi::ValueView*
    getValueView( cube::DataType type );

    static Settings*
    getSettings();

    static QString
    getOption( UserOption option );

    static bool
    optionIsSet( UserOption option );
    static void
    setOption( UserOption option,
               QString    value = "enabled" );

    /** returns version number of the cube library
     *  This is CUBELIB_VERSION_NUMBER for local cube but may differ if a cube server is used.
     */
    static int
    cubelibVersion();

private:
    static Globals*                  globals;
    static ColorMap*                 defaultColorMap;
    static cubepluginapi::ValueView* defaultValueView;

    PrecisionWidget* precisionWidget;
    ColorMap*        colorMap;
    MainWidget*      mainWidget;
    TabManager*      tabManager;
    QTextStream*     outStream;
    Settings*        settings;

    std::unordered_map<UserOption, QString, std::hash<int> >                       userOptions;
    std::unordered_map<cube::DataType, cubepluginapi::ValueView*, std::hash<int> > valueViews;

    Globals();
    void
    setSettings( Settings* newSettings );
    void
    setTabManager( TabManager* t );
    void
    setMainWidget( MainWidget* m );
    static PrecisionWidget*
    getPrecisionWidget();
    static void
    setColorMap( ColorMap* map = 0 );
    static ColorMap*
    getColorMap();
    static void
    setValueView( cube::DataType            type,
                  cubepluginapi::ValueView* view = 0 );
    static void
    removeValueView( cubepluginapi::ValueView* view );
};
}

// Qt::endl doesn't exist in Qt < 5.14.0
#if QT_VERSION < QT_VERSION_CHECK( 5, 14, 0 )
namespace Qt
{
using TextStreamFunction = QTextStream & ( * )( QTextStream & );
constexpr TextStreamFunction endl = ::endl;
}
#endif

#endif // GLOBALS_H
