/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2024-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/*-------------------------------------------------------------------------*/
/**
 *  @file
 *  @ingroup CUBE_gui.network
 *  @brief   Definition of the class cube::CubeQtWebSocket.
 **/
/*-------------------------------------------------------------------------*/


#include <config.h>

#include "CubeQtWebSocket.h"

#include <QApplication>
#include <QDateTime>
#include <QEventLoop>
#include <QMetaType>
#include <QObject>
#include <QWebSocketServer>
#include <QWebSocket>
#include <QThread>
#include <QTimer>
#include <QtGlobal>
#include <QDebug>
#include <unistd.h>
#include <assert.h>

#include "CubeError.h"
#include "CubeNetworkProxy.h"

#ifdef __EMSCRIPTEN__
#include <emscripten/val.h>
#endif

using namespace std;
using namespace cube;


#define SETUP_EVENT_LOOP_FOR( object, signal )                    \
    QTimer timer;                                                 \
    timer.setSingleShot( true );                                  \
    QEventLoop loop;                                              \
    loop.connect( &timer, SIGNAL( timeout() ), SLOT( quit() ) );  \
    loop.connect( object, SIGNAL( signal ), SLOT( quit() ) );


#define SETUP_CLIENT_EVENT_LOOP_FOR( object, signal )         \
    SETUP_EVENT_LOOP_FOR( object, signal )                    \
    QObject::connect( object, &QWebSocket::errorOccurred, &loop, &QEventLoop::quit )

#define PROCESS_EVENT_LOOP_WITH_TIMEOUT( timeout ) \
    timer.start( timeout );                        \
    loop.exec();

namespace
{
// default socket timeout of 1 second
int DEFAULT_TIMEOUT = 1000;
}


SocketPtr
CubeQtWebSocket::create()
{
    qDebug() << "emscripten built of CubeQtWebSocket on: " << __DATE__ << __TIME__;

    qRegisterMetaType< QAbstractSocket::SocketError >(
        "QAbstractSocket::SocketError" );
    qRegisterMetaType< QAbstractSocket::SocketState >(
        "QAbstractSocket::SocketState" );

    CubeNetworkProxy::exceptionPtr = nullptr;
    /*
     * A QWebSocket cannot be accessed from different threads. When CubeQtWebSocket is created, a new thread (socketIOThread) is created
     * and it's thread affinity moved to the new thread. All read/write operations will be processed in socketIOThread.
     * If methods of CubeProxy are called from other threads, signals are used to replace direct calls.
     */
    CubeQtWebSocket* qss = new CubeQtWebSocket();
    qss->socketIOThread = new QThread();
    QObject::connect( qss->socketIOThread, SIGNAL( finished() ), qss->socketIOThread, SLOT( deleteLater() ) );
    qss->moveToThread( qss->socketIOThread );
    qss->socketIOThread->start();
    // blocking io operations
    QObject::connect( qss, &CubeQtWebSocket::connectSignal, qss, &CubeQtWebSocket::socketConnect, Qt::BlockingQueuedConnection );
    QObject::connect( qss, &CubeQtWebSocket::receiveSignal, qss, &CubeQtWebSocket::socketReceive, Qt::BlockingQueuedConnection );
    QObject::connect( qss, &CubeQtWebSocket::disconnectSignal, qss, &CubeQtWebSocket::disconnectSlot, Qt::BlockingQueuedConnection );
    QObject::connect( qss, &CubeQtWebSocket::sendSignal, qss, &CubeQtWebSocket::socketSend, Qt::BlockingQueuedConnection );

    return SocketPtr( qss  );
}

void
CubeQtWebSocket::connect( const std::string& address,
                          int                port )
{
    // calls socketConnect() in main thread, waits for slot to be finished (Qt::BlockingQueuedConnection)
    emit( connectSignal( address.c_str(), port ) );
}


/**
   returns websocket path to jupyter cube extention
 */
QUrl
CubeQtWebSocket::getWebSocketUrl()
{
#ifdef __EMSCRIPTEN__
    // get location of jupyter-cube host
    emscripten::val location = emscripten::val::undefined();
    emscripten::val self     = emscripten::val::global( "self" );
    if ( !self.isUndefined() && !self[ "location" ].isUndefined() )
    {
        location = self[ "location" ];
    }
    else
    {
        qWarning() << "Cannot determine WebSocket URL, location is undefined!";
        return QUrl(); // error
    }
    QString protocol = location[ "protocol" ].as<std::string>().c_str();
    QString host     = location[ "host" ].as<std::string>().c_str();
    QString pathname = location[ "pathname" ].as<std::string>().c_str();
    // TODO: is there a better way to get the path to the cube websocket service?
    int idx = pathname.indexOf( "/static" );  // path to e.g. /cube/static/cube.js
    if ( idx != -1 )
    {
        pathname = pathname.left( idx );
    }
    QString wsProtocol = protocol == "https:" ? "wss://" : "ws://";
    QString wsPath     = "/ws"; // relative path to cube web sockets
    return QUrl( wsProtocol + host + pathname + wsPath );
#else
    QUrl url( "ws://localhost:8008/ws" ); // websocket test config
    return url;
#endif
}

void
CubeQtWebSocket::socketConnect( const QString& address,
                                int            port )
{
    if ( mSocket )
    {
        if ( mSocket->isValid() )
        {
            disconnect();
        }
    }
    else
    {
        mSocket = new QWebSocket();
    }

    // wait for connection to be etablished
    SETUP_CLIENT_EVENT_LOOP_FOR( mSocket,  connected() );

    QUrl url = getWebSocketUrl();
    qInfo() << "WebSocket connection to" << url;
    mSocket->open( url );
    // timeout for webassembly cannot be too short
    PROCESS_EVENT_LOOP_WITH_TIMEOUT( 2000 );

    if ( mSocket->state() == QAbstractSocket::ConnectedState )
    {
        QObject::connect( mSocket, &QWebSocket::binaryMessageReceived, this, &CubeQtWebSocket::receiveBytes );
    }
    else
    {
        CubeNetworkProxy::exceptionPtr = std::make_exception_ptr( UnrecoverableNetworkError(
                                                                      QObject::tr( "Connection timed out" ).toUtf8().data() ) );
    }
}

void
CubeQtWebSocket::disconnect()
{
    emit disconnectSignal();
}

void
CubeQtWebSocket::disconnectSlot()
{
    if ( !mSocket || !mSocket->isValid() )
    {
        return;
    }

    if ( mSocket && mSocket->isValid() )
    {
        SETUP_CLIENT_EVENT_LOOP_FOR( mSocket, disconnected() );
        mSocket->close();
        PROCESS_EVENT_LOOP_WITH_TIMEOUT( 100 );
    }
}

void
CubeQtWebSocket::listen( int port ) // server side (unused)
{
}


void
CubeQtWebSocket::accept() // server side (unused)
{
}


void
CubeQtWebSocket::shutdown() // server side (unused)
{
}


/**
   CubeQtWebSocket::send() is non-blocking but the actual send operation has to be processed in the thread socketIOThread.
 */
size_t
CubeQtWebSocket::send( const void* buffer,
                       size_t      num_bytes )
{
    // required for wasm: cannot block GUI thread
    assert( QThread::currentThread() != QCoreApplication::instance()->thread() );
    emit sendSignal( buffer, num_bytes );
    return num_bytes;
}

/**
   CubeQtWebSocket::receive() blocks calling thread and processes socketReceive() in the thread socketIOThread.
   This blocking method mustn't be called from the main thread to avoid deadlock.
 */
size_t
CubeQtWebSocket::receive( void*  buffer,
                          size_t num_bytes )
{
    assert( QThread::currentThread() != QCoreApplication::instance()->thread() );
    emit receiveSignal( buffer, num_bytes ); // blocks caller till slot socketReceive() has been finished
    return num_bytes;
}


void
CubeQtWebSocket::socketSend( const void* buffer,
                             size_t      num_bytes )
{
    try
    {
        if ( !mSocket || !mSocket->isValid() )
        {
            if ( CubeNetworkProxy::exceptionPtr )
            {
                return;  // only handle first error
            }
            throw UnrecoverableNetworkError( QObject::tr( "Unable to write to invalid socket." ).toUtf8().data() );
        }

        QByteArray bytes( ( const char* )buffer, num_bytes );
        qint64     ret = mSocket->sendBinaryMessage( bytes );
        if ( ret != num_bytes )
        {
            throw UnrecoverableNetworkError( QObject::tr( "Error sending data." ).toUtf8().data() );
        }
        mSocket->flush();
    }
    catch ( const UnrecoverableNetworkError& e )
    {
        CubeNetworkProxy::exceptionPtr = std::current_exception();
    }
}

void
CubeQtWebSocket::receiveBytes( const QByteArray& bytes )
{
    QMutexLocker locker( &mutex );
    receiveBuffer += bytes;
}

void
CubeQtWebSocket::socketReceive( void*  buffer,
                                size_t num_bytes )
{
    const int timeout = 50; // check every timeout ms if data is available

    //qInfo() << "\nsocketReceive for byte: " << num_bytes;
    // wait for available data, which is read in a different thread ->receiveBytes
    try
    {
        while ( num_bytes > receiveBuffer.size() )
        {
            QEventLoop loop;
            QTimer::singleShot( timeout, &loop, &QEventLoop::quit );
            loop.exec();

            // ensure to exit the function if the socket gets invalid
            if ( !( mSocket && mSocket->isValid() ) )
            {
                QDateTime current = QDateTime::currentDateTime();
                QString   time    = current.toString( "dd.MM hh:mm:ss" );

                qDebug() << time << " socket has been closed. buffer:" << receiveBuffer.size() << " bytes to read " << num_bytes;
                break;  // server has closed connection but buffer should contain data from the exit request
            }
            /*
               if ( receiveBuffer.size() > 0 )
               {
                qDebug() << "CubeQtWebSocket::receive " << num_bytes << receiveBuffer.size() << mSocket->state();
               }
             */
        }
    }
    catch ( const std::exception& e ) // handle exception in different thread
    {
        qDebug() << "CubeQtWebSocket::socketRecv() error: " << e.what();
        CubeNetworkProxy::exceptionPtr = std::current_exception();
    }

    if ( receiveBuffer.size() >= num_bytes ) // copy the data which has been read in a different thread in receiveBytes
    {
        QMutexLocker locker( &mutex );
        memcpy( buffer, receiveBuffer.constData(), num_bytes );
        receiveBuffer.remove( 0, num_bytes );
    }
    else if ( !mSocket || !mSocket->isValid() )   // socket has been closed
    {
        memset( buffer, 0, num_bytes );
    }
}


bool
CubeQtWebSocket::isConnected()
{
    if ( !mSocket )
    {
        CubeNetworkProxy::exceptionPtr = std::make_exception_ptr( UnrecoverableNetworkError(
                                                                      QObject::tr( "CubeQtWebSocket::Invalid socket." ).toUtf8().data() ) );
    }

    return mSocket->state() == QAbstractSocket::ConnectedState;
}


bool
CubeQtWebSocket::isListening()
{
    return true;
}


std::string
CubeQtWebSocket::getHostname()
{
    return QObject::tr( "Unknown host" ).toUtf8().data();
}


std::string
CubeQtWebSocket::getInfoString()
{
    return "CubeQtWebSocket";
}

std::string
CubeQtWebSocket::getPeerInfo()
{
    return "";
}


CubeQtWebSocket::CubeQtWebSocket()
    : mSocket( NULL )
{
}


CubeQtWebSocket::~CubeQtWebSocket()
{
    QObject::disconnect( this, &CubeQtWebSocket::connectSignal, this, &CubeQtWebSocket::socketConnect );
    QObject::disconnect( this, &CubeQtWebSocket::receiveSignal, this, &CubeQtWebSocket::socketReceive );
    QObject::disconnect( this, &CubeQtWebSocket::disconnectSignal, this, &CubeQtWebSocket::disconnectSlot );
    QObject::disconnect( this, &CubeQtWebSocket::sendSignal, this, &CubeQtWebSocket::socketSend );

    socketIOThread->quit();
    socketIOThread->wait();

    try
    {
        if ( mSocket && mSocket->isValid() )
        {
            if ( mSocket->isValid() )
            {
                emit disconnectSignal();
            }
            delete mSocket;
        }
    }
    catch ( UnrecoverableNetworkError& e )
    {
        qDebug() << "CubeQtWebSocket::~CubeQtWebSocket:: " << e.get_msg();
    }
}
