/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023                                                     **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <QVBoxLayout>
#include <QPushButton>
#include <QtPlugin>
#include <string>
#include <iostream>
#include "cubelib-version.h"
#include "ClientPlugin.h"
#include "Globals.h"
#include "PluginServices.h"

/**
 * If the plugin doesn't load, start cube with -verbose to get detailed information.
 * If the error message "Plugin verification data mismatch" is printed, check if the
 * same compiler and Qt version have been used.
 */

using namespace cubepluginapi;
using namespace clientserver;

ClientPlugin::ClientPlugin()
{
    // The constructor should be empty, use cubeOpened to initialize. If Qt widgets or
    // signals/slots are used in constructor, they have to be deleted in destructor,
    // otherwise cube may crash if the plugin is unloaded.
}

bool
ClientPlugin::cubeOpened( PluginServices* service )
{
    this->service = service;

    widget_ = new QWidget();
    qlabel_ = new QLabel( "" );
    QVBoxLayout* layout = new QVBoxLayout();
    widget_->setLayout( layout );
    layout->addWidget( qlabel_ );

    QPushButton* button = new QPushButton( "Send text to server" );
    layout->addWidget( button );
    connect( button, &QPushButton::clicked, this, &ClientPlugin::communicateWithServerPlugin );

    service->addTab( SYSTEM, this, cubegui::OTHER_PLUGIN_TAB );

    return true; // initialisation is ok => plugin should be shown
}

void
ClientPlugin::cubeClosed()
{
    delete widget_;
}

/** set a version number, the plugin with the highest version number will be loaded */
void
ClientPlugin::version( int& major, int& minor, int& bugfix ) const
{
    major  = 1;
    minor  = 0;
    bugfix = 0;
}

/** unique plugin name */
QString
ClientPlugin::name() const
{
    return "ClientServerDemoPlugin";
}

QString
ClientPlugin::getHelpText() const
{
    return "Demo to show communication between client and server plugins";
}

/** widget that will be placed into the tab */
QWidget*
ClientPlugin::widget()
{
    return widget_;
}

/** tab label */
QString
ClientPlugin::label() const
{
    return "client-server";
}

/** slot, which is called if a tree item is selected */
void
ClientPlugin::communicateWithServerPlugin()
{
    std::string                fn( "message from client plugin" );
    std::vector<unsigned char> messageToServer( fn.begin(), fn.end() );

    if (  cubegui::Globals::cubelibVersion() < CUBELIB_VERSION_CHECK( 4, 8, 0 ) ) // todo adjust version
    {
        qlabel_->setText( "remote cubelib version doesn't support plugins" );
        return; // service->getCube()->sendToPlugin would return empty answer
    }
    std::vector<unsigned char> answer = service->getCube()->sendToPlugin( name().toStdString(), messageToServer );

    std::string sourcef { answer.begin(), answer.end() };

    QString message = sourcef.size() > 0 ? sourcef.c_str() : "ServerPlugin doesn't answer";
    service->setMessage( message );
    qlabel_->setText( message );
}
