##*************************************************************************##
##  CUBE        http://www.scalasca.org/                                   ##
##*************************************************************************##
##  Copyright (c) 2024                                                     ##
##  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          ##
##                                                                         ##
##  This software may be modified and distributed under the terms of       ##
##  a BSD-style license.  See the COPYING file in the package base         ##
##  directory for details.                                                 ##
##*************************************************************************##


# usage: pdflatex( <TeX-file.tex> DEPENDS target1 target2...)
# used pdflatex to create the file "TeX-file.pdf"

function(pdflatex)
  # --- handle argument list
  set(TEX_FILE ${ARGV0})      # tex file ist 1st argument
  set(multiValueArgs DEPENDS) # list of dependencies preceeded by DEPENDS
  cmake_parse_arguments(PDFLATEX "" ${options}" ${oneValueArgs}" "${multiValueArgs}" ${ARGN})

  # --- end argument list
  get_filename_component(TARGET_DIR  ${TEX_FILE} DIRECTORY) # directory of the tex file
  get_filename_component(TARGET_NAME ${TEX_FILE} NAME_WE)   # target name without .tex
  get_filename_component(SUBDIR_NAME ${CMAKE_CURRENT_BINARY_DIR} NAME)
  set( PDF_FILE ${TARGET_DIR}/${TARGET_NAME}.pdf )
  list(APPEND PDFLATEX_DEPENDS ${TARGET_DIR}/${TARGET_NAME}.tex) # add tex file to list of dependencies

  # pdflatex command with arguments
  set (BUILD_PDFLATEX_DRAFT ${PDFLATEX_COMPILER} -draftmode -interaction=nonstopmode ${TARGET_NAME}.tex )
  string( REGEX REPLACE "-draftmode" "" BUILD_PDFLATEX "${BUILD_PDFLATEX_DRAFT}" )

  # generate pdf files
  add_custom_command( OUTPUT ${PDF_FILE}
                  COMMENT "Processing ${LATEX_DIR}/${TARGET_NAME}.tex"
                  # 1st run: Don't show LaTeX output, but in case of errors, print the location of the log file.
                  COMMAND ${BUILD_PDFLATEX_DRAFT}  > /dev/null
                  || (echo "ERROR generating ${TARGET_NAME}, see ${LATEX_DIR}/${TARGET_NAME}.log for details"
                  && grep -i error ${TARGET_NAME}.log && exit 111 )
                  # maximum 4 further runs and makeindex call
                  COMMAND bash -c  "[[ -e ${TARGET_NAME}.idx ]]" && makeindex ${TARGET_NAME}.idx > makeindex.log 2>&1 || true
                  COMMAND grep -q "to get cross-references right" ${TARGET_NAME}.log && ${BUILD_PDFLATEX_DRAFT} > /dev/null || true
                  COMMAND grep -q "to get cross-references right" ${TARGET_NAME}.log && ${BUILD_PDFLATEX_DRAFT} > /dev/null || true
                  COMMAND grep -q "to get cross-references right" ${TARGET_NAME}.log && ${BUILD_PDFLATEX_DRAFT} > /dev/null || true
                  COMMAND bash -c  "[[ -e ${TARGET_NAME}.idx ]]" && makeindex ${TARGET_NAME}.idx > /dev/null 2>&1 || true
                  COMMAND ${BUILD_PDFLATEX} > /dev/null # final run which generates pdf

                  WORKING_DIRECTORY ${LATEX_DIR}
                  DEPENDS ${PDFLATEX_DEPENDS}
  )

endfunction()
